﻿using UnityEditor;
using UnityEngine;
using System.Collections.Generic;
using System.Linq;
using PK;

namespace PK
{
    public class CreatureBalanceWindow : EditorWindow
    {
        // Existing fields
        private List<HexCreature> creatures = new List<HexCreature>();
        private List<HexCreature> filteredCreatures = new List<HexCreature>();
        private Vector2 scrollPosition;

        // Add filter enum and state
        private enum CreatureFilter { All, Base, Ranged, Cavalry }
        private CreatureFilter currentFilter = CreatureFilter.All;

        // The shared scriptable object properties with labels.
        private readonly string[] propertyNames = new string[]
        {
        "_health",
        "_damage",
        "_minSoldiersPerHex",
        "_maxSoldiersPerHex",
        "_soldiersPerUnit",
        "_stepsPerTurn"
        };

        private readonly string[] propertyLabels = new string[]
        {
        "Health",
        "Damage",
        "MinSoldiers",
        "MaxSoldiers",
        "Soldiers/Unit",
        "Steps/Turn"
        };

        [MenuItem("PK/Complex/Creature Balance tool")]
        public static void ShowWindow()
        {
            GetWindow<CreatureBalanceWindow>("Creature Editor");
        }

        private void OnEnable()
        {
            RefreshCreatureList();
        }
        private void RefreshCreatureList()
        {
            creatures.Clear();
            string[] guids = AssetDatabase.FindAssets("t:HexCreature", new[] { "Assets/Resources/Creatures" });
            foreach (string guid in guids)
            {
                string path = AssetDatabase.GUIDToAssetPath(guid);
                HexCreature creature = AssetDatabase.LoadAssetAtPath<HexCreature>(path);
                if (creature != null) creatures.Add(creature);
            }
            RefreshFilteredList(); // Add this to keep filter in sync
        }
        private readonly (string label, string name)[] rangedProperties =
        {
        ("Range", "_range"),
        ("Max Range", "_maxRange"),
        ("Ranged Damage", "_rangedDamage"),
        ("Projectile", "_projectile")
    };

        private readonly (string label, string name)[] cavalryProperties =
        {
        ("Min Charge", "_minChargeSteps"),
        ("Charge Dmg", "_chargeDamage")
    };


        private void RefreshFilteredList()
        {
            filteredCreatures.Clear();

            foreach (var creature in creatures)
            {
                if (currentFilter == CreatureFilter.All)
                {
                    filteredCreatures.Add(creature);
                    continue;
                }

                bool isRanged = creature is HexRangedCreature;
                bool isCavalry = creature is HexCavalryCreature;

                switch (currentFilter)
                {
                    case CreatureFilter.Base:
                        if (!isRanged && !isCavalry)
                            filteredCreatures.Add(creature);
                        break;
                    case CreatureFilter.Ranged:
                        if (isRanged)
                            filteredCreatures.Add(creature);
                        break;
                    case CreatureFilter.Cavalry:
                        if (isCavalry)
                            filteredCreatures.Add(creature);
                        break;
                }
            }
        }

        private void OnGUI()
        {
            GUILayout.Label("Creature Properties Table", EditorStyles.boldLabel);

            // Filter toolbar
            EditorGUILayout.BeginHorizontal();
            if (GUILayout.Button("Refresh List")) RefreshCreatureList();

            var newFilter = (CreatureFilter)GUILayout.Toolbar(
                (int)currentFilter,
                new[] { "All", "Base", "Ranged", "Cavalry" }
            );

            if (newFilter != currentFilter)
            {
                currentFilter = newFilter;
                RefreshFilteredList();
            }
            EditorGUILayout.EndHorizontal();

            scrollPosition = EditorGUILayout.BeginScrollView(scrollPosition);

            // Table header using FILTERED creatures
            EditorGUILayout.BeginHorizontal();
            EditorGUILayout.LabelField("Property", GUILayout.Width(120));
            foreach (var creature in filteredCreatures)
                EditorGUILayout.LabelField(creature.name, GUILayout.Width(120));
            EditorGUILayout.EndHorizontal();
            EditorGUILayout.BeginHorizontal();
            EditorGUILayout.LabelField("Property", GUILayout.Width(120));
            foreach (var creature in filteredCreatures)
            {
                if (GUILayout.Button("Open SO", GUILayout.Width(120)))
                {
                    AssetDatabase.OpenAsset(creature);
                }
            }
            EditorGUILayout.EndHorizontal();

            // Draw properties using FILTERED list
            DrawPropertySection("Core Stats", propertyLabels, propertyNames);
            DrawDerivedSection("Ranged Properties", rangedProperties, c => c as HexRangedCreature);
            DrawDerivedSection("Cavalry Properties", cavalryProperties, c => c as HexCavalryCreature);

            EditorGUILayout.EndScrollView();
        }

        private void DrawPropertySection(string sectionName, string[] labels, string[] properties)
        {
            for (int i = 0; i < properties.Length; i++)
            {
                EditorGUILayout.BeginHorizontal();
                EditorGUILayout.LabelField(labels[i], GUILayout.Width(120));

                foreach (var creature in filteredCreatures) // Changed to filtered
                {
                            var so = new SerializedObject(creature);
                            var prop = so.FindProperty(properties[i]);

                            EditorGUI.BeginChangeCheck();
                            EditorGUILayout.PropertyField(prop, GUIContent.none, true, GUILayout.Width(120));
                            if (EditorGUI.EndChangeCheck()) so.ApplyModifiedProperties();
                }
                EditorGUILayout.EndHorizontal();
            }
        }
        private void DrawDerivedSection<T>(string title,
                                 (string label, string name)[] properties,
                                 System.Func<HexCreature, T> castFunc) where T : HexCreature
        {
            // Only check FILTERED creatures for type presence
            bool hasAny = filteredCreatures.Exists(c => castFunc(c) != null);
            if (!hasAny) return;

            // Section header
            EditorGUILayout.BeginHorizontal();
            EditorGUILayout.LabelField(title, EditorStyles.boldLabel, GUILayout.Width(120));
            foreach (var _ in filteredCreatures) EditorGUILayout.LabelField("", GUILayout.Width(120));
            EditorGUILayout.EndHorizontal();

            // Property rows
            foreach (var prop in properties)
            {
                EditorGUILayout.BeginHorizontal();
                EditorGUILayout.LabelField(prop.label, GUILayout.Width(120));

                foreach (var creature in filteredCreatures)
                {
                    var derived = castFunc(creature);
                    if (derived == null)
                    {
                        EditorGUILayout.LabelField("-", GUILayout.Width(120));
                        continue;
                    }

                    var so = new SerializedObject(derived);
                    var sp = so.FindProperty(prop.name);

                    EditorGUI.BeginChangeCheck();
                    EditorGUILayout.PropertyField(sp, GUIContent.none, true, GUILayout.Width(120));
                    if (EditorGUI.EndChangeCheck()) so.ApplyModifiedProperties();
                }
                EditorGUILayout.EndHorizontal();
            }
        }

    }
}