using UnityEngine;
using UnityEditor;
using UnityEditor.Animations;
using System.Collections.Generic;
using System.IO;
using System.Text.RegularExpressions;
using System.Linq;
using PK;
using System;

public class CreatureGeneratorWindow : EditorWindow
{
    // --- Constants ---
    private const string DEFAULT_BASE_CONTROLLER_PATH = "Assets/Art/Creatures/BaseAnimationController/_BaseCreatureUnitAnimator.controller";
    // Define paths for the BASE prefabs we want to create variants FROM
    private const string BASE_STRATEGIC_PREFAB_PATH = "Assets/Resources/Creatures/DefaultCreatureStrategicMapVariant.prefab";
    private const string BASE_UNIT_TACTICAL_PREFAB_PATH = "Assets/Resources/Creatures/DefaultCreatureUnitTacticalMapVariant.prefab";

    private const string ART_CREATURES_ROOT = "Assets/Art/Creatures";
    private const string RESOURCES_CREATURES_ROOT = "Assets/Resources/Creatures";

    // --- Menu Item ---
    [MenuItem("PK/Complex/Generate Creature from Sprites")]
    public static void ShowWindow() => GetWindow<CreatureGeneratorWindow>("Generate Creature from Sprites");

    // --- Serialized Fields (Window State) ---
    [SerializeField] private float frameRate = 60f;
    [SerializeField] private string spritesDirectory = "Assets/Art/Creatures/NewCreature"; // Default to a specific creature
    [SerializeField] private string baseControllerPath = DEFAULT_BASE_CONTROLLER_PATH;
    [SerializeField] private int _selectedUnitIndex = 0;
    [SerializeField] private int _selectedCreatureTypeIndex = 0;

    // Base creature stats
    [SerializeField] private float health = 10.0f;
    [SerializeField] private float damage = 1.0f;
    [SerializeField] private int minSoldiersPerHex = 5;
    [SerializeField] private int maxSoldiersPerHex = 5;
    [SerializeField] private int soldiersPerUnit = 15;
    [SerializeField] private int stepsPerTurn = 5;

    // Animation State Config
    [System.Serializable]
    public struct AnimationState
    {
        public string name;
        public bool required;
        public bool looped;
        public string pattern; // Regex pattern to match sprite names
    }

    [SerializeField]
    private List<AnimationState> animationStates = new List<AnimationState>
    {
        new AnimationState { name = "Walk", required = true, looped = true, pattern = @"_walk_|_walk$|_run_|_run$" },
        new AnimationState { name = "Idle", required = true, looped = true, pattern = @"_idle_|_idle$|_stay_|_stay$" },
        new AnimationState { name = "Attack", required = false, looped = false, pattern = @"_attack_|_attack$" },
        new AnimationState { name = "Hit", required = false, looped = false, pattern = @"_hit_|_hit$|_suffer_|_suffer$" },
        new AnimationState { name = "Dead", required = false, looped = false, pattern = @"_dead_|_dead$" }
    };

    // Tactical Map Prefab Choices
    private readonly string[] _unitPrefabPaths = new string[]
    {
        "Assets/Resources/Creatures/1UnitCreatureTacticalMap.prefab",
        "Assets/Resources/Creatures/4UnitCreatureTacticalMap.prefab",
        "Assets/Resources/Creatures/6UnitCreatureTacticalMap.prefab",
        "Assets/Resources/Creatures/7UnitCreatureTacticalMap.prefab"
    };

    // Creature Type Configs (including specific fields)
    [System.Serializable]
    public abstract class CreatureTypeConfig
    {
        public string typeName;
        public Type classType; // The ScriptableObject type (e.g., typeof(HexCreature))

        public abstract void DrawTypeSpecificFields(CreatureGeneratorWindow window);
        public abstract void SetTypeSpecificProperties(SerializedObject so, CreatureGeneratorWindow window);
    }

    [System.Serializable]
    public class HexCreatureConfig : CreatureTypeConfig
    {
        public override void DrawTypeSpecificFields(CreatureGeneratorWindow window) { /* No specific fields */ }
        public override void SetTypeSpecificProperties(SerializedObject so, CreatureGeneratorWindow window) { /* No specific properties */ }
    }

    [System.Serializable]
    public class HexRangedCreatureConfig : CreatureTypeConfig
    {
        [Header("Ranged creature config")]
        public int range = 1;
        public int maxRange = 2;
        public float rangedDamage = 1.0f;
        public Sprite projectile;

        public override void DrawTypeSpecificFields(CreatureGeneratorWindow window)
        {
            EditorGUILayout.LabelField("Ranged Creature Config", EditorStyles.boldLabel);
            range = EditorGUILayout.IntSlider("Attack Range", range, 1, 25);
            maxRange = EditorGUILayout.IntSlider("Max Range", maxRange, range, 25);
            rangedDamage = EditorGUILayout.FloatField("Ranged Damage", rangedDamage);
            projectile = EditorGUILayout.ObjectField("Projectile Sprite", projectile, typeof(Sprite), false) as Sprite;
        }

        public override void SetTypeSpecificProperties(SerializedObject so, CreatureGeneratorWindow window)
        {
            SafeSetIntProperty(so, "_range", range);
            SafeSetIntProperty(so, "_maxRange", maxRange);
            SafeSetFloatProperty(so, "_rangedDamage", rangedDamage);
            SafeSetObjectProperty(so, "_projectile", projectile);
        }
    }

    [System.Serializable]
    public class HexCavalryCreatureConfig : CreatureTypeConfig
    {
        [Header("Cavalry creature config")]
        public int minChargeSteps = 3;
        public float chargeDamage = 2.0f;

        public override void DrawTypeSpecificFields(CreatureGeneratorWindow window)
        {
            EditorGUILayout.LabelField("Cavalry Creature Config", EditorStyles.boldLabel);
            minChargeSteps = EditorGUILayout.IntSlider("Minimum Charge Steps", minChargeSteps, 1, 25);
            chargeDamage = EditorGUILayout.FloatField("Charge Damage Multiplier", chargeDamage);
        }

        public override void SetTypeSpecificProperties(SerializedObject so, CreatureGeneratorWindow window)
        {
            SafeSetIntProperty(so, "_minChargeSteps", minChargeSteps);
            SafeSetFloatProperty(so, "_chargeDamage", chargeDamage);
        }
    }


    [SerializeField]
    private List<CreatureTypeConfig> creatureTypes = new List<CreatureTypeConfig>
    {
        new HexCreatureConfig { typeName = "Melee", classType = typeof(HexCreature)},
        new HexRangedCreatureConfig { typeName = "Ranged", classType = typeof(HexRangedCreature)},
        new HexCavalryCreatureConfig { typeName = "Cavalry", classType = typeof(HexCavalryCreature)}
    };

    // --- Non-Serialized Fields (Runtime State) ---
    private SerializedObject serializedObject;
    private SerializedProperty animationStatesProperty;
    private SerializedProperty creatureTypesProperty; // Needed if list itself is modified
    private string currentCharacterName = ""; // Store derived character name

    #region Unity Lifecycle & GUI

    private void OnEnable()
    {
        serializedObject = new SerializedObject(this);
        animationStatesProperty = serializedObject.FindProperty("animationStates");
        // Retrieve the serialized property for creatureTypes to allow editing type-specific fields
        creatureTypesProperty = serializedObject.FindProperty("creatureTypes");
        EnsureBaseControllerExists(); // Check/create base controller on enable
        DeriveCharacterName(); // Calculate name on enable
    }

    private void OnGUI()
    {
        serializedObject.Update();
        EditorGUI.BeginChangeCheck(); // Detect changes to update character name

        EditorGUILayout.LabelField("Core Settings", EditorStyles.boldLabel);
        EditorGUILayout.Space();

        // Sprites Directory - Crucial for deriving names and paths
        spritesDirectory = EditorGUILayout.TextField("Creature Sprites Directory", spritesDirectory);
        if (GUILayout.Button("Select Sprites Directory", GUILayout.Height(25)))
        {
            string selectedPath = EditorUtility.OpenFolderPanel("Select Sprites Directory", spritesDirectory, "");
            if (!string.IsNullOrEmpty(selectedPath))
            {
                spritesDirectory = GetValidAssetPath(selectedPath);
                GUI.changed = true; // Mark change to trigger name update
            }
        }
        EditorGUILayout.HelpBox($"Derived Creature Name: {currentCharacterName}", MessageType.None);
        EditorGUILayout.Space();

        // Other Core Settings
        baseControllerPath = EditorGUILayout.TextField("Base Animator Controller", baseControllerPath);
        frameRate = EditorGUILayout.FloatField("Animation Frame Rate", frameRate);

        EditorGUILayout.Space();
        EditorGUILayout.PropertyField(animationStatesProperty, true);
        EditorGUILayout.Space();

        // --- Creature Type Selection ---
        EditorGUILayout.LabelField("Creature Type", EditorStyles.boldLabel);
        _selectedCreatureTypeIndex = EditorGUILayout.Popup("Select Type", _selectedCreatureTypeIndex, creatureTypes.Select(t => t.typeName).ToArray());
        EditorGUILayout.Space();

        // --- Tactical Map Unit Selection ---
        EditorGUILayout.LabelField("Units Per Hex (Tactical Map Base)", EditorStyles.boldLabel);
        EditorGUILayout.BeginHorizontal();
        for (int i = 0; i < _unitPrefabPaths.Length; i++)
        {
            if (UnitButton(_unitPrefabPaths[i], i == _selectedUnitIndex)) { _selectedUnitIndex = i; }
        }
        EditorGUILayout.EndHorizontal();
        if (_selectedUnitIndex >= 0 && _selectedUnitIndex < _unitPrefabPaths.Length)
            EditorGUILayout.HelpBox($"Selected Base: {Path.GetFileName(_unitPrefabPaths[_selectedUnitIndex])}", MessageType.Info);
        else
            EditorGUILayout.HelpBox("Invalid tactical base selection.", MessageType.Warning);
        EditorGUILayout.Space();

        // --- Creature Properties ---
        EditorGUILayout.LabelField("Creature Stats / SO Properties", EditorStyles.boldLabel);
        health = EditorGUILayout.FloatField("Health", health);
        damage = EditorGUILayout.FloatField("Damage", damage);
        minSoldiersPerHex = EditorGUILayout.IntField("Min Soldiers Per Hex", minSoldiersPerHex);
        maxSoldiersPerHex = EditorGUILayout.IntField("Max Soldiers Per Hex", maxSoldiersPerHex);
        soldiersPerUnit = EditorGUILayout.IntField("Soldiers Per Unit", soldiersPerUnit);
        stepsPerTurn = EditorGUILayout.IntField("Steps Per Turn", stepsPerTurn);
        EditorGUILayout.Space();

        // --- Type Specific Properties ---
        // Use the SerializedProperty for the list element to ensure changes are saved
        if (_selectedCreatureTypeIndex >= 0 && _selectedCreatureTypeIndex < creatureTypes.Count && creatureTypesProperty != null && creatureTypesProperty.isArray)
        {
            // Get the SerializedProperty for the specific CreatureTypeConfig element
            SerializedProperty selectedTypeProp = creatureTypesProperty.GetArrayElementAtIndex(_selectedCreatureTypeIndex);
            // Draw the fields using the instance method, but changes are managed by SerializedObject/Property
            creatureTypes[_selectedCreatureTypeIndex].DrawTypeSpecificFields(this);
            // Note: SetTypeSpecificProperties is called during SO generation/update, not here in OnGUI.
        }


        EditorGUILayout.Space(20);
        EditorGUILayout.LabelField("Generation Steps", EditorStyles.boldLabel);

        // --- Generation Buttons ---
        using (new EditorGUI.DisabledScope(string.IsNullOrEmpty(currentCharacterName) || currentCharacterName == "InvalidOrMissingDirectory"))
        {
            if (GUILayout.Button("1. Generate Animations & Controller", GUILayout.Height(30)))
            {
                if (ConfirmGeneration("Generate Animations & Controller", "This will overwrite animations and the override controller."))
                {
                    GenerateAnimationsAndController();
                }
            }
            if (GUILayout.Button("2. Generate Prefab Variants", GUILayout.Height(30)))
            {
                if (ConfirmGeneration("Generate Prefab Variants", "Requires animations/controller to exist. This will overwrite strategic/unit prefab variants."))
                {
                    GeneratePrefabVariants();
                }
            }
            if (GUILayout.Button("3. Generate/Update Scriptable Object", GUILayout.Height(30)))
            {
                if (ConfirmGeneration("Generate/Update Scriptable Object", "Requires prefab variants to exist. This will create or update the SO, applying all current stat/type settings."))
                {
                    GenerateOrUpdateScriptableObject();
                }
            }

            EditorGUILayout.Space(10);
            EditorGUILayout.LabelField("Selective Updates", EditorStyles.boldLabel);

            if (GUILayout.Button("Update Prefabs in Existing SO", GUILayout.Height(30)))
            {
                if (ConfirmGeneration("Update Prefabs in SO", "Requires SO and prefab variants to exist. This will ONLY update the prefab references and icon in the SO, leaving other stats untouched."))
                {
                    UpdateScriptableObjectPrefabsOnly();
                }
            }
        } // End DisabledScope


        if (EditorGUI.EndChangeCheck()) // Update character name if relevant fields changed
        {
            DeriveCharacterName();
        }

        serializedObject.ApplyModifiedProperties(); // Apply changes made in the GUI
    }

    #endregion

    #region Path and Name Helpers

    private void DeriveCharacterName()
    {
        if (!string.IsNullOrEmpty(spritesDirectory) && AssetDatabase.IsValidFolder(spritesDirectory))
        {
            currentCharacterName = Path.GetFileName(spritesDirectory);
        }
        else
        {
            currentCharacterName = "InvalidOrMissingDirectory";
        }
        // No Repaint here, rely on EditorGUI.EndChangeCheck or regular updates
    }

    private string GetCharacterName()
    {
        // Ensure it's derived if called before OnGUI updates it
        if (string.IsNullOrEmpty(currentCharacterName) || currentCharacterName == "InvalidOrMissingDirectory")
        {
            DeriveCharacterName();
        }
        if (currentCharacterName == "InvalidOrMissingDirectory")
        {
            EditorUtility.DisplayDialog("Error", "Cannot determine character name. Please select a valid Sprites Directory.", "OK");
            Debug.LogError("Cannot determine character name. Please select a valid Sprites Directory.");
            return null;
        }
        return currentCharacterName;
    }

    private string GetArtDirectoryPath(string characterName) => Path.Combine(ART_CREATURES_ROOT, characterName);
    private string GetResourcesDirectoryPath(string characterName) => Path.Combine(RESOURCES_CREATURES_ROOT, characterName);
    private string GetScriptableObjectPath(string characterName) => Path.Combine(GetResourcesDirectoryPath(characterName), $"{characterName}.asset");
    private string GetOverrideControllerPath(string characterName) => Path.Combine(GetArtDirectoryPath(characterName), $"{characterName}CreatureUnitAnimator.overrideController");
    private string GetStrategicVariantPath(string characterName) => Path.Combine(GetResourcesDirectoryPath(characterName), $"{characterName}CreatureStrategicMap.prefab");
    private string GetUnitVariantPath(string characterName) => Path.Combine(GetResourcesDirectoryPath(characterName), $"{characterName}CreatureUnitTacticalMap.prefab");
    private string GetAnimationPath(string characterName, string stateName) => Path.Combine(GetArtDirectoryPath(characterName), $"{characterName}_{stateName}.anim");

    private string GetIdleSpritePathFromAnimation(string characterName)
    {
        string idleAnimPath = GetAnimationPath(characterName, "Idle"); // Assumes "Idle" is the state name
        AnimationClip idleClip = AssetDatabase.LoadAssetAtPath<AnimationClip>(idleAnimPath);
        if (idleClip != null)
        {
            foreach (var binding in AnimationUtility.GetObjectReferenceCurveBindings(idleClip))
            {
                if (binding.type == typeof(SpriteRenderer) && binding.propertyName == "m_Sprite")
                {
                    ObjectReferenceKeyframe[] keyframes = AnimationUtility.GetObjectReferenceCurve(idleClip, binding);
                    if (keyframes.Length > 0 && keyframes[0].value is Sprite idleSprite)
                    {
                        return AssetDatabase.GetAssetPath(idleSprite);
                    }
                }
            }
            Debug.LogWarning($"Could not find Sprite keyframes in Idle animation clip: {idleAnimPath}");
        }
        else
        {
            Debug.LogWarning($"Idle animation clip not found: {idleAnimPath}");
        }
        // Fallback: Could try finding *any* sprite in the folder if needed, but returning null is safer.
        Debug.LogWarning($"Could not determine Idle Sprite path from animation clip for {characterName}.");
        return null;
    }


    private string GetRelativePath(string absolutePath)
    {
        if (absolutePath.StartsWith(Application.dataPath))
        {
            return "Assets" + absolutePath.Substring(Application.dataPath.Length);
        }
        return absolutePath; // Assume already relative or outside project
    }

    private string GetValidAssetPath(string path) => path.StartsWith(Application.dataPath) ? GetRelativePath(path) : path;

    #endregion

    #region Directory and Asset Validation/Ensurance

    private void EnsureDirectoryExists(string relativePath)
    {
        if (string.IsNullOrEmpty(relativePath) || relativePath == "Assets" || AssetDatabase.IsValidFolder(relativePath)) return;

        string parent = Path.GetDirectoryName(relativePath);
        string folder = Path.GetFileName(relativePath);

        // Ensure parent exists first, recursively
        EnsureDirectoryExists(parent);

        // Create the folder if parent is valid *within the AssetDatabase*
        if (AssetDatabase.IsValidFolder(parent))
        {
            string guid = AssetDatabase.CreateFolder(parent, folder);
            if (!string.IsNullOrEmpty(guid))
            {
                Debug.Log($"Created directory: {relativePath}");
                // Refresh might not be needed immediately after CreateFolder, but can ensure visibility
                // AssetDatabase.Refresh(ImportAssetOptions.ForceSynchronousImport);
            }
            else
            {
                Debug.LogError($"Failed to create directory '{relativePath}' using AssetDatabase.CreateFolder.");
            }
        }
        else
        {
            Debug.LogError($"Could not create directory '{relativePath}' because parent '{parent}' is not a valid AssetDatabase folder.");
        }
    }

    private void EnsureBaseControllerExists()
    {
        string dir = Path.GetDirectoryName(DEFAULT_BASE_CONTROLLER_PATH);
        EnsureDirectoryExists(dir);

        if (AssetDatabase.LoadAssetAtPath<AnimatorController>(DEFAULT_BASE_CONTROLLER_PATH) == null)
        {
            Debug.Log($"Base controller not found at {DEFAULT_BASE_CONTROLLER_PATH}. Creating a new one.");
            var controller = AnimatorController.CreateAnimatorControllerAtPath(DEFAULT_BASE_CONTROLLER_PATH);
            foreach (var state in animationStates) // Use the configured states
            {
                var clip = new AnimationClip { name = state.name }; // Use state name for base clip
                AssetDatabase.AddObjectToAsset(clip, controller);
                var animatorState = controller.AddMotion(clip);
                // Optionally set default transitions here if needed for the base controller
            }
            AssetDatabase.SaveAssets();
            AssetDatabase.Refresh();
        }
    }

    private bool ValidateBaseAssets() // Validates assets needed for *any* generation step
    {
        string characterName = GetCharacterName();
        if (characterName == null) return false; // Error already logged by GetCharacterName

        if (!Directory.Exists(spritesDirectory))
        { // Check filesystem existence first
            EditorUtility.DisplayDialog("Error", $"Sprites directory not found or invalid: {spritesDirectory}", "OK");
            Debug.LogError($"Invalid sprites directory: {spritesDirectory}");
            return false;
        }
        // Check if it's a valid folder known by the AssetDatabase
        if (!AssetDatabase.IsValidFolder(spritesDirectory))
        {
            EditorUtility.DisplayDialog("Error", $"Sprites directory '{spritesDirectory}' is not recognized by the Asset Database. Please ensure it's within your project's Assets folder.", "OK");
            Debug.LogError($"Sprites directory not recognized by Asset Database: {spritesDirectory}");
            return false;
        }

        if (!File.Exists(baseControllerPath))
        {
            EditorUtility.DisplayDialog("Error", $"Base Animator Controller not found: {baseControllerPath}", "OK");
            Debug.LogError($"Base controller not found: {baseControllerPath}");
            return false;
        }
        // Prefab bases needed for variant creation or SO assignment
        if (!File.Exists(BASE_STRATEGIC_PREFAB_PATH))
        {
            EditorUtility.DisplayDialog("Error", $"Base strategic prefab not found: {BASE_STRATEGIC_PREFAB_PATH}", "OK");
            Debug.LogError($"Base strategic prefab not found: {BASE_STRATEGIC_PREFAB_PATH}");
            return false;
        }
        if (!File.Exists(BASE_UNIT_TACTICAL_PREFAB_PATH))
        {
            EditorUtility.DisplayDialog("Error", $"Base unit prefab not found: {BASE_UNIT_TACTICAL_PREFAB_PATH}", "OK");
            Debug.LogError($"Base unit prefab not found: {BASE_UNIT_TACTICAL_PREFAB_PATH}");
            return false;
        }

        // Validate selected tactical base prefab path
        if (_selectedUnitIndex < 0 || _selectedUnitIndex >= _unitPrefabPaths.Length)
        {
            EditorUtility.DisplayDialog("Error", "Invalid tactical base prefab selection index.", "OK");
            Debug.LogError("Invalid tactical base selection index.");
            return false;
        }
        string tacticalPrefabPath = _unitPrefabPaths[_selectedUnitIndex];
        if (!File.Exists(tacticalPrefabPath))
        {
            EditorUtility.DisplayDialog("Error", $"Selected tactical base prefab not found: {tacticalPrefabPath}", "OK");
            Debug.LogError($"Selected tactical base prefab not found: {tacticalPrefabPath}");
            return false;
        }

        return true;
    }

    // Checks if assets expected to be generated by previous steps exist
    private bool ValidateGeneratedAssetsForStep(params string[] assetPaths)
    {
        foreach (var path in assetPaths)
        {
            if (string.IsNullOrEmpty(path))
            {
                Debug.LogWarning($"Validation skipped for an empty asset path."); // e.g., if icon path is null
                continue;
            }
            // Use AssetDatabase to check existence, more reliable than File.Exists for project assets
            if (AssetDatabase.LoadAssetAtPath<UnityEngine.Object>(path) == null)
            {
                EditorUtility.DisplayDialog("Error", $"Required generated asset not found for this step: {path}\nPlease generate previous steps first or check paths.", "OK");
                Debug.LogError($"Required generated asset not found: {path}");
                return false;
            }
        }
        return true;
    }

    #endregion

    #region GUI Helpers
    private bool UnitButton(string path, bool isSelected)
    {
        var match = Regex.Match(path, @"(\d+)Unit");
        int unitCount = match.Success ? int.Parse(match.Groups[1].Value) : 1;
        string label = $"{unitCount} Units";

        Color originalColor = GUI.backgroundColor;
        GUI.backgroundColor = isSelected ? Color.green : originalColor;
        bool clicked = GUILayout.Button(label, GUILayout.Width(100));
        GUI.backgroundColor = originalColor;
        return clicked;
    }

    private bool ConfirmGeneration(string title, string message)
    {
        string characterName = GetCharacterName();
        if (characterName == null) return false; // Error already logged

        return EditorUtility.DisplayDialog(title,
              $"{message}\n\nCreature: '{characterName}'\nDirectory: '{spritesDirectory}'\n\nProceed?",
              "Yes", "Cancel");
    }

    #endregion

    #region Core Generation Logic

    // --- STEP 1: Animations and Controller ---
    public void GenerateAnimationsAndController()
    {
        if (!ValidateBaseAssets()) return;
        string characterName = GetCharacterName();
        if (characterName == null) return;

        string artDirectory = GetArtDirectoryPath(characterName);
        EnsureDirectoryExists(artDirectory);

        Debug.Log($"--- Step 1: Generating Animations & Controller for: {characterName} ---");

        // 1a. Load Sprites
        var groupedSprites = LoadAndGroupSprites(characterName, spritesDirectory);
        if (groupedSprites == null)
        {
            Debug.LogError($"Sprite loading/grouping failed for {characterName}. Aborting Step 1.");
            EditorUtility.DisplayDialog("Error", $"Sprite loading or validation failed for '{characterName}'. Check console.", "OK");
            return;
        }

        // 1b. Create Animation Clips
        CreateAnimationAssets(characterName, artDirectory, groupedSprites);

        // 1c. Create Override Controller
        CreateOrUpdateOverrideController(characterName, artDirectory, groupedSprites);

        // 1d. Save & Refresh
        AssetDatabase.SaveAssets();
        AssetDatabase.Refresh();
        Debug.Log($"--- Step 1 Complete for: {characterName} ---");
        EditorUtility.DisplayDialog("Step 1 Complete", $"Animations and override controller generated/updated for '{characterName}'.", "OK");
    }

    // --- STEP 2: Prefab Variants ---
    public void GeneratePrefabVariants()
    {
        if (!ValidateBaseAssets()) return;
        string characterName = GetCharacterName();
        if (characterName == null) return;

        string resourcesDirectory = GetResourcesDirectoryPath(characterName);
        string overrideControllerPath = GetOverrideControllerPath(characterName);
        string idleSpritePath = GetIdleSpritePathFromAnimation(characterName); // May return null
        string strategicVariantPath = GetStrategicVariantPath(characterName);
        string unitVariantPath = GetUnitVariantPath(characterName);
        string scriptableObjectPath = GetScriptableObjectPath(characterName); // Path to check for existing SO

        // Validate required generated assets from Step 1
        if (!ValidateGeneratedAssetsForStep(overrideControllerPath)) return;
        // Idle sprite is needed, but path might be null if lookup failed. Helper handles null.

        EnsureDirectoryExists(resourcesDirectory); // Ensure target dir exists

        Debug.Log($"--- Step 2: Generating Prefab Variants for: {characterName} ---");

        // 2a. Load required assets (Controller, Sprite)
        var overrideController = AssetDatabase.LoadAssetAtPath<AnimatorOverrideController>(overrideControllerPath);
        Sprite idleSprite = null;
        if (!string.IsNullOrEmpty(idleSpritePath))
        {
            idleSprite = AssetDatabase.LoadAssetAtPath<Sprite>(idleSpritePath);
        }
        // Override controller is essential, validated above by ValidateGeneratedAssetsForStep
        if (idleSprite == null)
        {
            Debug.LogWarning($"Idle sprite not found or path could not be determined (checked: {idleSpritePath ?? "null"}). Prefab variants will not have icon sprite set, and existing SO icon won't be updated.");
        }


        // 2b. Create Strategic Variant
        // The CreatePrefabVariant method now returns the root component of the saved asset
        var strategicComponent = CreatePrefabVariant(
            BASE_STRATEGIC_PREFAB_PATH,
            strategicVariantPath,
            typeof(HexCreatureView), // Expected root component
            idleSprite,
            overrideController
        ) as HexCreatureView; // Cast the result

        // 2c. Create Unit Variant
        var unitComponent = CreatePrefabVariant(
            BASE_UNIT_TACTICAL_PREFAB_PATH,
            unitVariantPath,
            typeof(HexCreatureSquadUnitView), // Expected root component
            idleSprite,
            overrideController
        ) as HexCreatureSquadUnitView; // Cast the result

        // Check if variant creation succeeded before trying to update SO
        if (strategicComponent == null || unitComponent == null)
        {
            Debug.LogError("One or both prefab variants failed to generate. Scriptable Object prefab references will not be updated.");
            // Don't proceed to SO update
        }
        else
        {
            // --- NEW: Update existing Scriptable Object's prefabs ---
            ScriptableObject existingCreatureSO = AssetDatabase.LoadAssetAtPath<ScriptableObject>(scriptableObjectPath);
            if (existingCreatureSO != null)
            {
                Debug.Log($"Existing ScriptableObject found at '{scriptableObjectPath}'. Updating its prefab references and icon.");
                SerializedObject so = new SerializedObject(existingCreatureSO);
                so.Update();

                // Assign the newly created/updated prefab variants (using the returned components)
                // Also re-assign the selected tactical base prefab and the determined icon sprite
                string tacticalBasePrefabPath = _unitPrefabPaths[_selectedUnitIndex];
                var tacticalBasePrefab = AssetDatabase.LoadAssetAtPath<HexCreatureSquadView>(tacticalBasePrefabPath);

                SafeSetObjectProperty(so, "_icon", idleSprite); // Update icon using the loaded sprite
                SafeSetObjectProperty(so, "_strategicMapPrefab", strategicComponent);
                SafeSetObjectProperty(so, "_tacticalMapUnitPrefab", unitComponent);

                if (so.ApplyModifiedProperties()) // Check if anything actually changed
                {
                    EditorUtility.SetDirty(existingCreatureSO);
                    Debug.Log("ScriptableObject prefab/icon references updated.");
                }
                else
                {
                    Debug.Log("ScriptableObject prefab/icon references were already up-to-date.");
                }
            }
            else
            {
                Debug.Log($"No existing ScriptableObject found at '{scriptableObjectPath}'. Prefab references will be set when Step 3 is run.");
            }
            // --- End NEW ---
        }

        // 2d. Save & Refresh (saves variants and potentially updated SO)
        AssetDatabase.SaveAssets();
        AssetDatabase.Refresh();
        Debug.Log($"--- Step 2 Complete for: {characterName} ---");
        EditorUtility.DisplayDialog("Step 2 Complete", $"Prefab variants generated/updated for '{characterName}'. Existing SO prefabs/icon updated if found.", "OK");
    }


    // --- STEP 3: Scriptable Object ---
    public void GenerateOrUpdateScriptableObject()
    {
        if (!ValidateBaseAssets()) return;
        string characterName = GetCharacterName();
        if (characterName == null) return;

        string strategicPrefabPath = GetStrategicVariantPath(characterName);
        string unitPrefabPath = GetUnitVariantPath(characterName);
        string tacticalBasePrefabPath = _unitPrefabPaths[_selectedUnitIndex]; // Path from selection
        string scriptableObjectPath = GetScriptableObjectPath(characterName);
        string idleSpritePath = GetIdleSpritePathFromAnimation(characterName); // For icon

        // Validate required generated assets from Step 2 (variants) and selected base tactical
        if (!ValidateGeneratedAssetsForStep(strategicPrefabPath, unitPrefabPath, tacticalBasePrefabPath)) return;

        string resourcesDirectory = GetResourcesDirectoryPath(characterName);
        EnsureDirectoryExists(resourcesDirectory);

        Debug.Log($"--- Step 3: Generating/Updating Scriptable Object for: {characterName} ---");

        // 3a. Load dependencies
        var strategicPrefab = AssetDatabase.LoadAssetAtPath<HexCreatureView>(strategicPrefabPath);
        var unitPrefab = AssetDatabase.LoadAssetAtPath<HexCreatureSquadUnitView>(unitPrefabPath);
        var tacticalBasePrefab = AssetDatabase.LoadAssetAtPath<HexCreatureSquadView>(tacticalBasePrefabPath);
        Sprite iconSprite = null;
        if (!string.IsNullOrEmpty(idleSpritePath))
        {
            iconSprite = AssetDatabase.LoadAssetAtPath<Sprite>(idleSpritePath);
        }

        // Assets existence checked by ValidateGeneratedAssetsForStep, check for null just in case loading failed weirdly
        if (strategicPrefab == null || unitPrefab == null || tacticalBasePrefab == null)
        {
            Debug.LogError($"Failed to load one or more required prefabs even after validation passed. Aborting Step 3."); return;
        }
        if (iconSprite == null)
        {
            Debug.LogWarning($"Icon sprite not found or path could not be determined (checked: {idleSpritePath ?? "null"}). SO icon will not be set.");
        }


        // 3b. Load or Create Scriptable Object
        var creatureTypeConfig = creatureTypes[_selectedCreatureTypeIndex];
        Type targetType = creatureTypeConfig.classType;
        ScriptableObject creature = AssetDatabase.LoadAssetAtPath<ScriptableObject>(scriptableObjectPath);
        bool createdNew = false;

        if (creature != null && creature.GetType() != targetType)
        {
            Debug.LogWarning($"Existing asset at '{scriptableObjectPath}' is type '{creature.GetType().Name}', expected '{targetType.Name}'. Deleting old.");
            AssetDatabase.DeleteAsset(scriptableObjectPath);
            creature = null;
        }

        if (creature == null)
        {
            creature = ScriptableObject.CreateInstance(targetType);
            AssetDatabase.CreateAsset(creature, scriptableObjectPath);
            createdNew = true;
            Debug.Log($"Created new ScriptableObject: {scriptableObjectPath} of type {targetType.Name}");
        }
        else
        {
            Debug.Log($"Updating existing ScriptableObject: {scriptableObjectPath}");
        }

        // 3c. Apply ALL properties
        var so = new SerializedObject(creature);
        so.Update();

        // UID
        SerializedProperty uidProp = so.FindProperty("_uid");
        if (uidProp != null) { HexUniqueScriptableObjectHelper.CheckUid(uidProp); }
        else { Debug.LogError($"SO type '{targetType.Name}' missing '_uid' field."); }

        // Assign Prefabs & Icon
        SafeSetObjectProperty(so, "_icon", iconSprite);
        SafeSetObjectProperty(so, "_strategicMapPrefab", strategicPrefab);
        SafeSetObjectProperty(so, "_tacticalMapUnitPrefab", unitPrefab);
        SafeSetObjectProperty(so, "_tacticalMapPrefab", tacticalBasePrefab);


        // Assign Base Stats from Window
        SafeSetFloatProperty(so, "_health", health);
        SafeSetFloatProperty(so, "_damage", damage);
        SafeSetIntProperty(so, "_minSoldiersPerHex", minSoldiersPerHex);
        SafeSetIntProperty(so, "_maxSoldiersPerHex", maxSoldiersPerHex);
        SafeSetIntProperty(so, "_soldiersPerUnit", soldiersPerUnit);
        SafeSetIntProperty(so, "_stepsPerTurn", stepsPerTurn);

        // Assign Type-Specific Properties
        creatureTypeConfig.SetTypeSpecificProperties(so, this);

        so.ApplyModifiedProperties();
        EditorUtility.SetDirty(creature);

        // 3d. Save & Refresh
        AssetDatabase.SaveAssets();
        AssetDatabase.Refresh();
        Debug.Log($"--- Step 3 Complete for: {characterName} ({(createdNew ? "Created" : "Updated")}) ---");
        EditorUtility.DisplayDialog("Step 3 Complete", $"Scriptable Object {(createdNew ? "created" : "updated")} for '{characterName}'.", "OK");
    }


    // --- Selective Update: Prefabs in SO ---
    public void UpdateScriptableObjectPrefabsOnly()
    {
        if (!ValidateBaseAssets()) return; // Need base assets to check paths
        string characterName = GetCharacterName();
        if (characterName == null) return;

        string scriptableObjectPath = GetScriptableObjectPath(characterName);
        string strategicPrefabPath = GetStrategicVariantPath(characterName);
        string unitPrefabPath = GetUnitVariantPath(characterName);
        string tacticalBasePrefabPath = _unitPrefabPaths[_selectedUnitIndex]; // Path from selection
        string idleSpritePath = GetIdleSpritePathFromAnimation(characterName); // For icon

        // Validate that the SO and the prefabs *to be assigned* exist
        if (!ValidateGeneratedAssetsForStep(scriptableObjectPath, strategicPrefabPath, unitPrefabPath, tacticalBasePrefabPath)) return;
        // Icon sprite path is optional for this update, so don't strictly validate its existence here.

        Debug.Log($"--- Updating Prefabs in SO for: {characterName} ---");

        // 1. Load SO
        ScriptableObject creature = AssetDatabase.LoadAssetAtPath<ScriptableObject>(scriptableObjectPath);
        // Existence checked by ValidateGeneratedAssetsForStep

        // 2. Load Prefabs & Icon
        var strategicPrefab = AssetDatabase.LoadAssetAtPath<HexCreatureView>(strategicPrefabPath);
        var unitPrefab = AssetDatabase.LoadAssetAtPath<HexCreatureSquadUnitView>(unitPrefabPath);
        var tacticalBasePrefab = AssetDatabase.LoadAssetAtPath<HexCreatureSquadView>(tacticalBasePrefabPath);
        Sprite iconSprite = null;
        if (!string.IsNullOrEmpty(idleSpritePath))
        {
            iconSprite = AssetDatabase.LoadAssetAtPath<Sprite>(idleSpritePath);
        }

        // Existence of prefabs checked by validation. Check for nulls just in case.
        if (strategicPrefab == null || unitPrefab == null || tacticalBasePrefab == null)
        {
            Debug.LogError($"Failed to load one or more required prefabs needed for assignment even after validation. Aborting update."); return;
        }
        if (iconSprite == null)
        {
            Debug.LogWarning($"Icon sprite not found or path could not be determined (checked: {idleSpritePath ?? "null"}). SO icon will not be updated.");
        }


        // 3. Apply ONLY prefab/icon properties
        var so = new SerializedObject(creature);
        so.Update();

        SafeSetObjectProperty(so, "_icon", iconSprite); // Update icon as well
        SafeSetObjectProperty(so, "_strategicMapPrefab", strategicPrefab);
        SafeSetObjectProperty(so, "_tacticalMapUnitPrefab", unitPrefab);
        SafeSetObjectProperty(so, "_tacticalMapPrefab", tacticalBasePrefab);

        so.ApplyModifiedProperties();
        EditorUtility.SetDirty(creature);

        // 4. Save & Refresh
        AssetDatabase.SaveAssets();
        AssetDatabase.Refresh();
        Debug.Log($"--- SO Prefab Update Complete for: {characterName} ---");
        EditorUtility.DisplayDialog("Update Complete", $"Prefab references and icon updated in Scriptable Object for '{characterName}'.", "OK");
    }


    #endregion

    #region Animation and Controller Creation (Internal)

    private Dictionary<string, List<Sprite>> LoadAndGroupSprites(string characterName, string spriteAssetDirectory)
    {
        var groupedSprites = new Dictionary<string, List<Sprite>>();
        bool errorsFound = false;

        // Find sprite assets using AssetDatabase for reliability within the project
        string[] guids = AssetDatabase.FindAssets($"t:Sprite", new[] { spriteAssetDirectory });
        var spritePaths = guids.Select(AssetDatabase.GUIDToAssetPath)
                               .Where(p => Path.GetDirectoryName(p).Replace("\\", "/") == spriteAssetDirectory.Replace("\\", "/")) // Ensure direct children
                               .Where(p => !Path.GetFileName(p).Contains("_outline")) // Exclude outlines
                               .ToList();

        // Custom sorting: extract trailing numbers for frame order
        spritePaths.Sort((a, b) => {
            string nameA = Path.GetFileNameWithoutExtension(a);
            string nameB = Path.GetFileNameWithoutExtension(b);
            var matchA = Regex.Match(nameA, @"(\d+)$"); // Match numbers at the very end
            var matchB = Regex.Match(nameB, @"(\d+)$");

            int numA = matchA.Success ? int.Parse(matchA.Groups[1].Value) : int.MaxValue; // Put non-numbered last
            int numB = matchB.Success ? int.Parse(matchB.Groups[1].Value) : int.MaxValue;

            int numCompare = numA.CompareTo(numB);
            return numCompare != 0 ? numCompare : String.Compare(nameA, nameB, StringComparison.OrdinalIgnoreCase);
        });


        if (spritePaths.Count == 0)
        {
            Debug.LogError($"No sprites found directly in directory: {spriteAssetDirectory}. Make sure sprites are imported correctly as 'Sprite (2D and UI)'.");
            return null; // Cannot proceed without sprites
        }

        Debug.Log($"Found {spritePaths.Count} sprites in '{spriteAssetDirectory}'. Processing...");


        foreach (var path in spritePaths)
        {
            var sprite = AssetDatabase.LoadAssetAtPath<Sprite>(path);
            if (sprite == null)
            {
                Debug.LogWarning($"Failed to load sprite at path: {path}");
                continue;
            }

            bool matched = false;
            foreach (var state in animationStates)
            {
                // Use IgnoreCase for flexibility, ensure pattern is valid regex
                try
                {
                    if (!string.IsNullOrEmpty(state.pattern) && Regex.IsMatch(sprite.name, state.pattern, RegexOptions.IgnoreCase))
                    {
                        if (!groupedSprites.ContainsKey(state.name))
                            groupedSprites[state.name] = new List<Sprite>();

                        groupedSprites[state.name].Add(sprite);
                        matched = true;
                        // Debug.Log($"Matched sprite '{sprite.name}' to state '{state.name}'");
                        break; // Match found, move to next sprite
                    }
                }
                catch (ArgumentException ex)
                {
                    Debug.LogError($"Invalid Regex pattern '{state.pattern}' for state '{state.name}': {ex.Message}");
                    errorsFound = true; // Mark error but continue processing other states/sprites
                }
            }
            if (!matched)
            {
                // Optional: Log unmatched sprites if needed for debugging naming conventions
                // Debug.LogWarning($"Sprite '{sprite.name}' did not match any animation state patterns.");
            }
        }

        // Validate required animations
        foreach (var state in animationStates)
        {
            if (state.required && (!groupedSprites.ContainsKey(state.name) || groupedSprites[state.name].Count == 0))
            {
                Debug.LogError($"Required animation state '{state.name}' is missing sprites for '{characterName}'. Pattern was: '{state.pattern}'");
                errorsFound = true;
            }
            else if (groupedSprites.ContainsKey(state.name))
            {
                //Debug.Log($"State '{state.name}': Found {groupedSprites[state.name].Count} sprites.");
            }
            else if (!state.required)
            {
                //Debug.Log($"Optional state '{state.name}': No sprites found (Pattern: '{state.pattern}').");
            }
        }

        if (groupedSprites.Count == 0 && !errorsFound)
        {
            Debug.LogError($"No sprites were matched to any animation patterns in '{spriteAssetDirectory}'. Check patterns and sprite names.");
            return null;
        }

        return errorsFound ? null : groupedSprites;
    }

    private void CreateAnimationAssets(string characterName, string artDirectory, Dictionary<string, List<Sprite>> groupedSprites)
    {
        int createdOrUpdated = 0;
        int deleted = 0;
        foreach (var state in animationStates)
        {
            string clipPath = GetAnimationPath(characterName, state.name);
            bool spritesExist = groupedSprites.TryGetValue(state.name, out var spriteList) && spriteList.Count > 0;

            if (spritesExist)
            {
                var clip = CreateAnimationClip($"{characterName}_{state.name}", spriteList, state.looped);
                bool existed = AssetDatabase.LoadAssetAtPath<AnimationClip>(clipPath) != null;
                if (existed) AssetDatabase.DeleteAsset(clipPath); // Overwrite
                AssetDatabase.CreateAsset(clip, clipPath);
                createdOrUpdated++;
                //Debug.Log($"{(existed ? "Updated" : "Created")} animation clip: {clipPath}");
            }
            else
            {
                // Delete existing optional clip if sprites were removed
                if (!state.required && AssetDatabase.LoadAssetAtPath<AnimationClip>(clipPath) != null)
                {
                    AssetDatabase.DeleteAsset(clipPath);
                    deleted++;
                    Debug.Log($"Deleted optional animation clip (no sprites): {clipPath}");
                }
                else if (state.required)
                {
                    // Error should have been caught by LoadAndGroupSprites
                    Debug.LogError($"Missing required animation sprites for '{state.name}'. Cannot create clip.");
                }
            }
        }
        Debug.Log($"Animation asset creation/update finished. Created/Updated: {createdOrUpdated}, Deleted: {deleted}.");
    }

    private AnimationClip CreateAnimationClip(string clipName, List<Sprite> sprites, bool loop)
    {
        var clip = new AnimationClip { name = clipName, frameRate = this.frameRate };

        var binding = new EditorCurveBinding
        {
            type = typeof(SpriteRenderer),
            path = "", // Assumes SpriteRenderer is on the root of the animated object
            propertyName = "m_Sprite"
        };

        ObjectReferenceKeyframe[] keyframes;

        if (sprites == null || sprites.Count == 0)
        {
            Debug.LogWarning($"No sprites provided for animation clip '{clipName}'. Creating empty clip.");
            keyframes = new ObjectReferenceKeyframe[0];
        }
        else if (sprites.Count == 1)
        {
            // For single frame animations, hold the frame for a minimal duration
            keyframes = new ObjectReferenceKeyframe[]
            {
                new ObjectReferenceKeyframe { time = 0, value = sprites[0] },
                new ObjectReferenceKeyframe { time = 1.0f / clip.frameRate, value = sprites[0] } // Ensure it displays
            };
        }
        else
        {
            keyframes = sprites.Select((s, i) => new ObjectReferenceKeyframe
            {
                time = i / clip.frameRate, // Time based on frame index and rate
                value = s
            }).ToArray();
        }

        AnimationUtility.SetObjectReferenceCurve(clip, binding, keyframes);

        // Set loop time using AnimationClipSettings
        AnimationClipSettings settings = AnimationUtility.GetAnimationClipSettings(clip);
        settings.loopTime = loop;
        AnimationUtility.SetAnimationClipSettings(clip, settings);

        return clip;
    }

    private AnimatorOverrideController CreateOrUpdateOverrideController(string characterName, string artDirectory, Dictionary<string, List<Sprite>> sprites)
    {
        string controllerPath = GetOverrideControllerPath(characterName);
        var baseController = AssetDatabase.LoadAssetAtPath<AnimatorController>(baseControllerPath);
        if (baseController == null) { Debug.LogError($"Base controller not found: {baseControllerPath}"); return null; }

        var overrideController = AssetDatabase.LoadAssetAtPath<AnimatorOverrideController>(controllerPath);
        bool createdNew = false;
        if (overrideController == null)
        {
            overrideController = new AnimatorOverrideController(baseController) { name = Path.GetFileNameWithoutExtension(controllerPath) };
            AssetDatabase.CreateAsset(overrideController, controllerPath);
            createdNew = true;
            Debug.Log($"Created new Override Controller: {controllerPath}");
        }
        else
        {
            if (overrideController.runtimeAnimatorController != baseController)
            {
                Debug.LogWarning($"Override controller '{controllerPath}' linked to wrong base ('{overrideController.runtimeAnimatorController?.name ?? "null"}'). Relinking to '{baseController.name}'.");
                // Re-assigning the runtime controller clears existing overrides, so we need to re-apply them all.
                overrideController.runtimeAnimatorController = baseController;
                // Force re-application by clearing the list (it will be rebuilt below)
                overrideController.ApplyOverrides(new List<KeyValuePair<AnimationClip, AnimationClip>>());
                EditorUtility.SetDirty(overrideController); // Mark dirty after changing base
                Debug.Log("Existing overrides cleared due to base controller change. Will re-apply based on current animations.");
            }
            //Debug.Log($"Updating existing Override Controller: {controllerPath}");
        }

        var overridesToApply = new List<KeyValuePair<AnimationClip, AnimationClip>>();
        var currentOverrides = new AnimationClipOverrides(overrideController.overridesCount);
        overrideController.GetOverrides(currentOverrides); // Get current state

        foreach (var state in animationStates)
        {
            // Find corresponding clip in base controller (case-insensitive fallback)
            AnimationClip originalClip = baseController.animationClips.FirstOrDefault(c => c.name == state.name)
                                      ?? baseController.animationClips.FirstOrDefault(c => c.name.Equals(state.name, StringComparison.OrdinalIgnoreCase));

            if (originalClip == null)
            {
                Debug.LogWarning($"Base clip for state '{state.name}' not found in '{baseController.name}'. Skipping override.");
                continue;
            }

            string newClipPath = GetAnimationPath(characterName, state.name);
            AnimationClip newClip = AssetDatabase.LoadAssetAtPath<AnimationClip>(newClipPath); // Load the generated clip asset

            AnimationClip currentOverrideClip = currentOverrides[originalClip.name];

            if (newClip != null)
            { // Animation exists for this state
                if (currentOverrideClip != newClip)
                { // Only add if it's different from current
                    overridesToApply.Add(new KeyValuePair<AnimationClip, AnimationClip>(originalClip, newClip));
                }
            }
            else
            { // Animation doesn't exist (or wasn't generated)
                if (currentOverrideClip != null)
                { // Remove existing override if there was one
                    overridesToApply.Add(new KeyValuePair<AnimationClip, AnimationClip>(originalClip, null));
                    Debug.Log($"Override for '{originalClip.name}' will be removed (clip missing/not generated).");
                }
            }
        }

        if (overridesToApply.Count > 0)
        {
            overrideController.ApplyOverrides(overridesToApply);
            EditorUtility.SetDirty(overrideController);
            Debug.Log($"Applied/Updated {overridesToApply.Count} overrides to {overrideController.name}.");
        }
        else if (!createdNew)
        {
            Debug.Log($"No override changes needed for {overrideController.name}.");
        }

        return overrideController;
    }


    #endregion

    #region Prefab Variant Creation (Internal - Refactored)

    // --- Prefab Variant Creation Helper ---
    /// <summary>
    /// Creates or updates a prefab variant, applying sprite and animator controller overrides.
    /// </summary>
    /// <returns>The root component (e.g., HexCreatureView) from the created/updated variant asset, or null on failure.</returns>
    private UnityEngine.Component CreatePrefabVariant(string basePrefabPath, string destinationVariantPath, Type rootComponentType, Sprite idleSprite, AnimatorOverrideController overrideController)
    {
        // 1. Load Base Prefab Asset
        GameObject basePrefab = AssetDatabase.LoadAssetAtPath<GameObject>(basePrefabPath);
        if (basePrefab == null) { Debug.LogError($"Base prefab not found: {basePrefabPath}"); return null; }

        // 2. Instantiate Base Prefab (creates an in-memory instance linked to the base)
        GameObject instance = PrefabUtility.InstantiatePrefab(basePrefab) as GameObject;
        if (instance == null) { Debug.LogError($"Failed to instantiate base prefab: {basePrefabPath}"); return null; }

        GameObject savedVariantAsset = null;
        bool needsSave = false; // Track if modifications occurred

        try
        {
            // 3. Modify the Instance - Apply Overrides
            // Check for expected root component (logging error if missing but not halting)
            var viewComponent = instance.GetComponent(rootComponentType);
            if (viewComponent == null)
            {
                Debug.LogError($"Required component '{rootComponentType.Name}' missing on base prefab: {basePrefabPath}");
            }

            // Apply Sprite override if necessary
            SpriteRenderer renderer = instance.GetComponentInChildren<SpriteRenderer>(true); // Check children too, include inactive
            if (renderer != null)
            {
                if (idleSprite != null && renderer.sprite != idleSprite)
                {
                    Undo.RecordObject(renderer, "Set Sprite"); // For Undo stack support
                    renderer.sprite = idleSprite;
                    PrefabUtility.RecordPrefabInstancePropertyModifications(renderer);
                    needsSave = true;
                    //Debug.Log($"Applied sprite '{idleSprite.name}' override.");
                }
                else if (idleSprite == null && renderer.sprite != null)
                {
                    // Optional: Clear sprite if provided sprite is null and current sprite isn't
                    // Undo.RecordObject(renderer, "Clear Sprite");
                    // renderer.sprite = null;
                    // PrefabUtility.RecordPrefabInstancePropertyModifications(renderer);
                    // needsSave = true;
                }
            }
            else { Debug.LogWarning($"SpriteRenderer not found on base prefab or its children: {basePrefabPath}"); }

            // Apply Animator override if necessary
            Animator animator = instance.GetComponentInChildren<Animator>(true); // Check children too, include inactive
            if (animator != null)
            {
                if (overrideController != null && animator.runtimeAnimatorController != overrideController)
                {
                    Undo.RecordObject(animator, "Set Animator Controller");
                    animator.runtimeAnimatorController = overrideController;
                    PrefabUtility.RecordPrefabInstancePropertyModifications(animator);
                    needsSave = true;
                    //Debug.Log($"Applied controller '{overrideController.name}' override.");
                }
                // Could add checks/overrides for other animator properties like Apply Root Motion if needed
            }
            else { Debug.LogWarning($"Animator not found on base prefab or its children: {basePrefabPath}"); }


            // 4. Save the Modified Instance as a Prefab Variant Asset
            // This step creates the .prefab file on disk or updates the existing one.
            savedVariantAsset = PrefabUtility.SaveAsPrefabAsset(instance, destinationVariantPath);

            if (savedVariantAsset != null)
            {
                // Return the specific component from the *saved asset* itself
                // Use GetComponent instead of GetComponent(Type) for direct casting later
                return savedVariantAsset.GetComponent(rootComponentType) as UnityEngine.Component;
            }
            else
            {
                Debug.LogError($"Failed to save prefab variant to: {destinationVariantPath}");
                return null;
            }
        }
        catch (Exception e)
        {
            Debug.LogError($"Error during prefab variant creation ({destinationVariantPath}): {e}");
            return null;
        }
        finally
        {
            // 5. IMPORTANT: Destroy the temporary in-memory instance
            if (instance != null)
            {
                UnityEngine.Object.DestroyImmediate(instance);
            }
        }
    }


    #endregion

    #region Scriptable Object Property Helpers (Internal)

    // Static helpers to safely set properties via SerializedObject, avoiding errors if property name changes/doesn't exist
    private static void SafeSetObjectProperty(SerializedObject so, string propertyName, UnityEngine.Object value)
    {
        SerializedProperty prop = so.FindProperty(propertyName);
        if (prop != null)
        {
            if (prop.objectReferenceValue != value) // Only set if changed
                prop.objectReferenceValue = value;
        }
        else
        {
            Debug.LogWarning($"Property '{propertyName}' not found on SO '{so.targetObject.name}'. Cannot set value.");
        }
    }
    private static void SafeSetFloatProperty(SerializedObject so, string propertyName, float value)
    {
        SerializedProperty prop = so.FindProperty(propertyName);
        if (prop != null && prop.propertyType == SerializedPropertyType.Float)
        {
            if (!Mathf.Approximately(prop.floatValue, value)) // Only set if changed
                prop.floatValue = value;
        }
        else
        {
            Debug.LogWarning($"Float Property '{propertyName}' not found or invalid type on SO '{so.targetObject.name}'.");
        }
    }

    private static void SafeSetIntProperty(SerializedObject so, string propertyName, int value)
    {
        SerializedProperty prop = so.FindProperty(propertyName);
        if (prop != null && prop.propertyType == SerializedPropertyType.Integer)
        {
            if (prop.intValue != value) // Only set if changed
                prop.intValue = value;
        }
        else
        {
            Debug.LogWarning($"Integer Property '{propertyName}' not found or invalid type on SO '{so.targetObject.name}'.");
        }
    }

    #endregion

    #region Helper Classes

    // Helper class for AnimatorOverrideController.GetOverrides
    // (Needed because the List<KeyValuePair> version is obsolete)
    public class AnimationClipOverrides : List<KeyValuePair<AnimationClip, AnimationClip>>
    {
        public AnimationClipOverrides(int capacity) : base(capacity) { }

        public AnimationClip this[string originalClipName]
        {
            get
            {
                // Find the override based on the original clip's name
                int index = this.FindIndex(kvp => kvp.Key != null && kvp.Key.name.Equals(originalClipName));
                return index != -1 ? this[index].Value : null;
            }
        }
    }

    #endregion
}