using UnityEditor;
using UnityEngine;
using PK;

[CustomEditor(typeof(WeatherSystem))]
public class WeatherSystemEditor : Editor
{
    private HexWeather[] weatherStates;
    private string[] weatherStateNames;
    private int selectedWeatherIndex = 0;
    private HexWeather.Severity selectedSeverity = HexWeather.Severity.None;

    private void OnEnable()
    {
        // Load all HexWeather assets
        weatherStates = Resources.LoadAll<HexWeather>("Weathers");
        weatherStateNames = new string[weatherStates.Length];
        for (int i = 0; i < weatherStates.Length; i++)
        {
            weatherStateNames[i] = weatherStates[i].weatherName;
        }
    }

    public override void OnInspectorGUI()
    {
        DrawDefaultInspector();

        if (WeatherSystem.Instance == null)
        {
            EditorGUILayout.HelpBox("WeatherSystem is not running.", MessageType.Warning);
            return;
        }

        EditorGUILayout.Space();

        WeatherSystem weatherSystem = (WeatherSystem)target;

        // Display current weather state
        EditorGUILayout.LabelField("Current Weather System State", EditorStyles.boldLabel);
        EditorGUILayout.LabelField("Weather Name:", weatherSystem.currentWeatherState != null ? weatherSystem.currentWeatherState.weatherName : "None");
        EditorGUILayout.LabelField("Severity:", weatherSystem.currentSeverity.ToString());

        // Dropdown to select weather state
        EditorGUILayout.Space();
        EditorGUILayout.LabelField("Change Weather", EditorStyles.boldLabel);
        selectedWeatherIndex = EditorGUILayout.Popup("Weather State", selectedWeatherIndex, weatherStateNames);
        selectedSeverity = (HexWeather.Severity)EditorGUILayout.EnumPopup("Severity", selectedSeverity);

        if (GUILayout.Button("Apply Weather"))
        {
            if (weatherStates.Length > 0 && selectedWeatherIndex < weatherStates.Length)
            {
                weatherSystem.SetWeather(weatherStates[selectedWeatherIndex], selectedSeverity);
            }
        }

        // Toggle particle system features
        EditorGUILayout.Space();
        EditorGUILayout.LabelField("Toggle Particle System Features", EditorStyles.boldLabel);

        weatherSystem.ToggleFeature("Noise", EditorGUILayout.Toggle("Enable Noise", weatherSystem.noiseModule.enabled));
        weatherSystem.ToggleFeature("VelocityOverLifetime", EditorGUILayout.Toggle("Enable Velocity Over Lifetime", weatherSystem.velocityModule.enabled));
        weatherSystem.ToggleFeature("Emission", EditorGUILayout.Toggle("Enable Emission", weatherSystem.emissionModule.enabled));

    }
}
