﻿using UnityEngine;

namespace PK
{
    [CreateAssetMenu(menuName = "Hex/Object")]
    public class HexObject : HexUniqueScriptableObject
    {
        [SerializeField] private KeyValue<HexTerrain, Sprite>[] _variants;
        [SerializeField] private bool _isAligned = true;
        [SerializeField] private KeyValue<Vector2Int, int>[] _interactionMask;
        [SerializeField] private Vector2 _offset;
        [SerializeField] private int _shadowGroup;
        [SerializeField] private Vector4 _shadowOffset;
        [SerializeReference] private BaseInteraction _interaction;
        [SerializeField] private AudioPreset _interactionSfx;
        [SerializeField] private AudioPreset _disableSfx;

        public bool IsAligned { get { return _isAligned; } }
        public KeyValue<Vector2Int, int>[] InteractionMask { get { return _interactionMask; } }
        public Vector2 Offset { get { return _offset; } }
        public int ShadowGroup { get { return _shadowGroup; } }
        public Vector4 ShadowOffset { get { return _shadowOffset; } }
        public BaseInteraction Interaction { get { return _interaction; } }
        public AudioPreset InteractionSfx { get { return _interactionSfx; } }
        public AudioPreset DisableSfx { get { return _disableSfx; } }

        public Sprite AnySprite
        {
            get
            {
                if (_variants == null || _variants.Length == 0)
                {
                    return null;
                }
                foreach (KeyValue<HexTerrain, Sprite> variant in _variants)
                {
                    if (variant.Value != null)
                    {
                        return variant.Value;
                    }
                }
                return null;
            }
        }

        public Sprite GetSprite(HexTerrain terrain)
        {
            foreach (KeyValue<HexTerrain, Sprite> variant in _variants)
            {
                if (variant.Key == terrain && variant.Value != null)
                {
                    return variant.Value;
                }
            }
            return AnySprite;
        }
    }

    // Mask details
    // 0 - empty
    // 1 - blocked
    // 2 - visitable
    // 3 - blockvisitable
    // Visitable and blockvisitable store additional data in 1-3 bytes about 6 nearby hex directions
    // Hex directions are stored counter-clockwise starting from south-west in 6 bits

    public enum HexObjectMask
    {
        Empty,
        Blocked,
        Visitable,
        BlockVisitable
    }

    public static class HexObjectMaskHelper
    {
        private static int FIRST_BYTE_MASK = 0xFF;
        private static int SECOND_BYTE_MASK = 0xFF00;
        private static int THIRD_BYTE_MASK = 0xFF0000;

        public static HexObjectMask GetType(int mask)
        {
            return (HexObjectMask)(mask & FIRST_BYTE_MASK);
        }

        public static void GetTilesData(int interactionMask, bool isEnter, bool[] tilesData)
        {
            int offset = isEnter ? 8 : 16;
            int values = interactionMask >> offset;
            tilesData[0] = (values & 1) > 0;
            tilesData[1] = (values & 2) > 0;
            tilesData[2] = (values & 4) > 0;
            tilesData[3] = (values & 8) > 0;
            tilesData[4] = (values & 16) > 0;
            tilesData[5] = (values & 32) > 0;
        }

        public static bool[] GetTilesData(int interactionMask, bool isEnter)
        {
            bool[] tilesData = new bool[6];
            GetTilesData(interactionMask, isEnter, tilesData);
            return tilesData;
        }

        public static int ModifyTilesData(int index, int interactionMask, bool isEnter, bool enabled)
        {
            int offset = isEnter ? 8 : 16;
            if (enabled)
            {
                return interactionMask | 1 << (offset + index);
            }
            return interactionMask & ~(1 << (offset + index));
        }

        public static int SwitchMaskModeToNext(int interactionMask)
        {
            int newInteractionMask = ((interactionMask & FIRST_BYTE_MASK) + 1) % 4;
            return newInteractionMask;
        }

        public static bool IsTileDataEmpty(int interactionMask)
        {
            return (interactionMask >> 8) == 0;
        }

        // Use this method in case of empty object data when initializing the map
        public static int GetDefaultTileData(int interactionMask)
        {
            switch (GetType(interactionMask))
            {
                case HexObjectMask.Visitable:
                    return interactionMask | (39 << 8) | (39 << 16);
                case HexObjectMask.BlockVisitable:
                    return interactionMask | (63 << 8);
                default:
                    return interactionMask;
            }
        }
    }
}
