﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    [System.Serializable]
    public class Savefile
    {
        [System.Serializable]
        public class MapData
        {
            [SerializeField] private HexMapModel _map;
            [SerializeField] private bool _ignoreInteractionMask;
            [SerializeField] private bool _isStrategic;
            [SerializeField] private bool[] _obstacleMask;
            [SerializeField] private bool[] _fogOfWarMask;
            [SerializeField] private byte[] _movementCostMask;

            public HexMapModel Map { get { return _map; } }
            public bool[] ObstacleMask { get { return _obstacleMask; } }
            public bool[] FogOfWarMask { get { return _fogOfWarMask; } }
            public byte[] MovementCostMask { get { return _movementCostMask; } }

            public MapData(HexMap map, bool ignoreInteractionMask)
            {
                _map = map.Model;
                _ignoreInteractionMask = ignoreInteractionMask;
                _isStrategic = !ignoreInteractionMask;
                _obstacleMask = new bool[map.Size.x * map.Size.y];
                _fogOfWarMask = new bool[_obstacleMask.Length];
                _movementCostMask = new byte[_obstacleMask.Length];
            }

            public void RefreshObstacleMask()
            {
                if (_ignoreInteractionMask)
                {
                    for (int i = 0, n = _map.Width * _map.Height; i < n; i++)
                    {
                        _obstacleMask[i] = _map.GetTerrainTile(i) == 0 || _map.GetInteractionMask(i) > 0;
                    }
                }
                else
                {
                    for (int i = 0, n = _map.Width * _map.Height; i < n; i++)
                    {
                        HexObjectMask mask = HexObjectMaskHelper.GetType(_map.GetInteractionMask(i));
                        _obstacleMask[i] = _map.GetTerrainTile(i) == 0 || (mask == HexObjectMask.Blocked || mask == HexObjectMask.BlockVisitable);
                    }
                }
                // TODO refactor
                foreach (HexEntityModel entity in Enumerable.Empty<HexEntityModel>().Union(_map.GetEnumerable<HexCreatureModel>()).Union(_map.GetEnumerable<HexHeroModel>()).Union(_map.GetEnumerable<HexCreatureSquadModel>()))
                {
                    Vector2Int position = entity.Position;
                    _obstacleMask[position.y * _map.Width + position.x] = true;
                }
            }

            public void RevealFogOfWar(Vector2Int position)
            {
                _fogOfWarMask[position.y * _map.Width + position.x] = true;
            }

            public void RefreshMovementCostMask()
            {
                for (int i = 0, n = _map.Width * _map.Height; i < n; i++)
                {
                    ulong tileUid = _map.GetTerrainTile(i);
                    if (tileUid != 0)
                    {
                        (byte type, byte index) = _map.GetRoad(i);
                        if (type > 0)
                        {
                            HexRoad road = HexDatabase.Instance.GetRoad(type);
                            _movementCostMask[i] = _isStrategic ? road.StrategicMovementCost : road.TacticalMovementCost;
                        }
                        else
                        {
                            HexTile tile = HexDatabase.Instance.GetTile(tileUid);
                            if (tile != null)
                            {
                                HexTerrain terrain = tile.Terrain;
                                _movementCostMask[i] = _isStrategic ? terrain.StrategicMovementCost : terrain.TacticalMovementCost;
                            }
                        }
                        
                    }
                    else
                    {
                        _movementCostMask[i] = byte.MaxValue;
                    }
                }
            }

            public bool IsEmpty(Vector2Int position)
            {
                if (!_map.IsInBounds(position))
                {
                    return false;
                }
                return !_obstacleMask[position.y * _map.Width + position.x];
            }

            public HexContent GetContent(Vector2Int position)
            {
                if (!_map.IsInBounds(position))
                {
                    return new HexContent(HexContent.ContentType.Obstacle);
                }
                if (_fogOfWarMask[position.y * _map.Width + position.x] == false)
                {
                    return new HexContent(HexContent.ContentType.Hidden);
                }
                foreach(HexEntityModel entity in _map.GetEnumerable())
                {
                    if (entity is HexCreatureModel creature)
                    {
                        if (creature.Position == position)
                        {
                            return new HexContent(HexContent.ContentType.Creature, creature.Id);
                        }
                        continue;
                    }
                    if (entity is HexCreatureSquadModel creatureSquad)
                    {
                        if (creatureSquad.Position == position)
                        {
                            return new HexContent(HexContent.ContentType.CreatureSquad, creatureSquad.Id, creatureSquad.Player);
                        }
                        continue;
                    }
                    if (entity is HexObjectModel prop)
                    {
                        if (prop.Position == position && prop.IsInteractable)
                        {
                            return new HexContent(HexContent.ContentType.Interactable, prop.Id);
                        }
                        continue;
                    }
                }
                // Avoid blocking of objects by heroes
                foreach (HexHeroModel hero in _map.GetEnumerable<HexHeroModel>())
                {
                    if (hero.Position == position)
                    {
                        return new HexContent(HexContent.ContentType.Hero, hero.Id, hero.Player);
                    }
                }
                if (_obstacleMask[position.y * _map.Width + position.x])
                {
                    return new HexContent(HexContent.ContentType.Obstacle);
                }
                return new HexContent(HexContent.ContentType.Empty);
            }

            public bool IsVisible(Vector2Int position)
            {
                if (!_map.IsInBounds(position))
                {
                    return false;
                }
                if (_fogOfWarMask[position.y * _map.Width + position.x] == false)
                {
                    return false;
                }
                return true;
            }
        }

        [System.Serializable]
        public class HeroData
        {
            [SerializeField] private List<HexCreatureSquadModel> _squads;

            public List<HexCreatureSquadModel> Squads { get { return _squads; } set { _squads = value; } }
        }

        [SerializeField] private MapData _strategicMap;
        [SerializeField] private MapData _tacticalMap;
        [SerializeField] private int _day;
        [SerializeField] private Player _player = Player.Unknown;
        [SerializeField] private SerializableDictionary<ulong, HeroData> _heroesData = new();

        public MapData StrategicMap { get { return _strategicMap; } }
        public MapData TacticalMap { get { return _tacticalMap; } }
        public bool IsStrategicMap { get { return _tacticalMap == null; } }
        public int Day { get { return _day; } set { _day = value; } }
        public Player Player { get { return _player; } set { _player = value; } }
        
        public Savefile(HexMap strategicMap)
        {
            strategicMap.Model.RefreshInteractionMask();
            _strategicMap = new MapData(strategicMap, false);
            _strategicMap.RefreshObstacleMask();
            _strategicMap.RefreshMovementCostMask();
        }

        public HeroData GetHeroData(HexHeroModel hero)
        {
            if (!_heroesData.TryGetValue(hero.Id, out HeroData data))
            {
                data = new HeroData()
                {
                    Squads = hero.StartingCreatures.Select((c) => new HexCreatureSquadModel(c.Creature.Uid, 0, Vector2Int.zero, hero.Player, Mathf.Min(c.Count, c.Creature.SoldiersPerUnit), c.Creature.Health)).ToList()
                };
                _heroesData[hero.Id] = data;
            }
            return data;
        }

        public void AddHeroSquads(HexHeroModel hero, HexCreature creature, int count)
        {
            HeroData heroData = GetHeroData(hero);
            heroData.Squads.Add(new HexCreatureSquadModel(creature.Uid, 0, Vector2Int.zero, hero.Player, Mathf.Min(count, creature.SoldiersPerUnit), creature.Health));
        }

        public void StartBattle(HexMap tacticalMap)
        {
            tacticalMap.Model.RefreshInteractionMask();
            _tacticalMap = new MapData(tacticalMap, true);
            _tacticalMap.RefreshObstacleMask();
            _tacticalMap.RefreshMovementCostMask();
        }

        public void EndBattle()
        {
            _tacticalMap = null;
        }

        public void Initialize()
        {
            _strategicMap.Map.RefreshInteractionMask();
            _strategicMap.RefreshObstacleMask();
            _strategicMap.RefreshMovementCostMask();
            _tacticalMap = null;
        }

        public bool IsEmpty(Vector2Int position)
        {
            if (_tacticalMap == null)
            {
                return _strategicMap.IsEmpty(position);
            }
            else
            {
                return _tacticalMap.IsEmpty(position);
            }
        }

        public HexContent GetContent(Vector2Int position)
        {
            if (_tacticalMap == null)
            {
                return _strategicMap.GetContent(position);
            }
            else
            {
                return _tacticalMap.GetContent(position);
            }
        }
    }
}
