﻿using DG.Tweening;
using PK.Strategic;
using PK.Tactical;
using UnityEngine;

namespace PK
{
    public class GameClient : MonoBehaviour
    {
        const string LOCKER = "GameClient";

        public static GameClient Instance { get { return _instance; } }

        private static GameClient _instance;

        [SerializeField] private Canvas _canvas;

        private Savefile _savefile;
        private HexMap _map;
        private StrategicGameMediator _strategicMediator;
        private TacticalGameMediator _tacticalMediator;
        private TacticalAIWrapper _aiWrapper;

        private BattleData _battleData;
        private System.Action<BattleData> _callback;

        public Canvas Canvas { get { return _canvas; } }

        private void Awake()
        {
            QualitySettings.vSyncCount = 1;

            _instance = this;
            _strategicMediator = GetComponentInChildren<StrategicGameMediator>();
            _tacticalMediator = GetComponentInChildren<TacticalGameMediator>();
            _aiWrapper = _tacticalMediator.gameObject.AddComponent<TacticalAIWrapper>();
        }

        private void Start()
        {
#if UNITY_EDITOR
            HexMap map = UnityEditor.AssetDatabase.LoadAssetAtPath<HexMap>(UnityEditor.AssetDatabase.GUIDToAssetPath(UnityEditor.EditorPrefs.GetString("EditorMap")));
            if (map != null)
            {
                InitializeWithMap(map);
            }
            else
            {
                Debug.LogError("Can't find a map");
            }
#endif
        }

        private void Update()
        {
            if (_savefile != null && Input.GetKeyDown(KeyCode.F5))
            {
                Save();
            }
            if (Input.GetKey(KeyCode.F9))
            {
                Load();
            }
        }

        private void OnDestroy()
        {
            if (!GameClient.Instance.IsSaveExist())
            {
                Save();
            }
        }

        public void InitializeWithMap(HexMap map)
        {
            IStrategicGameMediator strategicGameMediator = _strategicMediator;
            _map = Instantiate(map);
            _savefile = new Savefile(_map);

            _strategicMediator.SetViewActive(true);
            _strategicMediator.Initialize(_savefile, _map);
            strategicGameMediator.EventManager.Get<IStrategicEvents>().OnStartBattle -= OnStartBattle;
            strategicGameMediator.EventManager.Get<IStrategicEvents>().OnStartBattle += OnStartBattle;
            _strategicMediator.StartGame();

            Strategic.InputLock.Lock(LOCKER);
            FadeView.Instance.FadeIn(1f, 0.1f, true, () =>
            {
                Strategic.InputLock.Unlock(LOCKER);
            });
        }

        private void OnStartBattle(HexMap map, ulong heroId, ulong enemyId, HexCreatureSquadModel[] enemySquads, bool canRetry, Player beginer, System.Action<BattleData> callback)
        {
            _callback = callback;
            _battleData = new BattleData(map, heroId, enemyId, enemySquads, canRetry, beginer);

            Strategic.InputLock.Lock(LOCKER);
            FadeView.Instance.FadeOut(1f, 0f, true, () =>
            {
                StartBattle(_battleData);
                FadeView.Instance.FadeIn(1f, 0.1f, true, () =>
                {
                    Strategic.InputLock.Unlock(LOCKER);
                });
            });
        }

        private void OnEndBattle(Player player)
        {
            _battleData.Winner = player;
            void ReturnToStrategicMap()
            {
                Tactical.InputLock.Lock(LOCKER);
                FadeView.Instance.FadeOut(1f, 0f, true, () =>
                {
                    EndBattle();
                    _battleData = null;

                    FadeView.Instance.FadeIn(1f, 0.1f, true, () =>
                    {
                        Tactical.InputLock.Unlock(LOCKER);
                    });
                });
            }

            ITacticalGameMediator tacticalGameMediator = _tacticalMediator;
            DOVirtual.DelayedCall(4f, () =>
            {
                if (player != tacticalGameMediator.SelfPlayer)
                {
                    if (_battleData.CanRetry)
                    {
                        TryAgainWindow.Create(_canvas.transform, (bool isYes) =>
                        {
                            if (isYes)
                            {
                                FadeView.Instance.FadeOut(1f, 0f, true, () =>
                                {
                                    StartBattle(_battleData);
                                    FadeView.Instance.FadeIn(1f, 0.1f, true, () =>
                                    {
                                        Tactical.InputLock.Unlock(LOCKER);
                                    });
                                });
                            }
                            else
                            {
                                ReturnToStrategicMap();
                            }
                        });
                    }
                    else
                    {
                        ReturnToStrategicMap();
                    }
                }
                else
                {
                    ReturnToStrategicMap();
                }
            });
        }

        private void StartBattle(BattleData data)
        {
            ITacticalGameMediator tacticalGameMediator = _tacticalMediator;
            HexMap map = Instantiate(data.Map);
            _savefile.StartBattle(map);
            _strategicMediator.SetViewActive(false);
            _tacticalMediator.SetViewActive(false);
            _tacticalMediator.SetViewActive(true);
            _tacticalMediator.Initialize(_savefile, map);
            _aiWrapper.Initialize(_savefile.TacticalMap, Player.Blue);
            tacticalGameMediator.EventManager.Get<ITacticalEvents>().OnEndBattle -= OnEndBattle;
            tacticalGameMediator.EventManager.Get<ITacticalEvents>().OnEndBattle += OnEndBattle;
            _tacticalMediator.StartGame(data.HeroId, data.EnemySquads, data.Beginer);
        }

        private void EndBattle()
        {
            ITacticalGameMediator tacticalGameMediator = _tacticalMediator;
            tacticalGameMediator.EventManager.Get<ITacticalEvents>().OnEndBattle -= OnEndBattle;
            _tacticalMediator.SetViewActive(false);
            _savefile.EndBattle();
            _strategicMediator.SetViewActive(true);
            _strategicMediator.ContinueGame();
            _callback?.Invoke(_battleData);
        }

        public void Save()
        {
            System.IO.File.WriteAllText(GetSavePath("Save"), JsonUtility.ToJson(_savefile));
            Debug.Log("Game saved.");
        }

        public void Load()
        {
            string path = GetSavePath("Save");
            if (System.IO.File.Exists(path))
            {
                _savefile = JsonUtility.FromJson<Savefile>(System.IO.File.ReadAllText(path));
                _map = ScriptableObject.CreateInstance<HexMap>();
                _map.Initialize(_savefile.StrategicMap.Map);
                _savefile.Initialize();
                _strategicMediator.SetViewActive(false);
                _strategicMediator.Initialize(_savefile, _map);
                _strategicMediator.SetViewActive(true);
                IStrategicGameMediator strategicGameMediator = _strategicMediator;
                strategicGameMediator.EventManager.Get<IStrategicEvents>().OnStartBattle -= OnStartBattle;
                strategicGameMediator.EventManager.Get<IStrategicEvents>().OnStartBattle += OnStartBattle;
                _strategicMediator.ContinueGame();
                Debug.Log("Game loaded.");
            }
        }

        private string GetSavePath(string saveName)
        {
            return System.IO.Path.Combine(Application.persistentDataPath, $"{saveName}.save");
        }

        public bool IsInitialized()
        {
            return _map != null;
        }

        public bool IsSaveExist()
        {
            string path = GetSavePath("Save");
            return System.IO.File.Exists(path);
        }

        public bool IsOnStrategicMap()
        {
            return _battleData == null;
        }

        public HexMap GetHexMap()
        {
            return _map;
        }
    }
}
