using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace PK.Strategic
{
    public interface IStrategicGameMediator
    {
        public Player SelfPlayer { get; }
        public Player CurrentPlayer { get; }
        public IEventManager EventManager { get; }
        public IControllerLocker Locker { get; }
        public HexVariablesModel Variables { get; }

        public bool IsEmpty(Vector2Int position);
        public HexContent GetContent(Vector2Int position);

        public void GetAllSquads(ulong id, List<ICreatureSquadModelForView> resultSquads);

        public (IReadOnlyList<(Vector2Int, int)> path, int stepsLeft) GetHeroPath(ulong id, Vector2Int target);
        public void GetReachableByHeroArea(ulong id, List<Vector2Int> resultPositions);
        public IHeroModelForView GetAnyHero();
        public IHeroModelForView TrySelectHero(Vector2Int position);
        public void MoveHero(ulong id, Vector2Int target);

        public void EndPlayerTurn();
    }

    public class StrategicGameMediator : MonoBehaviour, IStrategicGameMediator
    {
        const string LOCKER = "StrategicGameMediator";

        private static StrategicGameMediator _instance;

        public static IStrategicGameMediator Instance { get { return _instance; } }

        [SerializeField] private StrategicMapView _view;

        private Savefile _savefile;
        private GameController _controller;

        Player IStrategicGameMediator.SelfPlayer { get { return _controller.SelfPlayer; } }
        Player IStrategicGameMediator.CurrentPlayer { get { return _controller.CurrentPlayer; } }
        IEventManager IStrategicGameMediator.EventManager { get { return _controller.EventManager; } }
        IControllerLocker IStrategicGameMediator.Locker { get { return _controller.Locker; } }
        HexVariablesModel IStrategicGameMediator.Variables { get { return _savefile.StrategicMap.Map.GlobalVariables; } }

        private void Awake()
        {
            _instance = this;
            _controller = new GameController();
        }

        public void SetViewActive(bool active)
        {
            for (int i = 0; i < transform.childCount; i++)
            {
                GameObject child = transform.GetChild(i).gameObject;
                child.SetActive(active);
            }
            foreach (IEventReceiver receiver in GetComponentsInChildren<IEventReceiver>(true))
            {
                if (active)
                {
                    receiver.AddEvents();
                }
                else
                {
                    receiver.RemoveEvents();
                }
            }
        }

        public void Initialize(Savefile savefile, HexMap map)
        {
            _savefile = savefile;
            _controller.Initialize(_savefile);
            _view.SetMap(map, _controller.EventManager.Get<IMapEvents>(), _controller.Locker);
        }

        public void StartGame()
        {
            if (_controller.StartGame())
            {
                Simulate(() =>
                {
                    if (_controller.StartNewTurn())
                    {
                        Simulate();
                    }
                });
            }
        }

        public void ContinueGame()
        {
            _controller.ContinueGame();
        }

        bool IStrategicGameMediator.IsEmpty(Vector2Int position)
        {
            return _savefile.IsEmpty(position);
        }

        HexContent IStrategicGameMediator.GetContent(Vector2Int position)
        {
            return _savefile.GetContent(position);
        }

        void IStrategicGameMediator.GetAllSquads(ulong id, List<ICreatureSquadModelForView> resultSquads)
        {
            _controller.GetAllSquads(id, resultSquads);
        }

        (IReadOnlyList<(Vector2Int, int)> path, int stepsLeft) IStrategicGameMediator.GetHeroPath(ulong id, Vector2Int target)
        {
            return _controller.GetHeroPath(id, target);
        }

        void IStrategicGameMediator.GetReachableByHeroArea(ulong id, List<Vector2Int> resultPositions)
        {
            _controller.GetReachableByHeroArea(id, resultPositions);
        }

        IHeroModelForView IStrategicGameMediator.GetAnyHero()
        {
            return _controller.GetAnyPlayerHero();
        }

        IHeroModelForView IStrategicGameMediator.TrySelectHero(Vector2Int position)
        {
            return _controller.TrySelectHero(position);
        }

        void IStrategicGameMediator.MoveHero(ulong id, Vector2Int target)
        {
            if (_controller.MoveHero(id, target))
            {
                Simulate();
            }
        }

        void IStrategicGameMediator.EndPlayerTurn()
        {
            if (_controller.EndPlayerTurn())
            {
                Simulate();
            }
        }

        private void Simulate(Action callback = null)
        {
            StartCoroutine(SimulationCoroutine(callback));
        }

        private IEnumerator SimulationCoroutine(Action callback)
        {
            InputLock.Lock(LOCKER);
            while (true)
            {
                bool result = _controller.Simulate();

                while (_controller.Locker.IsLock)
                {
                    yield return null;
                }

                if (!result)
                {
                    break;
                }
            }
            InputLock.Unlock(LOCKER);
            callback?.Invoke();
        }
    }
}
