﻿using DG.Tweening;
using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.EventSystems;

namespace PK.Tactical
{
    public partial class TacticalMapView
    {
        private static string LOCK = "TacticalMapViewLock";

        [SerializeField] private Camera _camera;

        [Space]
        [SerializeField] private Sprite _greenDot;
        [SerializeField] private Sprite _greenCross;
        [SerializeField] private Sprite _greenAttack;
        [SerializeField] private Sprite _greenShoot;
        [SerializeField] private Sprite _greenHalfShoot;
        [SerializeField] private Sprite _greenHorse;

        [Space]
        [SerializeField] private Sprite _redDot;
        [SerializeField] private Sprite _redCross;
        [SerializeField] private Sprite _redAttack;
        [SerializeField] private Sprite _redShoot;
        [SerializeField] private Sprite _redHorse;

        [Space]
        [SerializeField] private Sprite[] _attackDirections;

        [Space]
        [Header("Camera Controls")]
        [SerializeField] private float _zoomSpeed = 1.0f;
        [SerializeField] private float _snapThreshold = 0.1f; // Percentage difference (e.g., 0.1 = 10%)
        [SerializeField] private float _snapSpeed = 5.0f;
        [SerializeField] private float _zoomSnapDelay = 1.0f;
        [SerializeField] private float _slowZoomMultiplier = 0.5f; // Speed multiplier near snap points

        private SelectionLogic _selectionLogic;
        private List<ICreatureSquadModelForView> _hoveredSquads = new();
        private float _previousOrthographicSize;
        private readonly float[] _zoomLevelMultipliers = { 2f, 1f, 0.5f, 0.25f }; // Corresponds to 50%, 100%, 200%, 400%
        private float _minOrthographicSize;
        private float _maxOrthographicSize;
        private float _zoomIdleTimer = 0f;
        private bool _isSnapping = false;
        private float _snapTargetSize;

        public event Action<ulong> OnCreatureSquadSelected;
        public event Action<ulong> OnCreatureSquadHovered;

        private void OnDisable()
        {
            _camera.orthographicSize = _previousOrthographicSize;
        }

        private void EnablePlayerControls()
        {
            _previousOrthographicSize = _camera.orthographicSize; // Store size before applying initial zoom
            _minOrthographicSize = _previousOrthographicSize * _zoomLevelMultipliers.Last();
            _maxOrthographicSize = _previousOrthographicSize * _zoomLevelMultipliers.First();

            // Set initial zoom to 200%
            _camera.orthographicSize = _previousOrthographicSize * 0.5f;
        }

        private void AddPlayerEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnStartGame += OnStartGame;
            events.OnEndTurn += OnEndTurn;

            _hoveredSquads.Clear();
        }

        private void RemovePlayerEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnStartGame -= OnStartGame;
            events.OnEndTurn -= OnEndTurn;
        }

        private void OnStartGame(Vector2Int spawnAreaPosition)
        {
            TeleportCameraTo(HexHelper.GetTilePosition(spawnAreaPosition));
        }

        private void OnEndTurn(Player player)
        {
            _selectionLogic?.Disable();
        }

        private void UpdateSelection()
        {
            bool canInteract = TacticalGameMediator.Instance.CurrentPlayer == TacticalGameMediator.Instance.SelfPlayer && !InputLock.IsLock;
            bool canInteractWithMap = canInteract && !EventSystem.current.IsPointerOverGameObject();

            (HexContent content, Vector2Int position) = GetContent(GetCursorWorldPosition());
            bool isSelectionLogicActive = _selectionLogic != null;
            bool isCursorOnPlayerSquad = content.Type == HexContent.ContentType.CreatureSquad && content.Player == TacticalGameMediator.Instance.SelfPlayer;
            bool isSquadUnderCursorSelected = _selectionLogic != null && _selectionLogic.IsSelected(content.Id);
            if (canInteract)
            {
                // Avoid double click
                // Allow selection logic to run if hovering enemies and player squads that are not selected
                if (isSelectionLogicActive && (!isCursorOnPlayerSquad || isSquadUnderCursorSelected))
                {
                    _selectionLogic.Update(canInteractWithMap);
                    if (_selectionLogic == null)
                    {
                        return;
                    }
                }
            }

            if (canInteractWithMap)
            {
                bool hoveringAny = false;
                if (isCursorOnPlayerSquad && !isSquadUnderCursorSelected)
                {
                    HoverSquads(content.Id, isSelectionLogicActive);
                    hoveringAny = true;
                    if (_hoveredSquads.Count > 0 && _hoveredSquads[0].HasAction && ControlHandler.Instance.WasLeftClicked && ClickCooldown.TryClick())
                    {
                        HoverSquads(0, isSelectionLogicActive);
                        SelectSquads(content.Id);
                    }
                }
                if (!hoveringAny && _hoveredSquads.Count > 0)
                {
                    HoverSquads(0, isSelectionLogicActive);
                }
            }
        }

        private void UpdateCamera()
        {
            bool zoomChanged = false;
            float previousSize = _camera.orthographicSize;
            float scrollDelta = Input.mouseScrollDelta.y;
            // Handle zoom
            if (scrollDelta != 0)
            {
                _isSnapping = false;
                _zoomIdleTimer = 0f;

                float currentZoomSpeed = _zoomSpeed;
                float currentSize = _camera.orthographicSize;

                // Check if near any snap point
                bool nearSnapPoint = false;
                foreach (float multiplier in _zoomLevelMultipliers)
                {
                    float targetSize = _previousOrthographicSize * multiplier;
                    if (targetSize > 0.01f) // Avoid division by zero or tiny values
                    {
                        float diffRatio = Mathf.Abs(currentSize - targetSize) / targetSize;
                        if (diffRatio < _snapThreshold)
                        {
                            nearSnapPoint = true;
                            break;
                        }
                    }
                }

                if (nearSnapPoint)
                {
                    currentZoomSpeed *= _slowZoomMultiplier;
                }

                float newSize = _camera.orthographicSize - scrollDelta * currentZoomSpeed;
                _camera.orthographicSize = Mathf.Clamp(newSize, _minOrthographicSize, _maxOrthographicSize);
            }
            else
            {
                if (_isSnapping)
                {
                    _camera.orthographicSize = Mathf.MoveTowards(_camera.orthographicSize, _snapTargetSize, _snapSpeed * Time.deltaTime);
                    if (Mathf.Approximately(_camera.orthographicSize, _snapTargetSize))
                    {
                        _camera.orthographicSize = _snapTargetSize;
                        _isSnapping = false;
                    }
                }
                else
                {
                    _zoomIdleTimer += Time.deltaTime;
                    if (_zoomIdleTimer >= _zoomSnapDelay)
                    {
                        float currentSize = _camera.orthographicSize;
                        float closestSnapSize = currentSize;
                        float minDiff = float.MaxValue;
                        foreach (float multiplier in _zoomLevelMultipliers)
                        {
                            float targetSize = _previousOrthographicSize * multiplier;
                            float diff = Mathf.Abs(currentSize - targetSize);
                            if (diff < minDiff)
                            {
                                minDiff = diff;
                                closestSnapSize = targetSize;
                            }
                        }

                        if (closestSnapSize > 0.01f && !Mathf.Approximately(currentSize, closestSnapSize))
                        {
                            float diffRatio = Mathf.Abs(currentSize - closestSnapSize) / closestSnapSize;
                            if (diffRatio >= _snapThreshold)
                            {
                                _snapTargetSize = closestSnapSize;
                                _isSnapping = true;
                                _zoomIdleTimer = 0f;
                            }
                        }
                        else
                        {
                             _zoomIdleTimer = 0f;
                        }
                    }
                }
            }
            zoomChanged = !Mathf.Approximately(previousSize, _camera.orthographicSize);

            // Handle camera movement
            Vector2 delta = Vector2.zero;
            float screenBorderThreshold = 3f;
            Vector3 mousePosition = Input.mousePosition;

            if (Input.GetKey(KeyCode.D) || Input.GetKey(KeyCode.RightArrow) || mousePosition.x >= Screen.width - screenBorderThreshold)
            {
                delta.x = 1;
            }
            else if (Input.GetKey(KeyCode.A) || Input.GetKey(KeyCode.LeftArrow) || mousePosition.x <= screenBorderThreshold)
            {
                delta.x = -1;
            }

            if (Input.GetKey(KeyCode.W) || Input.GetKey(KeyCode.UpArrow) || mousePosition.y >= Screen.height - screenBorderThreshold)
            {
                delta.y = 1;
            }
            else if (Input.GetKey(KeyCode.S) || Input.GetKey(KeyCode.DownArrow) || mousePosition.y <= screenBorderThreshold)
            {
                delta.y = -1;
            }

            bool needsClamping = delta.sqrMagnitude > 0 || zoomChanged;
            if (needsClamping)
            {
                Vector3 targetPosition = _camera.transform.position;
                 if (delta.sqrMagnitude > 0)
                {
                     targetPosition += (Vector3)delta.normalized * Time.deltaTime * Preferences.MapScrollingSpeed;
                }

                Rect mapBounds = HexHelper.GetMapWorldBounds(_map.Model.Width, _map.Model.Height);
                targetPosition.x = Mathf.Clamp(targetPosition.x, mapBounds.xMin, mapBounds.xMax);
                targetPosition.y = Mathf.Clamp(targetPosition.y, mapBounds.yMin, mapBounds.yMax);
                _camera.transform.position = targetPosition;
            }
        }

        private Vector3 GetCursorWorldPosition()
        {
            Vector3 cursorWorldPosition = _camera.ScreenToWorldPoint(Input.mousePosition);
            cursorWorldPosition.z = 0;
            return cursorWorldPosition;
        }

        private (HexContent, Vector2Int) GetContent(Vector3 cursorWorldPosition)
        {
            // TODO: Improve baner hover / selection logic in future
            foreach (var baner in _baners)
            {
                if(!baner.Value.IsHovered())
                {
                    continue;
                }
                HexCreatureSquadModel idSquad = _map.Model.GetEntity(baner.Key) as HexCreatureSquadModel;
                HexContent content = new HexContent(HexContent.ContentType.CreatureSquad, baner.Key, idSquad.Player);
                return (content, idSquad.Position);
            }

            Vector2Int position = HexHelper.GetHexPosition(cursorWorldPosition);
            return (TacticalGameMediator.Instance.GetContent(position), position);
        }

        private void SetSelectionState(IEnumerable<ICreatureSquadModelForView> squads, SquadSelectionState state)
        {
            foreach (ICreatureSquadModelForView squad in squads)
            {
                if (_entities.TryGetValue(squad.Id, out HexEntityView view) && view is HexCreatureSquadView creatureSquadView)
                {
                    creatureSquadView.SetSelectionState(state);
                }
                if (_baners.TryGetValue(squad.Id, out HexBanerView baner))
                {
                    baner.SetSelectionState(state);
                }
            }
        }

        public void TeleportCameraTo(Vector3 worldPosition)
        {
            _camera.transform.position = new Vector3(worldPosition.x, worldPosition.y, _camera.transform.position.z);
        }

        public void MoveCameraTo(Vector3 worldPosition, Action callback = null)
        {
            InputLock.Lock(LOCK);
            _locker.Lock(LOCK);
            Vector3 targetPosition = new Vector3(worldPosition.x, worldPosition.y, _camera.transform.position.z);
            float time = (targetPosition - _camera.transform.position).magnitude * Time.deltaTime * Preferences.MapScrollingSpeed;
            _camera.transform.DOMove(targetPosition, time).OnComplete(() =>
            {
                _locker.Unlock(LOCK);
                InputLock.Unlock(LOCK);
                callback?.Invoke();
            });
        }

        public void HoverSquads(ulong id, bool ignoreSelectionLogic = false)
        {
            if (_selectionLogic != null && !ignoreSelectionLogic)
            {
                return;
            }

            if (_hoveredSquads.Count > 0)
            {
                SetSelectionState(_hoveredSquads, SquadSelectionState.None);
                _hoveredSquads.Clear();
            }

            if (id != 0)
            {
                TacticalGameMediator.Instance.GetAllSquads(id, _hoveredSquads);
                if (_hoveredSquads[0].HasAction)
                {
                    SetSelectionState(_hoveredSquads, SquadSelectionState.Hovered);
                    OnCreatureSquadHovered?.Invoke(_hoveredSquads[0].Id);
                    return;
                }
            }
            OnCreatureSquadHovered?.Invoke(0);
        }

        public void SelectSquads(ulong id)
        {
            if (_baners.TryGetValue(id, out HexBanerView baner))
            {
                Vector3 targetPosition = baner.transform.position;
                if (!CameraHelper.IsVisible(targetPosition, _camera))
                {
                    MoveCameraTo(targetPosition);
                }
            }

            if (_selectionLogic != null)
            {
                if (_selectionLogic.IsSelected(id))
                {
                    return;
                }
                _selectionLogic.Disable();
            }
            _selectionLogic = new SelectAndAutoPlaceSquadsLogic(this, id);
            OnCreatureSquadSelected?.Invoke(id);
        }

        private abstract class SelectionLogic
        {
            private const int CANCEL_RANGE = 4;

            protected TacticalMapView _view;
            protected PlacementData _placementData;

            public SelectionLogic(TacticalMapView view, PlacementData placementData)
            {
                _view = view;
                _placementData = placementData;
            }

            public abstract void Update(bool interactable);

            public bool IsSelected(ulong id)
            {
                foreach (ICreatureSquadModelForView squad in _placementData.squads)
                {
                    if (squad.Id == id)
                    {
                        return true;
                    }
                }
                return false;
            }

            public void Switch(SelectionLogic logic)
            {
                OnDisable();
                _view._selectionLogic = logic;
            }

            public void Disable()
            {
                OnDisable();
                _view._selectionLogic = null;
                _view.SetSelectionState(_placementData.squads, SquadSelectionState.None);
                _view.OnCreatureSquadSelected?.Invoke(0);
            }

            protected virtual void FillReachableArea()
            {
                _placementData.oddReachableArea.Clear();
                _placementData.oddAttackableArea.Clear();
                List<Vector2Int> oddPositions = new();
                for (int i = 0; i < _placementData.squads.Count; i++)
                {
                    oddPositions.Clear();
                    TacticalGameMediator.Instance.GetReachableBySquadArea(_placementData.squads[i].Id, oddPositions, _placementData.oddSquadsPositions);
                    oddPositions.Add(_placementData.oddSquadsPositions[i]);
                }
                foreach (Vector2Int oddPosition in oddPositions)
                {
                    _placementData.oddReachableArea.Add(oddPosition);
                    Vector2Int axialPosition = HexHelper.OddToAxial(oddPosition);
                    for (int i = 0; i < 6; i++)
                    {
                        _placementData.oddAttackableArea.Add(HexHelper.AxialToOdd(axialPosition + HexHelper.AXIAL_DIRECTIONS[i]));
                    }
                }
            }

            protected void DrawReachableArea()
            {
                if (_placementData.oddReachableArea.Count > 0)
                {
                    Sprite sprite = HexDatabase.Instance.Mask.GetSelectionSprite();
                    foreach (Vector2Int position in _placementData.oddReachableArea)
                    {
                        ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(position), sprite, SortingOrder.PATH, _view.TerrainView.GetTerrain(position).SelectionTint);
                    }
                }
            }

            protected void PlaceSquadsNearbyToTarget(Vector2Int targetPosition)
            {
                int squadsToPlace = _placementData.squads.Count - _placementData.oddTargets.Count;
                Vector2Int axialTargetPosition = HexHelper.OddToAxial(targetPosition);
                for (int i = 0; i < squadsToPlace; i++)
                {
                    Vector2Int axialClosestPosition = HexHelper.INVALID_POSITION;
                    int closestHeuristic = int.MaxValue;
                    foreach (Vector2Int oddPosition in _placementData.oddReachableArea)
                    {
                        Vector2Int axialPosition = HexHelper.OddToAxial(oddPosition);
                        int heuristic = HexHelper.AxialDistance(axialTargetPosition, axialPosition);
                        if (heuristic < closestHeuristic)
                        {
                            closestHeuristic = heuristic;
                            axialClosestPosition = axialPosition;
                        }
                    }
                    if (closestHeuristic != int.MaxValue)
                    {
                        _placementData.oddTargets.Add(GetClosestSquad(axialClosestPosition), HexHelper.AxialToOdd(axialClosestPosition));
                    }
                    FillReachableArea();
                }
            }

            protected ulong GetClosestSquad(Vector2Int axialPosition)
            {
                int closestDistance = int.MaxValue;
                ulong closestId = 0;
                foreach (ICreatureSquadModelForView squad in _placementData.squads)
                {
                    if (_placementData.oddTargets.ContainsKey(squad.Id))
                    {
                        continue;
                    }
                    int distance = HexHelper.AxialDistance(HexHelper.OddToAxial(squad.Position), axialPosition);
                    if (distance < closestDistance)
                    {
                        closestDistance = distance;
                        closestId = squad.Id;
                    }
                }
                return closestId;
            }

            protected bool IsInCancelRange(Vector2Int position)
            {
                Vector2Int axialPosition = HexHelper.OddToAxial(position);
                bool isOutOfRange = true;
                foreach (KeyValuePair<ulong, Vector2Int> target in _placementData.oddTargets)
                {
                    if (HexHelper.AxialDistance(HexHelper.OddToAxial(target.Value), axialPosition) <= CANCEL_RANGE)
                    {
                        isOutOfRange = false;
                    }
                }
                return isOutOfRange;
            }

            protected virtual void OnDisable()
            {
            }
        }

        private class PlacementData
        {
            public ulong mainId;
            public HexCreature creature;
            public HashSet<Vector2Int> oddReachableArea = new();
            public HashSet<Vector2Int> oddAttackableArea = new();
            public List<ICreatureSquadModelForView> squads = new();
            public List<Vector2Int> oddSquadsPositions = new();
            public Dictionary<ulong, Vector2Int> oddTargets = new();
        }

        private class SelectAndAutoPlaceSquadsLogic : SelectionLogic
        {
            private List<ICreatureSquadModelForView> _attackableEnemySquads = new();
            private List<ICreatureSquadModelForView> _hoveredEnemySquads = new();

            private Vector2Int[] _oddTargets;
            private Vector2Int[] _oddResults;

            private Vector2Int[] _axialOffsets;
            private (HexContent, Vector2Int) _content;

            public SelectAndAutoPlaceSquadsLogic(TacticalMapView view, ulong id) : base(view, new PlacementData() { mainId = id })
            {
                TacticalGameMediator.Instance.GetAllSquads(id, _placementData.squads);
                _placementData.oddSquadsPositions = _placementData.squads.Select((s) => s.Position).ToList();
                _axialOffsets = _placementData.oddSquadsPositions.Select((p) => HexHelper.OddToAxial(p) - HexHelper.OddToAxial(_placementData.oddSquadsPositions[0])).ToArray();
                _placementData.creature = HexDatabase.Instance.GetCreature(_placementData.squads[0].Uid);
                _view.SetSelectionState(_placementData.squads, SquadSelectionState.Selected);
                FillReachableArea();
                FillAndShowAttackableSquads();

                int count = _placementData.oddSquadsPositions.Count;
                _oddTargets = new Vector2Int[count];
                _oddResults = new Vector2Int[count];
            }

            public override void Update(bool interactable)
            {
                if (interactable)
                {
                    Vector3 cursorWorldPosition = _view.GetCursorWorldPosition();
                    if (!ControlHandler.Instance.IsLeftPressed && !ControlHandler.Instance.WasLeftClicked)
                    {
                        _content = _view.GetContent(cursorWorldPosition);
                    }

                    (HexContent content, Vector2Int position) = _content;
                    bool isAttack = content.Type == HexContent.ContentType.CreatureSquad && content.Player != TacticalGameMediator.Instance.SelfPlayer;
                    HoverSquads(isAttack ? content.Id : 0);

                    if (isAttack)
                    {
                        if (_placementData.creature is HexRangedCreature rangedCreature)
                        {
                            bool canAttackMelee = false;
                            bool canAttackRanged = false;
                            bool canAttackAOE = false;
                            int aoeRadius = 0;

                            ClearPositions();
                            if (TacticalGameMediator.Instance.AreSquadsNearby(_placementData.mainId, content.Id))
                            {
                                canAttackMelee = true;
                            }
                            else
                            {
                                int distance = HexHelper.AxialDistance(HexHelper.OddToAxial(_placementData.oddSquadsPositions[0]), HexHelper.OddToAxial(position));
                                if (distance <= rangedCreature.Range)
                                {
                                    canAttackRanged = true;
                                }
                                else if (distance <= rangedCreature.MaxRange)
                                {
                                    canAttackAOE = true;
                                }

                                // TODO refactor
                                if (canAttackAOE)
                                {
                                    Sprite sprite = HexDatabase.Instance.Mask.GetSelectionSprite();
                                    Vector3Int cubePosition = HexHelper.OddToCube(position);
                                    aoeRadius = distance > (rangedCreature.Range + rangedCreature.MaxRange) / 2 ? 2 : 1;
                                    for (int i = -aoeRadius; i <= aoeRadius; i++)
                                    {
                                        for (int j = -aoeRadius; j <= aoeRadius; j++)
                                        {
                                            for (int k = -aoeRadius; k <= aoeRadius; k++)
                                            {
                                                if (i + j + k == 0)
                                                {
                                                    ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(HexHelper.CubeToOdd(cubePosition + new Vector3Int(i, j, k))), sprite, SortingOrder.PATH, Color.green);
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            if (ControlHandler.Instance.WasLeftClicked && ClickCooldown.TryClick())
                            {
                                if (canAttackMelee)
                                {
                                    Disable();
                                    TacticalGameMediator.Instance.AttackSquadsMelee(_placementData.mainId, content.Id);
                                }
                                else
                                {
                                    if (canAttackAOE)
                                    {
                                        Disable();
                                        TacticalGameMediator.Instance.AttackSquadsRangedAOE(_placementData.mainId, position, aoeRadius);
                                    }
                                    else if (canAttackRanged)
                                    {
                                        Disable();
                                        TacticalGameMediator.Instance.AttackSquadsRanged(_placementData.mainId, content.Id);
                                    }
                                }
                            }
                            ImmediateSpriteRenderer.DrawSprite(cursorWorldPosition, canAttackMelee ? _view._greenAttack : (canAttackRanged ? _view._greenShoot : (canAttackAOE ? _view._greenHalfShoot : _view._redShoot)), SortingOrder.CURSOR);
                        }
                        else
                        {
                            bool isReachable = FindNearbyAttackPositions(position, content.Id);
                            ImmediateSpriteRenderer.DrawSprite(cursorWorldPosition, isReachable ? _view._greenAttack : _view._redAttack, SortingOrder.CURSOR);
                            if (isReachable && ControlHandler.Instance.WasLeftClicked && ClickCooldown.TryClick())
                            {
                                for (int i = 0; i < _oddResults.Length; i++)
                                {
                                    _placementData.oddTargets.Add(GetClosestSquad(HexHelper.OddToAxial(_oddResults[i])), _oddResults[i]);
                                }
                                Disable();
                                TacticalGameMediator.Instance.MoveAndAttackSquadsMelee(_placementData.oddTargets, content.Id);
                            }
                        }
                    }
                    else
                    {
                        bool isReachable = FindNearbyMovementPositions(position);
                        if (isReachable && ControlHandler.Instance.WasLeftClicked && ClickCooldown.TryClick())
                        {
                            for (int i = 0; i < _oddResults.Length; i++)
                            {
                                _placementData.oddTargets.Add(GetClosestSquad(HexHelper.OddToAxial(_oddResults[i])), _oddResults[i]);
                            }
                            Disable();
                            if (IsPositionAndTargetDifferent())
                            {
                                TacticalGameMediator.Instance.MoveSquads(_placementData.oddTargets);
                            }
                        }
                    }

                    // TODO attack swords
                    for (int i = 0; i < _oddResults.Length; i++)
                    {
                        if (_oddResults[i] != HexHelper.INVALID_POSITION)
                        {
                            if (isAttack)
                            {
                                Sprite icon = null;
                                Vector2Int axialResult = HexHelper.OddToAxial(_oddResults[i]);
                                for (int j = 0; j < 6; j++)
                                {
                                    if (CanAttack(HexHelper.AxialToOdd(axialResult + HexHelper.AXIAL_DIRECTIONS[j])))
                                    {
                                        icon = _view._attackDirections[j];
                                        break;
                                    }
                                }
                                if (icon == null)
                                {
                                    icon = _view._greenDot;
                                }
                                ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(_oddResults[i]), icon, SortingOrder.PATH);
                            }
                            else
                            {
                                ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(_oddResults[i]), _view._greenDot, SortingOrder.PATH);
                            }
                        }
                    }

                    if (ControlHandler.Instance.IsStartingDragging && ClickCooldown.TryClick())
                    {
                        if (isAttack)
                        {
                            if (_placementData.oddAttackableArea.Contains(position))
                            {
                                Switch(new PlaceSquadsManuallyAttackMeleeLogic(_view, _placementData, content.Id));
                            }
                        }
                        else
                        {
                            if (_placementData.oddReachableArea.Contains(position))
                            {
                                Switch(new PlaceSquadsManuallyMovementLogic(_view, _placementData, position));
                            }
                        }
                    }

                    if (ControlHandler.Instance.WasRightClicked && ClickCooldown.TryClick())
                    {
                        Disable();
                    }
                }
                DrawReachableArea();
            }

            protected override void OnDisable()
            {
                ClearAttackableSquads();
            }

            private void ClearPositions()
            {
                for (int i = 0; i < _oddResults.Length; i++)
                {
                    _oddResults[i] = HexHelper.INVALID_POSITION;
                }
            }

            private bool FindNearbyAttackPositions(Vector2Int targetPosition, ulong enemyId)
            {
                // Prepare
                ClearPositions();
                List<ICreatureSquadModelForView> enemySquads = new();
                TacticalGameMediator.Instance.GetAllSquads(enemyId, enemySquads);

                // Check if reachable
                HashSet<Vector2Int> axialNearby = new();
                bool canAttack = false;
                foreach (ICreatureSquadModelForView squad in enemySquads)
                {
                    if (_placementData.oddAttackableArea.Contains(squad.Position))
                    {
                        canAttack = true;
                        Vector2Int axialPosition = HexHelper.OddToAxial(squad.Position);
                        for (int i = 0; i < 6; i++)
                        {
                            Vector2Int axialNext = axialPosition + HexHelper.AXIAL_DIRECTIONS[i];
                            Vector2Int oddNext = HexHelper.AxialToOdd(axialNext);
                            if (_placementData.oddReachableArea.Contains(oddNext))
                            {
                                axialNearby.Add(axialNext);
                            }
                        }
                    }
                }
                if (!canAttack)
                {
                    return false;
                }
                Vector2Int axialStartingPosition = HexHelper.OddToAxial(_placementData.oddSquadsPositions[0]);
                Vector2Int axialTargetPosition = HexHelper.OddToAxial(targetPosition);
                HashSet<Vector2Int> axialVisitedSet = new();

                // Find first position
                Vector2Int axialClosestPosition = HexHelper.INVALID_POSITION;
                int closestHeuristic = int.MaxValue;
                foreach (Vector2Int axialPosition in axialNearby)
                {
                    int heuristic = HexHelper.AxialDistance(axialStartingPosition, axialPosition) + HexHelper.AxialDistance(axialTargetPosition, axialPosition);
                    if (heuristic < closestHeuristic)
                    {
                        closestHeuristic = heuristic;
                        axialClosestPosition = axialPosition;
                    }
                }
                if (closestHeuristic != int.MaxValue)
                {
                    _oddResults[0] = HexHelper.AxialToOdd(axialClosestPosition);
                    axialVisitedSet.Add(axialClosestPosition);
                }
                else
                {
                    return false;
                }

                for (int i = 1; i < _oddResults.Length; i++)
                {
                    // Try to place nearby to enemy
                    bool positionFound = false;
                    foreach (Vector2Int axialVisitedPosition in axialVisitedSet)
                    {
                        for (int j = 0; j < 6; j++)
                        {
                            Vector2Int axialNext = axialVisitedPosition + HexHelper.AXIAL_DIRECTIONS[j];
                            if (axialVisitedSet.Contains(axialNext))
                            {
                                continue;
                            }
                            if (axialNearby.Contains(axialNext))
                            {
                                _oddResults[i] = HexHelper.AxialToOdd(axialNext);
                                positionFound = true;
                                axialVisitedSet.Add(axialNext);
                                break;
                            }
                        }
                        if (positionFound)
                        {
                            break;
                        }
                    }
                    // Try to place in other nearby positions
                    if (!positionFound)
                    {
                        axialClosestPosition = HexHelper.INVALID_POSITION;
                        closestHeuristic = int.MaxValue;
                        foreach (Vector2Int axialVisitedPosition in axialVisitedSet)
                        {
                            for (int j = 0; j < 6; j++)
                            {
                                Vector2Int axialNext = axialVisitedPosition + HexHelper.AXIAL_DIRECTIONS[j];
                                if (axialVisitedSet.Contains(axialNext))
                                {
                                    continue;
                                }
                                Vector2Int oddNext = HexHelper.AxialToOdd(axialNext);
                                if (!_placementData.oddReachableArea.Contains(oddNext))
                                {
                                    continue;
                                }
                                int heuristic = HexHelper.AxialDistance(axialTargetPosition, axialNext);
                                if (heuristic < closestHeuristic)
                                {
                                    axialClosestPosition = axialNext;
                                    closestHeuristic = heuristic;
                                }
                            }
                        }
                        if (closestHeuristic != int.MaxValue)
                        {
                            _oddResults[i] = HexHelper.AxialToOdd(axialClosestPosition);
                            axialVisitedSet.Add(axialClosestPosition);
                        }
                    }
                }
                return _oddResults[_oddResults.Length - 1] != HexHelper.INVALID_POSITION;
            }

            private bool FindNearbyMovementPositions(Vector2Int oddTargetPosition)
            {
                // Prepare
                Vector2Int axialTargetPosition = HexHelper.OddToAxial(oddTargetPosition);
                for (int i = 0; i < _oddResults.Length; i++)
                {
                    _oddTargets[i] = HexHelper.AxialToOdd(axialTargetPosition + _axialOffsets[i]);
                }

                // Try to place without changing formation
                bool validFormation = true;
                for (int i = 0; i < _oddTargets.Length; i++)
                {
                    _oddResults[i] = _oddTargets[i];
                    if (!_placementData.oddReachableArea.Contains(_oddTargets[i]))
                    {
                        validFormation = false;
                    }
                }
                if (!validFormation)
                {
                    HashSet<Vector2Int> axialVisitedSet = new();
                    ClearPositions();
                    // Find first position
                    Vector2Int axialClosestPosition = HexHelper.INVALID_POSITION;
                    int closestHeuristic = int.MaxValue;
                    foreach (Vector2Int oddPosition in _placementData.oddReachableArea)
                    {
                        Vector2Int axialPosition = HexHelper.OddToAxial(oddPosition);
                        int heuristic = HexHelper.AxialDistance(axialTargetPosition, axialPosition);
                        if (heuristic < closestHeuristic)
                        {
                            closestHeuristic = heuristic;
                            axialClosestPosition = axialPosition;
                        }
                    }
                    if (closestHeuristic != int.MaxValue)
                    {
                        _oddResults[0] = HexHelper.AxialToOdd(axialClosestPosition);
                        axialVisitedSet.Add(axialClosestPosition);
                    }
                    else
                    {
                        return false;
                    }

                    // Find other positions
                    for (int i = 1; i < _oddTargets.Length; i++)
                    {
                        axialClosestPosition = HexHelper.INVALID_POSITION;
                        closestHeuristic = int.MaxValue;
                        foreach (Vector2Int axialPosition in axialVisitedSet)
                        {
                            for (int j = 0; j < 6; j++)
                            {
                                Vector2Int axialNext = axialPosition + HexHelper.AXIAL_DIRECTIONS[j];
                                Vector2Int oddNext = HexHelper.AxialToOdd(axialNext);

                                if (axialVisitedSet.Contains(axialNext))
                                {
                                    continue;
                                }
                                if (!_placementData.oddReachableArea.Contains(oddNext))
                                {
                                    continue;
                                }
                                int heuristic = HexHelper.AxialDistance(axialTargetPosition, axialNext);
                                if (heuristic < closestHeuristic)
                                {
                                    closestHeuristic = heuristic;
                                    axialClosestPosition = axialNext;
                                }
                            }
                        }
                        if (closestHeuristic != int.MaxValue)
                        {
                            _oddResults[i] = HexHelper.AxialToOdd(axialClosestPosition);
                            axialVisitedSet.Add(axialClosestPosition);
                        }
                    }
                }

                return _oddResults[_oddResults.Length - 1] != HexHelper.INVALID_POSITION;
            }

            private void FillAndShowAttackableSquads()
            {
                if (_attackableEnemySquads.Count > 0)
                {
                    ClearAttackableSquads();
                }
                // TODO refactor
                IEnumerable<HexCreatureSquadModel> enemySquads = _view.Map.Model.GetEnumerable<HexCreatureSquadModel>().Where((s) => s.Player != TacticalGameMediator.Instance.SelfPlayer);
                if (_placementData.creature is HexRangedCreature rangedCreature)
                {
                    foreach (ulong id in enemySquads.Where((s) => HexHelper.AxialDistance(HexHelper.OddToAxial(_placementData.oddSquadsPositions[0]), HexHelper.OddToAxial(s.Position)) <= rangedCreature.MaxRange).Select((s) => s.ParentSquadId == 0 ? s.Id : s.ParentSquadId))
                    {
                        TacticalGameMediator.Instance.GetAllSquads(id, _attackableEnemySquads);
                    }
                }
                foreach (ulong id in enemySquads.Where((s) => _placementData.oddAttackableArea.Contains(s.Position)).Select((s) => s.ParentSquadId == 0 ? s.Id : s.ParentSquadId))
                {
                    TacticalGameMediator.Instance.GetAllSquads(id, _attackableEnemySquads);
                }
                ShowAttackableSquads();
            }

            private void ClearAttackableSquads()
            {
                _view.SetSelectionState(_attackableEnemySquads, SquadSelectionState.None);
                _attackableEnemySquads.Clear();
            }

            private void ShowAttackableSquads()
            {
                _view.SetSelectionState(_attackableEnemySquads, SquadSelectionState.AttackableEnemy);
            }

            private void HoverSquads(ulong id)
            {
                if (_hoveredEnemySquads.Count > 0)
                {
                    _view.SetSelectionState(_hoveredEnemySquads, SquadSelectionState.AttackableEnemy);
                    _hoveredEnemySquads.Clear();
                }

                if (id != 0 && _attackableEnemySquads.Any((s) => s.Id == id))
                {
                    TacticalGameMediator.Instance.GetAllSquads(id, _hoveredEnemySquads);
                    _view.SetSelectionState(_hoveredEnemySquads, SquadSelectionState.HoveredEnemy);
                }
            }

            private bool CanAttack(Vector2Int oddPosition)
            {
                foreach (ICreatureSquadModelForView squad in _hoveredEnemySquads)
                {
                    if (oddPosition == squad.Position)
                    {
                        return true;
                    }
                }
                return false;
            }

            private bool IsPositionAndTargetDifferent()
            {
                foreach (HexCreatureSquadModel squadModel in _placementData.squads)
                {
                    if (false == _placementData.oddTargets.Values.Contains(squadModel.Position))
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        private class PlaceSquadsManuallyAttackMeleeLogic : SelectionLogic
        {
            private List<ICreatureSquadModelForView> _targetSquads = new();
            private ulong _targetId;

            public PlaceSquadsManuallyAttackMeleeLogic(TacticalMapView view, PlacementData placementData, ulong targetId) : base(view, placementData)
            {
                TacticalGameMediator.Instance.GetAllSquads(targetId, _targetSquads);
                _targetId = targetId;
                FillReachableArea();
            }

            public override void Update(bool interactable)
            {
                if (interactable)
                {
                    Vector3 cursorWorldPosition = _view.GetCursorWorldPosition();
                    (HexContent content, Vector2Int position) = _view.GetContent(cursorWorldPosition);

                    if (_placementData.oddReachableArea.Count > 0)
                    {
                        if (_placementData.oddReachableArea.Contains(position))
                        {
                            _placementData.oddTargets.Add(GetClosestSquad(HexHelper.OddToAxial(position)), position);
                            FillReachableArea();
                        }
                    }

                    foreach (KeyValuePair<ulong, Vector2Int> pair in _placementData.oddTargets)
                    {
                        Sprite icon = null;
                        Vector2Int axialResult = HexHelper.OddToAxial(pair.Value);
                        for (int j = 0; j < 6; j++)
                        {
                            if (CanAttack(HexHelper.AxialToOdd(axialResult + HexHelper.AXIAL_DIRECTIONS[j])))
                            {
                                icon = _view._attackDirections[j];
                                break;
                            }
                        }
                        if (icon == null)
                        {
                            icon = _view._greenDot;
                        }
                        ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(pair.Value), icon, SortingOrder.PATH);
                    }

                    if (ControlHandler.Instance.IsEndingDragging)
                    {
                        Disable();
                        if (!IsInCancelRange(position))
                        {
                            if (_placementData.oddTargets.Count < _placementData.squads.Count)
                            {
                                PlaceSquadsNearbyToTarget(_targetSquads[0].Position);
                            }
                            TacticalGameMediator.Instance.MoveAndAttackSquadsMelee(_placementData.oddTargets, _targetId);
                        }
                    }

                    if (ControlHandler.Instance.WasRightClicked && ClickCooldown.TryClick())
                    {
                        Disable();
                    }
                }
                DrawReachableArea();
            }

            protected override void FillReachableArea()
            {
                if (_placementData.oddTargets.Values.Count == _placementData.squads.Count)
                {
                    _placementData.oddReachableArea.Clear();
                    return;
                }

                base.FillReachableArea();
                List<Vector2Int> oddPositions = new();
                IEnumerable<Vector2Int> oddTargets = _placementData.oddTargets.Values;
                if (_placementData.oddTargets.Count == 0)
                {
                    oddTargets = oddTargets.Union(_targetSquads.Select((s) => s.Position));
                }
                foreach (Vector2Int targetPosition in oddTargets)
                {
                    Vector2Int axialTargetPosition = HexHelper.OddToAxial(targetPosition);
                    for (int i = 0; i < 6; i++)
                    {
                        Vector2Int oddNext = HexHelper.AxialToOdd(axialTargetPosition + HexHelper.AXIAL_DIRECTIONS[i]);
                        if (_placementData.oddReachableArea.Contains(oddNext) && !_placementData.oddTargets.Values.Contains(oddNext))
                        {
                            oddPositions.Add(oddNext);
                        }
                    }
                }
                _placementData.oddReachableArea.Clear();
                _placementData.oddReachableArea.UnionWith(oddPositions);
            }

            private bool CanAttack(Vector2Int oddPosition)
            {
                foreach (ICreatureSquadModelForView squad in _targetSquads)
                {
                    if (oddPosition == squad.Position)
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        private class PlaceSquadsManuallyMovementLogic : SelectionLogic
        {
            public PlaceSquadsManuallyMovementLogic(TacticalMapView view, PlacementData placementData, Vector2Int target) : base(view, placementData)
            {
                _placementData.oddTargets.Add(GetClosestSquad(HexHelper.OddToAxial(target)), target);
                FillReachableArea();
            }

            public override void Update(bool interactable)
            {
                if (interactable)
                {
                    Vector3 cursorWorldPosition = _view.GetCursorWorldPosition();
                    (HexContent content, Vector2Int position) = _view.GetContent(cursorWorldPosition);

                    if (_placementData.oddReachableArea.Count > 0)
                    {
                        if (_placementData.oddReachableArea.Contains(position))
                        {
                            _placementData.oddTargets.Add(GetClosestSquad(HexHelper.OddToAxial(position)), position);
                            FillReachableArea();
                        }
                    }

                    foreach (KeyValuePair<ulong, Vector2Int> pair in _placementData.oddTargets)
                    {
                        ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(pair.Value), _view._greenDot, SortingOrder.PATH);
                    }

                    if (ControlHandler.Instance.IsEndingDragging)
                    {
                        Disable();
                        if (!IsInCancelRange(position))
                        {
                            if (_placementData.oddTargets.Count < _placementData.squads.Count)
                            {
                                PlaceSquadsNearbyToTarget(position);
                            }
                            TacticalGameMediator.Instance.MoveSquads(_placementData.oddTargets);
                        }
                    }

                    if (ControlHandler.Instance.WasRightClicked && ClickCooldown.TryClick())
                    {
                        Disable();
                    }
                }
                DrawReachableArea();
            }

            protected override void FillReachableArea()
            {
                if (_placementData.oddTargets.Values.Count == _placementData.squads.Count)
                {
                    _placementData.oddReachableArea.Clear();
                    return;
                }

                base.FillReachableArea();
                List<Vector2Int> oddPositions = new();
                foreach (Vector2Int targetPosition in _placementData.oddTargets.Values)
                {
                    Vector2Int axialTargetPosition = HexHelper.OddToAxial(targetPosition);
                    for (int i = 0; i < 6; i++)
                    {
                        Vector2Int oddNext = HexHelper.AxialToOdd(axialTargetPosition + HexHelper.AXIAL_DIRECTIONS[i]);
                        if (_placementData.oddReachableArea.Contains(oddNext) && !_placementData.oddTargets.Values.Contains(oddNext))
                        {
                            oddPositions.Add(oddNext);
                        }
                    }
                }
                _placementData.oddReachableArea.Clear();
                _placementData.oddReachableArea.UnionWith(oddPositions);
            }
        }
    }
}
