﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.Localization;

namespace PK
{
    public interface IDialogChoicesForView : IDialogMessageForView
    {
        IReadOnlyList<ChoiceLineData> Choices { get; }
        void SelectChoice(int index);
    }

    [System.Serializable]
    public class ChoiceLineData
    {
        [SerializeField] private LocalizedString _line;
        [SerializeField] private string _conditionPortGuid;
        [SerializeField] private string _outPortGuid;
        [SerializeField] private bool _chooseOnce;
        [SerializeField] private bool _wasChoosed;

        public LocalizedString Line { get { return _line; } }
        public string ConditionPortGuid { get { return _conditionPortGuid; } set { _conditionPortGuid = value; } }
        public string OutPortGuid { get { return _outPortGuid; } set { _outPortGuid = value; } }
        public bool ChooseOnce { get { return _chooseOnce; } }
        public bool WasChoosed { get { return _wasChoosed; } set { _wasChoosed = value; } }
    }

    public class ActionDialogChoicesNode : ActionDefaultNode, IFlowNode, IDialogChoicesForView
    {
        private static string LOCK = "ActionDialogChoicesNodeLock";

        [SerializeField] private LocalizedString _line;
        [SerializeField] private HexCharacter _character;
        [SerializeField] private HexBackground _background;

        [SerializeField] private List<ChoiceLineData> _choices = new();

        IReadOnlyList<ChoiceLineData> IDialogChoicesForView.Choices => _availableChoices;

        LocalizedString IDialogMessageForView.Message => _line;
        HexCharacter IDialogMessageForView.Character => _character;
        HexBackground IDialogMessageForView.Background => _background;

        private IActionContext _context;
        private List<ChoiceLineData> _availableChoices = new();

        void IFlowNode.Execute(IActionContext context)
        {
            _context = context;
            _availableChoices.Clear();
            foreach (ChoiceLineData choice in _choices)
            {
                if (choice.ChooseOnce && choice.WasChoosed)
                {
                    continue;
                }

                if (!context.Graph.IsConnected(choice.ConditionPortGuid) || (context.Graph.GetConnectedNode(choice.ConditionPortGuid) is IConditionNode conditionNode && conditionNode.Evaluate(_context)))
                {
                    _availableChoices.Add(choice);
                }
            }
            context.Locker.Lock(LOCK);
            context.EventManager.Get<InteractionEvents>().CallOnDialog(this, context.Player);
        }

        void IDialogChoicesForView.SelectChoice(int index)
        {
            ChoiceLineData choice = _choices.First((c) => c == _availableChoices[index]);
            choice.WasChoosed = true;
            _context.Locker.Unlock(LOCK);
            _context.Graph.MoveToNextNode(choice.OutPortGuid);
        }

        void IDialogMessageForView.Skip()
        {
        }
    }
}
