﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEditor;
using UnityEditor.Experimental.GraphView;
using UnityEngine;

namespace PK
{
    [Serializable]
    public class CopyPasteGraph
    {
        private class CopyPasteGraphContainer : ScriptableObject
        {
            [SerializeField] private CopyPasteGraph _graph;

            public CopyPasteGraph Graph { get { return _graph; } set { _graph = value; } }
        }

        [SerializeReference] private List<BaseNode> _nodes = new();
        [SerializeField] private List<NodeConnection> _connections = new();

        public List<BaseNode> Nodes { get { return _nodes; } }
        public List<NodeConnection> Connections { get { return _connections; } }

        public CopyPasteGraph() { }

        public CopyPasteGraph(IEnumerable<BaseGraphEditorNode> nodes, IEnumerable<Edge> edges)
        {
            _nodes.AddRange(nodes.Select((n) => n.Node as BaseNode));
            foreach (Edge edge in edges)
            {
                NodeConnectionPort input = new NodeConnectionPort(edge.input.node.viewDataKey, edge.input.viewDataKey);
                NodeConnectionPort output = new NodeConnectionPort(edge.output.node.viewDataKey, edge.output.viewDataKey);
                if (_nodes.Any((n) => n.Guid == input.NodeGuid) && _nodes.Any((n) => n.Guid == output.NodeGuid))
                {
                    _connections.Add(new NodeConnection(input, output));
                }
            }
        }

        public void UpdateGuids()
        {
            CopyPasteGraphContainer container = ScriptableObject.CreateInstance<CopyPasteGraphContainer>();
            container.Graph = this;
            Dictionary<string, string> replacementGuids = new();
            SerializedObject serializedObject = new SerializedObject(container);
            SerializedProperty nodeProperty = serializedObject.FindProperty("_graph").FindPropertyRelative("_nodes");
            while (nodeProperty.Next(true))
            {
                if (nodeProperty.propertyType == SerializedPropertyType.String)
                {
                    string name = nodeProperty.name;
                    if (name.Contains("guid", StringComparison.OrdinalIgnoreCase))
                    {
                        string oldValue = nodeProperty.stringValue;
                        if (!string.IsNullOrEmpty(oldValue))
                        {
                            if (!replacementGuids.TryGetValue(oldValue, out string newValue))
                            {
                                newValue = GUID.Generate().ToString();
                                replacementGuids.Add(oldValue, newValue);
                            }
                            nodeProperty.stringValue = newValue;
                        }
                    }
                }
            }
            serializedObject.ApplyModifiedProperties();
            serializedObject.Dispose();
            UnityEngine.Object.DestroyImmediate(container);
        }
    }
}
