﻿using System.Linq;
using UnityEditor;
using UnityEditor.Experimental.GraphView;
using UnityEditor.Localization;
using UnityEngine.Localization;
using UnityEngine.Localization.Tables;
using UnityEngine.UIElements;

namespace PK
{
    [EditorNode(typeof(ActionDialogChoicesNode), "Dialogs/Choices")]
    public class ActionDialogChoicesEditorNode : ActionDefaultEditorNode
    {
        private VisualElement _linesContainer;
        private SerializedProperty _choicesArrayProperty;

        public ActionDialogChoicesEditorNode(SerializedProperty property) : base(property)
        {
            _choicesArrayProperty = property.FindPropertyRelative("_choices");
            title = "Dialog Choices";
            SetTitleColor(Styles.ChoicesTitleColor);
            InitializeDefaultPorts(true, false);

            SerializedProperty characterProperty = property.FindPropertyRelative("_character");
            SerializedProperty backgroundProperty = property.FindPropertyRelative("_background");
            SerializedProperty lineProperty = property.FindPropertyRelative("_line");

            Label label = new Label();
            label.style.whiteSpace = WhiteSpace.Normal;
            label.style.maxWidth = 200f;
            RefreshLabel(lineProperty, label);
            object value = lineProperty.boxedValue;

            IMGUIContainer propertiesField = new IMGUIContainer(() =>
            {
                EditorGUI.BeginChangeCheck();
                PropertyHelper.PropertyField(characterProperty);
                PropertyHelper.PropertyField(backgroundProperty);
                if (EditorGUI.EndChangeCheck())
                {
                    _property.serializedObject.ApplyModifiedProperties();
                }

                EditorGUILayout.PropertyField(lineProperty);
                if (value != lineProperty.boxedValue)
                {
                    value = lineProperty.boxedValue;
                    _property.serializedObject.ApplyModifiedProperties();
                    RefreshLabel(lineProperty, label);
                }
            });
            extensionContainer.Add(propertiesField);
            extensionContainer.Add(label);

            VisualElement separator = new VisualElement { name = "divider" };
            separator.AddToClassList("horizontal");
            separator.AddToClassList("horizontal-divider");
            extensionContainer.Add(separator);

            _linesContainer = new VisualElement();
            extensionContainer.Add(_linesContainer);

            Button button = new Button(() => AddLine(true)) { text = "Create Dialogue Line" };
            extensionContainer.Add(button);

            RefreshExtension();

            Refresh();
        }

        private void AddLine(bool create, int index = -1)
        {
            VisualElement conversationContainer = new VisualElement { name = "conversation-container" };
            conversationContainer.style.flexDirection = FlexDirection.Column;

            VisualElement row1 = new VisualElement();
            row1.style.alignItems = Align.Center;
            row1.style.flexDirection = FlexDirection.Row;
            VisualElement row2 = new VisualElement();
            row2.style.flexDirection = FlexDirection.Row;

            Port conditionPort = InstantiatePort(Orientation.Horizontal, Direction.Input, Port.Capacity.Single, "Condition", typeof(ConditionPort));
            Port outPort = InstantiatePort(Orientation.Horizontal, Direction.Output, Port.Capacity.Single, "Out", typeof(FlowPort));

            conditionPort.style.flexBasis = Length.Percent(50);
            outPort.style.flexBasis = Length.Percent(50);

            SerializedProperty choiceProperty;
            if (create)
            {
                index = _choicesArrayProperty.arraySize;
                ChoiceLineData data = new ChoiceLineData();
                data.ConditionPortGuid = conditionPort.viewDataKey;
                data.OutPortGuid = outPort.viewDataKey;
                _choicesArrayProperty.InsertArrayElementAtIndex(index);
                choiceProperty = _choicesArrayProperty.GetArrayElementAtIndex(index);
                choiceProperty.boxedValue = data;
                _choicesArrayProperty.serializedObject.ApplyModifiedProperties();
            }
            else
            {
                choiceProperty = _choicesArrayProperty.GetArrayElementAtIndex(index);
                ChoiceLineData data = (ChoiceLineData)choiceProperty.boxedValue;
                conditionPort.viewDataKey = data.ConditionPortGuid;
                outPort.viewDataKey = data.OutPortGuid;
            }

            SerializedProperty lineProperty = choiceProperty.FindPropertyRelative("_line");
            SerializedProperty chooseOnceProperty = choiceProperty.FindPropertyRelative("_chooseOnce");

            Label label = new Label();
            label.style.whiteSpace = WhiteSpace.Normal;
            label.style.maxWidth = 200f;
            RefreshLabel(lineProperty, label);
            object value = lineProperty.boxedValue;

            IMGUIContainer lineField = new IMGUIContainer(() =>
            {
                EditorGUILayout.PropertyField(lineProperty);
                if (value != lineProperty.boxedValue)
                {
                    _property.serializedObject.ApplyModifiedProperties();
                    value = lineProperty.boxedValue;
                    RefreshLabel(lineProperty, label);
                }
                EditorGUI.BeginChangeCheck();
                EditorGUILayout.PropertyField(chooseOnceProperty);
                if (EditorGUI.EndChangeCheck())
                {
                    _property.serializedObject.ApplyModifiedProperties();
                }
            });
            lineField.style.flexBasis = Length.Percent(90);

            Button removeButton = new Button(() => RemoveLine(index));
            removeButton.text = "x";
            removeButton.style.flexBasis = Length.Percent(10);

            row1.Add(lineField);
            row1.Add(removeButton);
            row2.Add(conditionPort);
            row2.Add(outPort);
            
            conversationContainer.Add(row1);
            conversationContainer.Add(label);
            conversationContainer.Add(row2);

            _linesContainer.Add(conversationContainer);

            AddPort(conditionPort);
            AddPort(outPort);

            VisualElement separator = new VisualElement { name = "divider" };
            separator.AddToClassList("horizontal");
            separator.AddToClassList("horizontal-divider");
            _linesContainer.Add(separator);

            Refresh();
        }

        private void RemoveLine(int index)
        {
            ChoiceLineData line = (ChoiceLineData)_choicesArrayProperty.GetArrayElementAtIndex(index).boxedValue;
            View.RemovePort(Ports[line.ConditionPortGuid]);
            View.RemovePort(Ports[line.OutPortGuid]);
            _choicesArrayProperty.DeleteArrayElementAtIndex(index);
            _choicesArrayProperty.serializedObject.ApplyModifiedProperties();
            View.UpdateView();
        }

        private void RefreshExtension()
        {
            _linesContainer.Clear();
            for (var i = 0; i < _choicesArrayProperty.arraySize; i++)
            {
                AddLine(false, i);
            }
        }

        private void RefreshLabel(SerializedProperty property, Label label)
        {
            LocalizedString str = (property.boxedValue as LocalizedString);
            if (str == null || str.IsEmpty)
            {
                label.text = string.Empty;
                return;
            }
            StringTableCollection collection = LocalizationEditorSettings.GetStringTableCollection(str.TableReference);
            StringTable table = collection.GetTable("en") as StringTable;
            StringTableEntry entry = table.GetEntryFromReference(str.TableEntryReference);
            if (entry == null)
            {
                label.text = "Missing";
            }
            else
            {
                label.text = entry.LocalizedValue;
            }
        }
    }
}
