﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEditor.Searcher;
using UnityEngine;
using static PK.HexVariablesModel;

namespace PK
{
    public class VariableSearchWindowProvider : ScriptableObject
    {
        private Dictionary<ulong, NodeEntry> _nodeEntries = new();

        public Variable GetVariable(ulong id)
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            if (_nodeEntries.TryGetValue(id, out NodeEntry entry))
            {
                return entry.Variable;
            }

            return null;
        }

        public Variable GetVariable(string name)
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            foreach (NodeEntry entry in _nodeEntries.Values)
            {
                if (entry.Name == name)
                {
                    return entry.Variable;
                }
            }

            return null;
        }

        public void Clear()
        {
            _nodeEntries.Clear();
        }

        public Searcher LoadSearchWindow()
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            List<SearcherItem> root = new();
            NodeEntry dummyEntry = new NodeEntry();

            foreach (NodeEntry nodeEntry in _nodeEntries.Values)
            {
                SearcherItem parent = null;
                string[] entryTitle = nodeEntry.MenuItem.Split('/');

                for (int i = 0; i < entryTitle.Length; i++)
                {
                    var pathEntry = entryTitle[i];
                    var children = parent != null ? parent.Children : root;
                    var item = children.Find(x => x.Name == pathEntry);

                    if (item == null)
                    {
                        if (i == entryTitle.Length - 1)
                        {
                            item = new SearchNodeItem(pathEntry, nodeEntry);
                        }
                        else
                        {
                            item = new SearchNodeItem(pathEntry, dummyEntry);
                        }
                        
                        if (parent != null)
                        {
                            parent.AddChild(item);
                        }
                        else
                        {
                            children.Add(item);
                        }
                    }

                    parent = item;

                    if (parent.Depth == 0 && !root.Contains(parent))
                    {
                        root.Add(parent);
                    }
                }
            }
            SearcherDatabase nodeDatabase = SearcherDatabase.Create(root, string.Empty, false);
            return new Searcher(nodeDatabase, new SearchWindowAdapter("Select Variable"));
        }

        public bool OnSelectEntry(SearcherItem selectedEntry, Action<ulong> callback)
        {
            if (selectedEntry == null)
            {
                return false;
            }
            callback?.Invoke((selectedEntry as SearchNodeItem).NodeEntry.Id);
            return true;
        }

        private void GenerateEntries()
        {
            _nodeEntries.Clear();
            HexMapEditorWindow view = HexMapEditorWindow.Instance;
            if (view == null)
            {
                return;
            }
            IEnumerable<Variable> variables = Enumerable.Empty<Variable>();
            foreach (HexPlayer player in HexDatabase.Instance.Players)
            {
                variables = variables.Union(player.Variables.VariablesDict.Select((p) => p.Value));
            }
            foreach (Variable variable in variables)
            {
                _nodeEntries.Add(variable.Id, new NodeEntry(variable.Id, variable.Name, variable, $"Player/{variable.Name}"));
            }
            foreach (KeyValuePair<ulong, Variable> pair in view.Map.Model.GlobalVariables.VariablesDict)
            {
                _nodeEntries.Add(pair.Value.Id, new NodeEntry(pair.Value.Id, pair.Value.Name, pair.Value, $"Global/{pair.Value.Name}"));
            }
        }

        public class SearchWindowAdapter : SearcherAdapter
        {
            public override bool HasDetailsPanel => false;

            public SearchWindowAdapter(string title) : base(title)
            {
            }
        }

        public class SearchNodeItem : SearcherItem
        {
            private NodeEntry _nodeEntry;

            public NodeEntry NodeEntry { get { return _nodeEntry; } }

            public SearchNodeItem(string name, NodeEntry nodeEntry) : base(name)
            {
                _nodeEntry = nodeEntry;
            }
        }

        public struct NodeEntry : IEquatable<NodeEntry>
        {
            private ulong _id;
            private string _name;
            private Variable _variable;
            private string _menuItem;

            public ulong Id { get { return _id; } }
            public string Name { get { return _name; } }
            public Variable Variable { get { return _variable; } }
            public string MenuItem { get { return _menuItem; } }

            public NodeEntry(ulong id, string name, Variable variable, string menuItem)
            {
                _id = id;
                _name = name;
                _variable = variable;
                _menuItem = menuItem;
            }

            public bool Equals(NodeEntry other)
            {
                return Equals(_menuItem, other._menuItem) && _id == other._id;
            }
        }
    }
}
