﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    public enum VariableType
    {
        Bool,
        Int,
        Unknown
    }

    [System.Serializable]
    public class HexVariablesModel : ISerializationCallbackReceiver
    {
        [System.Serializable]
        public class Variable
        {
            [SerializeField] private ulong _id;
            [SerializeField] private string _name;
            [SerializeField] private VariableType _type;
            [SerializeField] private bool _boolValue;
            [SerializeField] private int _intValue;

            public ulong Id { get { return _id; } }
            public string Name { get { return _name; } }
            public VariableType Type { get { return _type; } }
            public bool BoolValue { get { return _boolValue; } set { _boolValue = value; } }
            public int IntValue { get { return _intValue; } set { _intValue = value; } }

            public Variable()
            {
            }

            public Variable(ulong id, string name, VariableType type) 
            {
                _id = id;
                _name = name;
                _type = type;
            }
        }

        [SerializeField] private List<Variable> _variables = new();

        private Dictionary<ulong, Variable> _variablesDict = new();

        public Dictionary<ulong, Variable> VariablesDict { get { return _variablesDict; } }

        public bool HasVariable(string name)
        {
            foreach (Variable variable in _variablesDict.Values)
            {
                if (variable.Name == name)
                {
                    return true;
                }
            }
            return false;
        }

        public ulong AddVariableIfNotExists(string name, VariableType type)
        {
            Variable variable = _variables.FirstOrDefault((v) => v.Name == name);
            if (variable != null)
            {
                return variable.Id;
            }
            variable = new Variable(HexHelper.GenerateUID(), name, type);
            _variables.Add(variable);
            _variablesDict.Add(variable.Id, variable);
            return variable.Id;
        }

        public bool GetBool(string name)
        {
            foreach (Variable variable in _variablesDict.Values)
            {
                if (variable.Type == VariableType.Bool && variable.Name == name)
                {
                    return variable.BoolValue;
                }
            }
            return false;
        }

        public VariableType GetVariableType(ulong id)
        {
            if (_variablesDict.TryGetValue(id, out Variable variable))
            {
                return variable.Type;
            }
            return VariableType.Unknown;
        }

        public VariableType GetVariableType(string name)
        {
            foreach (Variable variable in _variablesDict.Values)
            {
                if (variable.Name == name)
                {
                    return variable.Type;
                }
            }
            return VariableType.Unknown;
        }

        public bool GetBool(ulong id)
        {
            if (_variablesDict.TryGetValue(id, out Variable variable) && variable.Type == VariableType.Bool)
            {
                return variable.BoolValue;
            }
            return false;
        }

        public void SetBool(string name, bool value)
        {
            foreach (Variable variable in _variablesDict.Values)
            {
                if (variable.Type == VariableType.Bool && variable.Name == name)
                {
                    variable.BoolValue = value;
                    return;
                }
            }
            Variable newVariable = new Variable(HexHelper.GenerateUID(), name, VariableType.Bool);
            newVariable.BoolValue = value;
            _variablesDict.Add(newVariable.Id, newVariable);
        }

        public void SetBool(ulong id, bool value)
        {
            if (_variablesDict.TryGetValue(id, out Variable variable) && variable.Type == VariableType.Bool)
            {
                variable.BoolValue = value;
            }
        }

        public int GetInt(string name)
        {
            foreach (Variable variable in _variablesDict.Values)
            {
                if (variable.Type == VariableType.Int && variable.Name == name)
                {
                    return variable.IntValue;
                }
            }
            return 0;
        }

        public int GetInt(ulong id)
        {
            if (_variablesDict.TryGetValue(id, out Variable variable) && variable.Type == VariableType.Int)
            {
                return variable.IntValue;
            }
            return 0;
        }

        public void SetInt(string name, int value)
        {
            foreach (Variable variable in _variablesDict.Values)
            {
                if (variable.Type == VariableType.Int && variable.Name == name)
                {
                    variable.IntValue = value;
                    return;
                }
            }
            Variable newVariable = new Variable(HexHelper.GenerateUID(), name, VariableType.Int);
            newVariable.IntValue = value;
            _variablesDict.Add(newVariable.Id, newVariable);
        }

        public void SetInt(ulong id, int value)
        {
            if (_variablesDict.TryGetValue(id, out Variable variable) && variable.Type == VariableType.Int)
            {
                variable.IntValue = value;
            }
        }

        void ISerializationCallbackReceiver.OnAfterDeserialize()
        {
            _variablesDict.Clear();
            foreach (Variable variable in _variables)
            {
                _variablesDict.Add(variable.Id, variable);
            }
        }

        void ISerializationCallbackReceiver.OnBeforeSerialize()
        {
            _variables.Clear();
            foreach (KeyValuePair<ulong, Variable> pair in _variablesDict)
            {
                _variables.Add(pair.Value);
            }
        }
    }
}
