﻿using System.Collections.Generic;
using UnityEngine;

namespace PK
{
    // https://www.redblobgames.com/pathfinding/a-star/introduction.html

    public static class PathfindingHelper
    {
        private static Queue<Vector2Int> _queue = new();
        private static PriorityQueue<Vector2Int, int> _frontier = new();
        private static Dictionary<Vector2Int, int> _costSoFar = new();
        private static Dictionary<Vector2Int, Vector2Int> _cameFrom = new();

        private static bool[] _currentTilesData = new bool[6];
        private static bool[] _tilesData = new bool[6];

        public static List<(Vector2Int, int)> FindPath(Vector2Int oddStart, Vector2Int oddEnd, bool[] obstacleMask, byte[] movementCostMask, int[] interactionMask, Vector2Int mapSize, List<Vector2Int> obstaclesToIgnore = null)
        {
            if (oddEnd.x < 0 || oddEnd.y < 0 || oddEnd.x >= mapSize.x || oddEnd.y >= mapSize.y)
            {
                return null;
            }

            List<(Vector2Int, int)> result = new();

            Vector2Int start = HexHelper.OddToAxial(oddStart);
            Vector2Int end = HexHelper.OddToAxial(oddEnd);

            PriorityQueue<Vector2Int, int> frontier = _frontier;
            Dictionary<Vector2Int, int> costSoFar = _costSoFar;
            Dictionary<Vector2Int, Vector2Int> cameFrom = _cameFrom;
            frontier.Clear();
            frontier.Enqueue(start, 0);
            costSoFar.Clear();
            costSoFar[start] = 0;
            cameFrom.Clear();
            cameFrom[start] = start;

            while (frontier.Count > 0)
            {
                Vector2Int current = frontier.Dequeue();
                Vector2Int oddCurrent = HexHelper.AxialToOdd(current);
                int currentIndex = oddCurrent.y * mapSize.x + oddCurrent.x;
                if (current == end)
                {
                    break;
                }
                bool hasInteractionMask = false;
                if (interactionMask[currentIndex] > 0)
                {
                    HexObjectMaskHelper.GetTilesData(interactionMask[currentIndex], false, _currentTilesData);
                    hasInteractionMask = true;
                }

                for (int i = 0; i < 6; i++)
                {
                    if (hasInteractionMask && !_currentTilesData[i])
                    {
                        continue;
                    }

                    Vector2Int next = current + HexHelper.AXIAL_DIRECTIONS[i];
                    Vector2Int oddNext = HexHelper.AxialToOdd(next);
                    int index = oddNext.y * mapSize.x + oddNext.x;
                    if (oddNext.x < 0 || oddNext.y < 0 || oddNext.x >= mapSize.x || oddNext.y >= mapSize.y)
                    {
                        continue;
                    }
                    if (obstacleMask[index] && (obstaclesToIgnore == null || !obstaclesToIgnore.Contains(oddNext)))
                    {
                        continue;
                    }
                    if (interactionMask[index] > 0)
                    {
                        HexObjectMaskHelper.GetTilesData(interactionMask[index], true, _tilesData);
                        if (!_tilesData[(i + 3) % 6]) // Flip side
                        {
                            continue;
                        }
                    }

                    int newCost = costSoFar[current] + movementCostMask[index]; // TODO roads work bad with creature squads //+ (roadMask[index] > 0 ? 1 : 2);
                    if (!costSoFar.ContainsKey(next) || newCost < costSoFar[next])
                    {
                        costSoFar[next] = newCost;
                        int heuristic = HexHelper.AxialDistance(end, next);
                        int priority = newCost + heuristic;
                        frontier.Enqueue(next, priority);
                        cameFrom[next] = current;
                    }
                }
            }

            Vector2Int position = end, previousPosition;
            while (position != start)
            {
                result.Insert(0, (HexHelper.AxialToOdd(position), costSoFar.ContainsKey(position) ? costSoFar[position] : int.MaxValue));
                previousPosition = position;
                if (!cameFrom.TryGetValue(previousPosition, out position))
                {
                    return null;
                }
                else
                {
                    cameFrom.Remove(previousPosition);
                }
            }
            result.Insert(0, (HexHelper.AxialToOdd(start), 0));
            return result;
        }
    
        public static void FindArea(Vector2Int oddStart, int maxCost, bool[] obstacleMask, byte[] movementCostMask, int[] interactionMask, Vector2Int mapSize, List<Vector2Int> result, List<Vector2Int> obstaclesToIgnore = null)
        {
            Vector2Int start = HexHelper.OddToAxial(oddStart);

            Dictionary<Vector2Int, int> costSoFar = _costSoFar;
            Queue<Vector2Int> queue = _queue;
            queue.Clear();
            queue.Enqueue(start);
            costSoFar.Clear();
            costSoFar[start] = 0;

            while (queue.Count > 0)
            {
                Vector2Int current = queue.Dequeue();
                Vector2Int oddCurrent = HexHelper.AxialToOdd(current);
                int currentIndex = oddCurrent.y * mapSize.x + oddCurrent.x;
                bool hasInteractionMask = false;
                if (interactionMask[currentIndex] > 0)
                {
                    HexObjectMaskHelper.GetTilesData(interactionMask[currentIndex], false, _currentTilesData);
                    hasInteractionMask = true;
                }

                for (int i = 0; i < 6; i++)
                {
                    if (hasInteractionMask && !_currentTilesData[i])
                    {
                        continue;
                    }

                    Vector2Int next = current + HexHelper.AXIAL_DIRECTIONS[i];
                    Vector2Int oddNext = HexHelper.AxialToOdd(next);
                    int index = oddNext.y * mapSize.x + oddNext.x;
                    if (oddNext.x < 0 || oddNext.y < 0 || oddNext.x >= mapSize.x || oddNext.y >= mapSize.y)
                    {
                        continue;
                    }
                    if (obstacleMask[index] && (obstaclesToIgnore == null || !obstaclesToIgnore.Contains(oddNext)))
                    {
                        continue;
                    }
                    if (interactionMask[index] > 0)
                    {
                        HexObjectMaskHelper.GetTilesData(interactionMask[index], true, _tilesData);
                        if (!_tilesData[(i + 3) % 6]) // Flip side
                        {
                            continue;
                        }
                    }

                    int newCost = costSoFar[current] + movementCostMask[index];
                    if (newCost > maxCost)
                    {
                        continue;
                    }
                    if (!costSoFar.ContainsKey(next) || newCost < costSoFar[next])
                    {
                        costSoFar[next] = newCost;
                        queue.Enqueue(next);
                        result.Add(oddNext);
                    }
                }
            }
        }

        public static Vector2Int FindEnemy(Vector2Int oddStart, HashSet<Vector2Int> enemyPositions, bool[] obstacleMask, Vector2Int mapSize, HashSet<Vector2Int> obstaclesToIgnore)
        {
            Vector2Int start = HexHelper.OddToAxial(oddStart);

            HashSet<Vector2Int> visitedSet = new();
            Queue<Vector2Int> queue = _queue;
            queue.Clear();
            queue.Enqueue(start);

            while (queue.Count > 0)
            {
                Vector2Int current = queue.Dequeue();
                for (int i = 0; i < 6; i++)
                {
                    Vector2Int next = current + HexHelper.AXIAL_DIRECTIONS[i];
                    Vector2Int oddNext = HexHelper.AxialToOdd(next);
                    if (enemyPositions.Contains(oddNext))
                    {
                        return oddNext;
                    }
                    int index = oddNext.y * mapSize.x + oddNext.x;
                    if (oddNext.x < 0 || oddNext.y < 0 || oddNext.x >= mapSize.x || oddNext.y >= mapSize.y)
                    {
                        continue;
                    }
                    if (obstacleMask[index] && (obstaclesToIgnore == null || !obstaclesToIgnore.Contains(oddNext)))
                    {
                        continue;
                    }
                    if (!visitedSet.Contains(next))
                    {
                        visitedSet.Add(next);
                        queue.Enqueue(next);
                    }
                }
            }

            return HexHelper.INVALID_POSITION;
        }

        public static bool CanMove(Vector2Int oddStart, int maxCost, byte[] movementCostMask, Vector2Int mapSize)
        {
            Vector2Int start = HexHelper.OddToAxial(oddStart);
            int index = oddStart.y * mapSize.x + oddStart.x;
            int cost = movementCostMask[index];
            bool canMove = cost < maxCost;
            for (int i = 0; i < 6; i++)
            {
                Vector2Int next = start + HexHelper.AXIAL_DIRECTIONS[i];
                Vector2Int oddNext = HexHelper.AxialToOdd(next);
                if (oddNext.x < 0 || oddNext.y < 0 || oddNext.x >= mapSize.x || oddNext.y >= mapSize.y)
                {
                    continue;
                }
                if (canMove)
                {
                    index = oddNext.y * mapSize.x + oddNext.x;
                    canMove = movementCostMask[index] + cost < maxCost;
                }
            }
            return canMove;
        }
    }
}
