﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.Pool;
using UnityEngine.U2D;

namespace PK
{
    public class SpriteHelper : IDisposable
    {
        private static Dictionary<Sprite, Bounds> _spriteBounds = new();

        private List<Vector3> _positions = ListPool<Vector3>.Get();
        private List<Vector2> _uvs = ListPool<Vector2>.Get();
        private List<int> _indicies = ListPool<int>.Get();

        public int IndexCount => _indicies.Count;

        public void Add(Vector3 position, Sprite sprite)
        {
            Vector2[] vertices = sprite.vertices;
            Vector2[] uv = sprite.uv;
            ushort[] triangles = sprite.triangles;

            int indexOffset = _positions.Count;
            for (int i = 0; i < vertices.Length; i++)
            {
                Vector3 vertex = vertices[i];
                _positions.Add(position + vertex);
                _uvs.Add(uv[i]);
            }

            for (int i = 0; i < triangles.Length; i++)
            {
                _indicies.Add(indexOffset + triangles[i]);
            }
        }

        public void Clear()
        {
            _positions.Clear();
            _uvs.Clear();
            _indicies.Clear();
        }

        public void Combine(SpriteHelper helper)
        {
            int indexOffset = _positions.Count;
            _positions.AddRange(helper._positions);
            _uvs.AddRange(helper._uvs);
            _indicies.AddRange(helper._indicies.Select((i) => indexOffset + i));
        }

        public void FillMesh(Mesh mesh)
        {
            mesh.Clear();
            mesh.indexFormat = _positions.Count >= 65000 ? UnityEngine.Rendering.IndexFormat.UInt32 : UnityEngine.Rendering.IndexFormat.UInt16;

            mesh.SetVertices(_positions);
            mesh.SetUVs(0, _uvs);
            mesh.SetTriangles(_indicies, 0);
            mesh.RecalculateBounds();
        }

        public static Bounds GetBounds(SpriteRenderer renderer)
        {
            if (renderer.sprite == null)
            {
                return new Bounds();
            }

            Bounds bounds = GetBounds(renderer.sprite);
            bounds.center += renderer.transform.position;
            return bounds;
        }

        public static void FillSecondaryUV(Sprite sprite)
        {
            if (sprite.HasVertexAttribute(UnityEngine.Rendering.VertexAttribute.TexCoord1))
            {
                return;
            }
            Vector2[] vertices = sprite.vertices;
            Bounds bounds = GetBounds(sprite);
            Vector4[] uv = vertices.Select((v) => new Vector4(0, Mathf.InverseLerp(bounds.min.y, bounds.max.y, v.y), bounds.size.x, bounds.size.y)).ToArray();
            sprite.SetVertexAttribute(UnityEngine.Rendering.VertexAttribute.Tangent, new Unity.Collections.NativeArray<Vector4>(uv, Unity.Collections.Allocator.Temp));
        }

        private static Bounds GetBounds(Sprite sprite)
        {
            if (!_spriteBounds.TryGetValue(sprite, out Bounds bounds))
            {
                bounds = new Bounds();
                foreach (Vector2 vertex in sprite.vertices)
                {
                    bounds.Encapsulate(vertex);
                }
                _spriteBounds[sprite] = bounds;
            }
            return bounds;
        }

        void IDisposable.Dispose()
        {
            ListPool<Vector3>.Release(_positions);
            ListPool<Vector2>.Release(_uvs);
            ListPool<int>.Release(_indicies);
        }
    }
}
