﻿using UnityEditor;
using UnityEngine;

namespace PK
{
    public static class PreviewSceneViewMotion
    {
        public static void HandleMotion(PreviewSceneView view)
        {
            switch (Event.current.type)
            {
                case EventType.MouseDrag:
                    if (Event.current.button != 0)
                    {
                        HandlePan(view);
                    }
                    break;
                case EventType.ScrollWheel:
                    HandleZoom(view);
                    break;
            }
        }

        private static void HandlePan(PreviewSceneView view)
        {
            Vector2 screenDelta = Event.current.delta;
            Vector3 worldDelta = ScreenToWorldDistance(view, new Vector2(-screenDelta.x, screenDelta.y));
            view.Pivot += worldDelta;
        }

        private static void HandleZoom(PreviewSceneView view)
        {
            float initialDistance = view.CameraDistance;

            float scrollDelta = Event.current.delta.y;
            float targetSize = Mathf.Abs(view.Size) * (scrollDelta * .015f + 1.0f);
            if (!(float.IsNaN(targetSize) || float.IsInfinity(targetSize)))
            {
                targetSize = Mathf.Clamp(targetSize, PreviewSceneView.MIN_VIEW_SIZE, PreviewSceneView.MAX_VIEW_SIZE);
                view.Size = targetSize;
            }

            if (Mathf.Abs(view.CameraDistance) < 1.0e7f)
            {
                float percentage = 1f - (view.CameraDistance / initialDistance);

                Ray mouseRay = HandleUtility.GUIPointToWorldRay(Event.current.mousePosition);
                Vector3 mousePivot = mouseRay.origin + mouseRay.direction * initialDistance;
                Vector3 pivotVector = mousePivot - view.Pivot;

                view.Pivot += pivotVector * percentage;
            }
        }

        private static Vector3 ScreenToWorldDistance(PreviewSceneView sceneElement, Vector2 delta)
        {
            // Ensures that the camera matrix doesn't end up with gigantic or minuscule values in the clip to world matrix
            const float k_MaxCameraSizeForWorldToScreen = 2.5E+7f;

            // store original camera and view values
            Camera camera = sceneElement.Camera;
            Vector3 pos = camera.transform.position;
            Quaternion rotation = camera.transform.rotation;
            float size = sceneElement.Size;

            // set camera transform and clip values to safe values
            sceneElement.Size = Mathf.Min(sceneElement.Size, k_MaxCameraSizeForWorldToScreen);
            // account for the distance clamping
            float scale = size / sceneElement.Size;
            sceneElement.Camera.transform.position = Vector3.zero;
            sceneElement.Camera.transform.rotation = Quaternion.identity;

            // do the distance calculation
            Vector3 pivotWorld = camera.transform.rotation * new Vector3(0f, 0f, sceneElement.CameraDistance);
            Vector3 pivotScreen = camera.WorldToScreenPoint(pivotWorld);
            pivotScreen += new Vector3(delta.x, delta.y, 0);

            Vector3 worldDelta = camera.ScreenToWorldPoint(pivotScreen) - pivotWorld;
            // We're clearing z here as ScreenToWorldPoint(WorldToScreenPoint(worldPoint)) does not always result in the exact same worldPoint that was inputed (for example, when camera is ortho).
            // https://jira.unity3d.com/browse/UUM-56425
            worldDelta.z = 0f;
            worldDelta = rotation * worldDelta;
            worldDelta *= EditorGUIUtility.pixelsPerPoint * scale;

            // restore original cam and scene values
            sceneElement.Size = size;
            sceneElement.Camera.transform.position = pos;
            sceneElement.Camera.transform.rotation = rotation;

            return worldDelta;
        }
    }
}
