﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using UnityEditor;
using UnityEngine;

namespace PK
{
    public class EditEntitiesTool : MapEditorTool
    {
        private ulong? _selectedInstanceId = null;
        private Dictionary<Type, Type> _customEditors = new();
        private SerializedObject _serializedMap;

        private EntityEditor _entityEditor;

        public ulong? SelectedInstanceId { get { return _selectedInstanceId; } }

        public EditEntitiesTool()
        {
            foreach (Type type in TypeCache.GetTypesDerivedFrom<EntityEditor>())
            {
                if (type.GetCustomAttribute<CustomEntityEditorAttribute>() is CustomEntityEditorAttribute attribute)
                {
                    _customEditors.Add(attribute.EntityType, type);
                }
            }
        }

        public override void OnMouseDown()
        {
            _entityEditor?.OnMouseDown();
        }

        public override void OnMouseClick()
        {
            _entityEditor?.OnMouseClick();

            if (Event.current.type == EventType.Used)
            {
                return;
            }

            if (Event.current.button != 0)
            {
                return;
            }

            Ray selectionRay = HandleUtility.GUIPointToWorldRay(Event.current.mousePosition);
            Vector3 origin = selectionRay.origin;
            List<(Vector3 position, ulong id)> intersectedInstances = new();
            foreach ((ulong id, Bounds bounds) data in GetBounds())
            {
                Bounds bounds = data.bounds;
                if (bounds.IntersectRay(selectionRay))
                {
                    intersectedInstances.Add((bounds.center, data.id));
                }
            }
            if (intersectedInstances.Count > 0)
            {
                SetSelectedInstanceId(intersectedInstances.OrderBy((o) => Vector3.Distance(o.position, origin)).FirstOrDefault().id);
                _serializedMap = new SerializedObject(_view.Map);
                Undo.undoRedoPerformed += OnUndoRedoPerformed;
            }
            else
            {
                SetSelectedInstanceId(null);
                if (_serializedMap != null)
                {
                    _serializedMap.Dispose();
                    _serializedMap = null;
                }
                Undo.undoRedoPerformed -= OnUndoRedoPerformed;
            }
            _editor.Repaint();
            Event.current.Use();
        }

        public override void OnMouseMove()
        {
            if (_entityEditor != null)
            {
                _entityEditor.HoveredTile = _view.GetHexPosition(GetMouseWorldPosition());
            }
        }

        public override void OnInspectorPanel()
        {
            if (!_selectedInstanceId.HasValue)
            {
                EditorGUILayout.HelpBox("Click on the object to select.", MessageType.Info);
            }
            else
            {
                _entityEditor?.OnInspectorPanel();
            }
        }

        public override void OnDrawHandle()
        {
            if (_selectedInstanceId.HasValue)
            {
                if (_view.Entities.TryGetValue(_selectedInstanceId.Value, out HexEntityView view))
                {
                    Bounds bounds = view.Bounds;
                    Handles.DrawWireCube(bounds.center, bounds.size);

                    Vector3 position = view.Position;
                    Vector3 newPosition = Handles.PositionHandle(position, Quaternion.identity);
                    if (position != newPosition)
                    {
                        RecordUndo();
                        Vector3 tilePosition = HexHelper.GetTilePosition(_view.GetHexPosition(newPosition));
                        Vector2 offset = HexHelper.GetOffset(tilePosition, newPosition);
                        _view.MoveObject(HexHelper.GetHexPosition(tilePosition), offset, _selectedInstanceId.Value);
                    }
                }
                else
                {
                    SetSelectedInstanceId(null);
                }
            }
        }

        public override void OnClear()
        {
            _selectedInstanceId = null;
            _entityEditor = null;
            if (_serializedMap != null)
            {
                _serializedMap.Dispose();
                _serializedMap = null;
            }
        }

        public void OnDrawSelection()
        {
            _entityEditor?.OnDrawSelection();
        }

        public override void OnKeyDown()
        {
            if (Event.current.keyCode == KeyCode.Delete && _selectedInstanceId.HasValue)
            {
                try
                {
                    RecordUndo();
                    _view.Map.Model.RemoveInteraction(_selectedInstanceId.Value);
                    _view.RemoveObject(_selectedInstanceId.Value);
                    SetSelectedInstanceId(null);
                    _serializedMap.Update();
                }
                catch
                {
                }
                Event.current.Use();
            }
            if (Event.current.control && Event.current.keyCode == KeyCode.C)
            {
                try
                {
                    if (_selectedInstanceId.HasValue)
                    {
                        (string entityType, string entity, string interactionType, string interaction) data = default;
                        HexEntityModel entity = _view.Map.Model.GetEntity(_selectedInstanceId.Value);
                        data.entityType = entity.GetType().AssemblyQualifiedName;
                        data.entity = JsonUtility.ToJson(entity);
                        if (_view.Map.Model.HasInteraction(entity.Id))
                        {
                            BaseInteraction interaction = _view.Map.Model.GetInteraction(entity.Id);
                            data.interactionType = interaction.GetType().AssemblyQualifiedName;
                            data.interaction = JsonUtility.ToJson(interaction);
                        }
                        EditorGUIUtility.systemCopyBuffer = JsonUtility.ToJson(data);
                    }
                }
                catch
                {
                }
                Event.current.Use();
            }
            if (Event.current.control && Event.current.keyCode == KeyCode.V)
            {
                (string entityType, string entity, string interactionType, string interaction) data = JsonUtility.FromJson<(string, string, string, string)>(EditorGUIUtility.systemCopyBuffer);
                try
                {
                    RecordUndo();
                    Type entityType = Type.GetType(data.entityType);
                    HexEntityModel entity = (HexEntityModel)JsonUtility.FromJson(data.entity, entityType);
                    if (!string.IsNullOrEmpty(entity.Name))
                    {
                        entity.Name += " Copy";
                    }
                    _selectedInstanceId = _view.AddEntity(entity);
                    if (!string.IsNullOrEmpty(data.interactionType))
                    {
                        Type interactionType = Type.GetType(data.interactionType);
                        BaseInteraction interaction = (BaseInteraction)JsonUtility.FromJson(data.interaction, interactionType);
                        _view.Map.Model.AddInteraction(interaction, entity.Id);
                    }
                    _serializedMap.Update();
                }
                catch
                {
                }
                Event.current.Use();
            }
        }

        public void SetSelectedInstanceId(ulong? id)
        {
            _selectedInstanceId = id;
            if (_serializedMap == null)
            {
                _serializedMap = new SerializedObject(_view.Map);
            }

            if (id == null)
            {
                _entityEditor = null;
                return;
            }

            int objectIndex = _view.Map.Model.GetObjectIndex(_selectedInstanceId.Value);    
            if (objectIndex != -1)
            {
                Type type = _view.Map.Model.GetEntity(id.Value).GetType();
                if (_customEditors.TryGetValue(type, out Type editorType))
                {
                    _entityEditor = Activator.CreateInstance(editorType) as EntityEditor;
                }
                else
                {
                    _entityEditor = new EntityEditor();
                }

                _serializedMap.UpdateIfRequiredOrScript();
                SerializedProperty objects = _serializedMap.FindProperty("_model").FindPropertyRelative("_entities");
                SerializedProperty objectProperty = objects.GetArrayElementAtIndex(objectIndex);

                _entityEditor.SerializedProperty = objectProperty;
                _entityEditor.View = _view;
                _entityEditor.Editor = _editor;
                _entityEditor.Map = _view.Map.Model;
                _entityEditor.Target = objectProperty.managedReferenceValue as HexEntityModel;
            }
            else
            {
                _entityEditor = null;
            }
        }

        private IEnumerable<(ulong id, Bounds bounds)> GetBounds()
        {
            return _view.Entities.Select((c) => (c.Key, c.Value.Bounds));
        }

        private void OnUndoRedoPerformed()
        {
            if (_serializedMap != null)
            {
                _serializedMap.UpdateIfRequiredOrScript();
            }
        }
    }
}
