using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    [DisallowMultipleComponent]
    public class HexMapView : MonoBehaviour
    {
        private static string LOCK = "MapViewLock";

        [SerializeField, HideInInspector] protected HexMap _map;

        protected bool _updates_paused = false;

        protected IMapEvents _events;
        protected IControllerLocker _locker;

        protected HexTerrainView _terrain;
        protected Dictionary<ulong, HexEntityView> _entities = new();

        public HexMap Map { get { return _map; } }
        public Dictionary<ulong, HexEntityView> Entities { get { return _entities; } }
        public HexTerrainView TerrainView { get { return _terrain; } }

        protected virtual void OnEnable()
        {
            Clear();
            UpdateAll();
        }

        protected virtual void AddEvents()
        {
            if (_map == null)
            {
                return;
            }

            IMapEvents events = _events;
            if (events != null)
            {
                events.OnEntityCreate += OnEntityCreate;
                events.OnEntityMove += OnEntityMove;
                events.OnEntityDestroy += OnEntityDestroy;
                events.OnEntityChange += OnEntityChange;
                events.OnEntitySetActive += OnEntitySetActive;
                events.OnHexTerrainChanged += OnHexTerrainChanged;
            }
        }

        protected virtual void RemoveEvents()
        {
            if (_map == null)
            {
                return;
            }

            IMapEvents events = _events;
            if (events != null)
            {
                events.OnEntityCreate -= OnEntityCreate;
                events.OnEntityMove -= OnEntityMove;
                events.OnEntityDestroy -= OnEntityDestroy;
                events.OnEntityChange -= OnEntityChange;
            }
        }

        private void OnEntityCreate(IEntityModelForView modelForView)
        {
            HexEntityView view = HexEntityViewFactory.Create(transform, modelForView, _map.IsStrategic);
            if (view != null)
            {
                view.Refresh(this);
                _entities[modelForView.Id] = view;
            }
        }

        private void OnEntityMove(ulong id, Vector2Int position, Vector2 offset)
        {
            if (_entities.TryGetValue(id, out HexEntityView view))
            {
                view.Move(position, offset);
            }
        }

        private void OnEntityDestroy(ulong id)
        {
            if (_entities.TryGetValue(id, out HexEntityView view))
            {
                _entities.Remove(id);
                view.Destroy();
            }
        }

        private void OnEntityChange(IEntityModelForView modelForView)
        {
            if (_entities.TryGetValue(modelForView.Id, out HexEntityView view))
            {
                view.Initialize(modelForView);
            }
        }

        private void OnEntitySetActive(ulong id, bool active)
        {
            if (_entities.TryGetValue(id, out HexEntityView view))
            {
                view.SetActive(active);
            }
        }

        private void OnHexTerrainChanged()
        {
            UpdateAll();
        }

        public Vector2Int GetHexPosition(Vector3 worldPosition)
        {
            Vector3 localPosition = transform.InverseTransformPoint(worldPosition);
            return HexHelper.GetHexPosition(localPosition);
        }

        public Bounds GetRendererBounds()
        {
            return _terrain.Bounds;
        }

        public void SetMap(HexMap hexMap, IMapEvents events = null, IControllerLocker locker = null)
        {
            Clear();
            _map = hexMap;
            _events = events;
            _locker = locker;
            UpdateAll();
        }

        public void UpdateAll()
        {
            RemoveEvents();
            AddEvents();
            UpdateTerrain();
            UpdateObjects();
        }

        public void UpdateTerrain()
        {
            if (_map == null || _updates_paused)
            {
                return;
            }
            _map.Model.Initialize();
            _map.Model.Validate();
            if (_terrain == null)
            {
                _terrain = gameObject.AddComponent<HexTerrainView>();
            }
            _terrain.Initialize(_map);
            _terrain.UpdateTiles();
        }

        public void UpdateObjects()
        {
            if (_map == null || _updates_paused)
            {
                return;
            }
            HashSet<ulong> existingObjects = new();
            foreach (HexEntityModel entity in _map.Model.GetEnumerable(true))
            {
                if (!_entities.TryGetValue(entity.Id, out HexEntityView view))
                {
                    OnEntityCreate(entity);
                    if (!_entities.TryGetValue(entity.Id, out view))
                    {
                        continue;
                    }
                }
                view.Refresh(this);
                existingObjects.Add(entity.Id);
            }
            foreach (KeyValuePair<ulong, HexEntityView> pair in _entities.Where((v) => !existingObjects.Contains(v.Key)).ToArray())
            {
                _entities.Remove(pair.Key);
                if (pair.Value != null)
                {
                    pair.Value.Destroy();
                }
            }
        }

        public void Clear()
        {
            foreach (Transform childTransform in transform.GetComponentsInChildren<Transform>(true))
            {
                if (childTransform != transform)
                {
                    if (Application.isPlaying)
                    {
                        Destroy(childTransform.gameObject);
                    }
                    else
                    {
                        DestroyImmediate(childTransform.gameObject);
                    }
                }
            }
            _entities.Clear();
            if (_terrain != null)
            {
                _terrain.Clear();
            }
        }
    }
}
