﻿using DG.Tweening;
using UnityEngine;

namespace PK
{
    public enum ObjectSelectionState
    {
        None,
        Reachable,
        Hovered
    }

    public class HexObjectView : HexEntityView
    {
        private const float FADE_OUT_TIME = 0.75f;

        private static int _shadowDataPropertyid = Shader.PropertyToID("_ShadowData");
        private static int _shadowOffsetMultiplierPropertyId = Shader.PropertyToID("_ShadowOffsetMultiplier");
        
        private HexObject _object;

        private SpriteRenderer _spriteRenderer;
        private Transform _transform;
        private MaterialPropertyBlock _propertyBlock;

        public override Bounds Bounds { get { return SpriteHelper.GetBounds(_spriteRenderer); } }
        public override Vector3 Position { get { return _transform.position; } }

        private MaterialPropertyBlock PropertyBlock
        {
            get
            {
                if (_propertyBlock == null)
                {
                    _propertyBlock = new();
                }
                return _propertyBlock;
            }
        }

        public override void Initialize(IEntityModelForView modelForView)
        {
            _uid = modelForView.Uid;
            _id = modelForView.Id;
            _object = HexDatabase.Instance.GetObject(_uid);

            if (_spriteRenderer == null)
            {
                _spriteRenderer = gameObject.AddComponent<SpriteRenderer>();
                _spriteRenderer.sharedMaterial = HexMaterials.Instance.ObjectMaterial;
                _spriteRenderer.color = Color.white;
                _spriteRenderer.spriteSortPoint = SpriteSortPoint.Pivot;
                _spriteRenderer.sortingOrder = SortingOrder.ENTITY;
                _transform = transform;
            }

            if (Application.isPlaying)
            {
                _spriteRenderer.enabled = modelForView.Enabled;
            }
            else
            {
                _spriteRenderer.sharedMaterial = modelForView.Enabled ? HexMaterials.Instance.ObjectMaterial : HexMaterials.Instance.ObjectInvisibleMaterial;
            }

            Move(modelForView.Position, ((IObjectModelForView)modelForView).Offset);
        }

        public override void SetActive(bool active)
        {
            if (Application.isPlaying)
            {
                if (active)
                {
                    _spriteRenderer.color = Color.white;
                    _spriteRenderer.enabled = active;
                }
                else
                {
                    _spriteRenderer.DOColor(new Color(1, 1, 1, 0), FADE_OUT_TIME).OnComplete(() => _spriteRenderer.enabled = false);
                }
            }
            else
            {
                _spriteRenderer.color = new Color(1, 1, 1, active ? 1 : 0.5f);
            }
        }

        public override void Refresh(HexMapView view)
        {
            HexTerrain terrain = view == null ? null : view.TerrainView.GetTerrain(view.Map.Model.GetEntity(_id).Position);
            Sprite sprite = terrain == null ? _object.AnySprite : _object.GetSprite(terrain);
            SpriteHelper.FillSecondaryUV(sprite);
            _spriteRenderer.sprite = sprite;

            MaterialPropertyBlock propertyBlock = PropertyBlock;
            _spriteRenderer.GetPropertyBlock(propertyBlock);
            propertyBlock.SetVector(_shadowDataPropertyid, new Vector4(HexShadowGroups.Instance.GetCastShadowMask(_object.ShadowGroup), HexShadowGroups.Instance.GetReceiveShadowMask(_object.ShadowGroup), _transform.position.y));
            _spriteRenderer.SetPropertyBlock(propertyBlock);
        }

        public void SetSelectionState(ObjectSelectionState state)
        {
            switch (state)
            {
                case ObjectSelectionState.None:
                    _spriteRenderer.sharedMaterial = HexMaterials.Instance.ObjectMaterial;
                    break;
                case ObjectSelectionState.Reachable:
                    _spriteRenderer.sharedMaterial = HexMaterials.Instance.ObjectReachableMaterial;
                    break;
                case ObjectSelectionState.Hovered:
                    _spriteRenderer.sharedMaterial = HexMaterials.Instance.ObjectHoveredMaterial;
                    break;
            }
        }

        public override void Move(Vector2Int position, Vector2 offset)
        {
            _transform.position = GetPosition(_object, position, offset);
            MaterialPropertyBlock propertyBlock = PropertyBlock;
            _spriteRenderer.GetPropertyBlock(propertyBlock);
            propertyBlock.SetVector(_shadowOffsetMultiplierPropertyId, _spriteRenderer.sharedMaterial.GetVector(_shadowOffsetMultiplierPropertyId) + _object.ShadowOffset);
            _spriteRenderer.SetPropertyBlock(propertyBlock);
        }

        private static Vector3 GetPosition(HexObject @object, Vector2Int position, Vector2 offset)
        {
            return HexHelper.GetTilePosition(position, offset + (@object.IsAligned ? @object.Offset : Vector2.zero)) + Vector3.back * 0.1f;
        }
    }
}
