using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEngine.UI;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using Unity.VisualScripting;

namespace PK
{
    public class MainMenuController : MonoBehaviour
    {
        private bool isMenuOpen = true;

        [SerializeField]
        private Canvas _backgroundCanvas;

        [SerializeField]
        private Button _playButton;

        [SerializeField]
        private Button _resumeButton;

        [SerializeField]
        private Button _battleButton;

        [SerializeField]
        private Button _saveButton;

        [SerializeField]
        private Button _loadButton;

        [SerializeField]
        private Button _quitButton;

        [SerializeField]
        private HexMap _storyMap;

        [SerializeField]
        private HexMap _battleMap;

        [SerializeField]
        private Slider _volumeSlider;

        [SerializeField]
        private Button _demoPitchButton;

        [SerializeField]
        private Button _demoEmailButton;

        private void Awake()
        {
            float savedVolume = PlayerPrefs.GetFloat("MasterVolume", 0.7f);
            _volumeSlider.onValueChanged.AddListener(OnVolumeSliderChanged);
            _volumeSlider.value = savedVolume;

            ToggleMenu();
        }

        void Start()
        {
            _playButton.onClick.AddListener(OnMouseDownPlayButton);
            _resumeButton.onClick.AddListener(OnMouseDownResumeButton);
            _battleButton.onClick.AddListener(OnMouseDownBattleButton);
            _saveButton.onClick.AddListener(OnMouseDownSaveButton);
            _loadButton.onClick.AddListener(OnMouseDownLoadButton);
            _quitButton.onClick.AddListener(OnMouseDownQuitButton);
            _demoPitchButton.onClick.AddListener(OnMouseDownDemoPitchButton);
            _demoEmailButton.onClick.AddListener(OnMouseDownDemoEmailButton);
            
#if UNITY_EDITOR
            HexMap map = UnityEditor.AssetDatabase.LoadAssetAtPath<HexMap>(UnityEditor.AssetDatabase.GUIDToAssetPath(UnityEditor.EditorPrefs.GetString("EditorMap")));
            if (map != null)
            {
                Destroy(gameObject);
                return;
            }
#endif

            if (!IsGameInitialized())
            {
                HexDatabase.Instance.TryValidate();
                ToggleMenu();
            }
        }

        void Update()
        {
            if (Input.GetKeyDown(KeyCode.Escape))
            {
                ToggleMenu();
            }
        }

        private bool IsGameInitialized()
        {
            return GameClient.Instance != null && GameClient.Instance.IsInitialized();
        }

        private bool IsSavingAvailable()
        {
            return IsGameInitialized() && GameClient.Instance.IsOnStrategicMap();
        }

        private bool IsLoadingAvailable()
        {
            return GameClient.Instance.IsSaveExist() && GameClient.Instance.IsOnStrategicMap();
        }

        private void ToggleMenu()
        {
            isMenuOpen = !isMenuOpen;
            if (isMenuOpen)
            {
                _backgroundCanvas.gameObject.SetActive(true);
                AudioHandler.Instance.PlayAmbience("MenuAmbience", 10);

                _playButton.gameObject.SetActive(!IsGameInitialized());
                _resumeButton.gameObject.SetActive(IsGameInitialized());
                _saveButton.gameObject.SetActive(IsSavingAvailable());
                _loadButton.gameObject.SetActive(IsLoadingAvailable());
#if UNITY_EDITOR
                _battleButton.gameObject.SetActive(true);
                _battleButton.interactable = true;
#else
                _battleButton.gameObject.SetActive(false);
#endif

                _resumeButton.interactable = true;
                _loadButton.interactable = true;

                if (IsGameInitialized() || IsLoadingAvailable())
                {
                    _demoPitchButton.gameObject.SetActive(true);
                    _demoEmailButton.gameObject.SetActive(true);
                }
                else
                {
                    _demoPitchButton.gameObject.SetActive(false);
                    _demoEmailButton.gameObject.SetActive(false);
                }
            }
            else
            {
                _backgroundCanvas.gameObject.SetActive(false);
                if (AudioHandler.Instance != null)
                {
                    AudioHandler.Instance.PopAmbience(10);
                }
            }
        }

        private void OnMouseDownPlayButton()
        {
            _playButton.interactable = false;
            GameClient.Instance.InitializeWithMap(_storyMap);
            ToggleMenu();
        }

        private void OnMouseDownResumeButton()
        {
            _resumeButton.interactable = false;
            ToggleMenu();
        }

        private void OnMouseDownBattleButton()
        {
            _battleButton.interactable = false;
            GameClient.Instance.InitializeWithMap(_battleMap);
            ToggleMenu();
        }

        private void OnMouseDownSaveButton()
        {
            _saveButton.interactable = false;
            if (IsSavingAvailable())
            {
                GameClient.Instance.Save();
                ToggleMenu();
            }
        }

        private void OnMouseDownLoadButton()
        {
            _loadButton.interactable = false;
            if (IsLoadingAvailable())
            {
                GameClient.Instance.Load();
                ToggleMenu();
            }
        }

        private void OnMouseDownQuitButton()
        {
            _quitButton.interactable = false;
#if UNITY_EDITOR
            UnityEditor.EditorApplication.isPlaying = false;
#else
            Application.Quit();
#endif
        }

        private void OnMouseDownDemoPitchButton()
        {
			Application.OpenURL("https://drive.google.com/drive/u/0/folders/1X96VOaQ5_995QF69yviVqcTjwKLLz6b3");
        }

        private void OnMouseDownDemoEmailButton()
        {
			Application.OpenURL("mailto:flame.beholder@gmail.com");
        }

        private void OnVolumeSliderChanged(float value)
        {
            if (AudioHandler.Instance == null)
            {
                return;
            }
            PlayerPrefs.SetFloat("MasterVolume", value);
            AudioHandler.Instance.SetMasterVolume(value);
        }
    }
}
