using UnityEngine;
using System.Collections;

namespace PK
{
    /// <summary>
    /// Manages a Particle System to simulate weather effects based on HexWeather ScriptableObjects.
    /// Controls emission, appearance, wind, gusts, and Level of Detail (LOD).
    /// Requires HexMapView reference for positioning and GameClient/HexMap access for map size data.
    /// </summary>
    [RequireComponent(typeof(ParticleSystem))]
    public class WeatherSystem : MonoBehaviour
    {
        // --- Singleton ---
        public static WeatherSystem Instance { get; private set; }

        // --- Inspector References ---
        [Header("Map Reference")]
        [Tooltip("Reference to the HexMapView component for map bounds information.")]
        public HexMapView strategicMapView; // Assign in Inspector; required for positioning/scaling emitter.

        // --- Particle System Modules (Cached) ---
        private ParticleSystem particleSystem;
        private ParticleSystem.MainModule mainModule;
        public ParticleSystem.EmissionModule emissionModule; // Public for potential external access/monitoring
        public ParticleSystem.ShapeModule shapeModule;
        public ParticleSystem.VelocityOverLifetimeModule velocityModule;
        public ParticleSystem.NoiseModule noiseModule;
        private ParticleSystemRenderer particleRenderer;

        // --- Current Weather State ---
        public HexWeather currentWeatherState { get; private set; } // The active HexWeather SO.
        public HexWeather.Severity currentSeverity { get; private set; } // The active severity level.
        private HexWeather.SeveritySettings currentSeveritySettings; // Cached settings for current state/severity.

        // --- Wind State ---
        private Vector3 baseWindDirection; // Base wind used in calculations, affected by random changes.
        private Vector3 temporaryWindAdjustment = Vector3.zero; // Currently unused placeholder.
        private Coroutine windChangeCoroutine; // Handles periodic random changes to baseWindDirection.
        public Coroutine windGustCoroutine; // Handles particle velocity bursts (gusts). Public for potential external monitoring.

        // --- Constants ---
        private const float WIND_OCCLUSION_DISTANCE = 10f; // Max distance for wind occlusion raycast. Consider making configurable if needed.

        // --- Unity Methods ---
        void Awake()
        {
            // Singleton Setup
            if (Instance != null && Instance != this)
            {
                Destroy(gameObject);
                return;
            }
            Instance = this;

            // Component & Module References
            particleSystem = GetComponent<ParticleSystem>();
            particleRenderer = GetComponent<ParticleSystemRenderer>();
            if (particleSystem == null || particleRenderer == null)
            {
                Debug.LogError("[WeatherSystem] Missing ParticleSystem or ParticleSystemRenderer component!", this);
                this.enabled = false; // Disable script if components missing
                return;
            }
            mainModule = particleSystem.main;
            emissionModule = particleSystem.emission;
            shapeModule = particleSystem.shape;
            velocityModule = particleSystem.velocityOverLifetime;
            noiseModule = particleSystem.noise;

            // Initial State
            particleSystem.Stop(true, ParticleSystemStopBehavior.StopEmittingAndClear);
        }

        void OnDestroy()
        {
            if (Instance == this)
            {
                Instance = null;
            }
            StopWeatherCoroutines(); // Ensure coroutines are cleaned up
        }

        // Update is used for continuous adjustments like wind velocity and LOD.
        void Update()
        {
            // Only update if weather is active and settings are loaded.
            if (currentWeatherState == null || currentSeverity == HexWeather.Severity.None || currentSeveritySettings == null)
            {
                return;
            }

            UpdateWindDirection(); // Apply wind forces.
            UpdateLOD();           // Adjust particle properties based on camera distance.
        }

        // --- Public API ---

        /// <summary>
        /// Sets the active weather state and severity, configuring and starting the particle system.
        /// </summary>
        /// <param name="state">The HexWeather ScriptableObject defining the weather type.</param>
        /// <param name="severity">The intensity level of the weather.</param>
        public void SetWeather(HexWeather state, HexWeather.Severity severity)
        {
            // Prevent unnecessary updates if the state and severity haven't changed.
            if (currentWeatherState == state && currentSeverity == severity) return;

            // Stop existing effects before applying new ones.
            StopWeatherCoroutines();
            particleSystem.Stop(true, ParticleSystemStopBehavior.StopEmittingAndClear);

            // Update internal state references.
            currentWeatherState = state;
            currentSeverity = severity;
            currentSeveritySettings = null; // Clear cached settings

            // Apply new state only if it's valid (not null state and not None severity).
            if (currentWeatherState != null && currentSeverity != HexWeather.Severity.None)
            {
                // Attempt to load settings for the new state/severity from the HexWeather SO.
                currentSeveritySettings = currentWeatherState.GetSettingsForSeverity(currentSeverity);

                // Check if settings were successfully retrieved.
                if (currentSeveritySettings == null)
                {
                    Debug.LogError($"[WeatherSystem] Failed to get SeveritySettings for {currentWeatherState.name} / {currentSeverity}. Weather cannot be applied.", this);
                    particleSystem.Stop(true, ParticleSystemStopBehavior.StopEmittingAndClear); // Ensure system remains stopped.
                    return; // Exit early if settings are missing.
                }

                // Apply the core particle system module settings based on the loaded state and severity.
                ApplyWeatherState();

                // Position and scale the emitter based on the map view and camera.
                // Called *after* ApplyWeatherState as some calculations might depend on settings.
                AdjustEmitterToMap();

                // Start emission and dynamic wind effects now that configuration is complete.
                particleSystem.Play();
                windChangeCoroutine = StartCoroutine(RandomWindChanges());
                windGustCoroutine = StartCoroutine(WindGustRoutine());
            }
            // If state is null or severity is None, the system remains stopped from the earlier call.
        }

        /// <summary>
        /// Enables or disables specific Particle System modules at runtime. Used for debugging or special effects.
        /// </summary>
        /// <param name="featureName">Case-sensitive name of the module ("Noise", "VelocityOverLifetime", "Emission").</param>
        /// <param name="isEnabled">True to enable, false to disable.</param>
        public void ToggleFeature(string featureName, bool isEnabled)
        {
            if (particleSystem == null) return;

            switch (featureName)
            {
                case "Noise":
                    if (noiseModule.enabled != isEnabled) noiseModule.enabled = isEnabled;
                    break;
                case "VelocityOverLifetime":
                    if (velocityModule.enabled != isEnabled) velocityModule.enabled = isEnabled;
                    break;
                case "Emission":
                    if (emissionModule.enabled != isEnabled) emissionModule.enabled = isEnabled;
                    break;
                default:
                    Debug.LogWarning($"[WeatherSystem] ToggleFeature: Unknown feature name '{featureName}'.", this);
                    break;
            }
        }


        // --- Internal Logic ---

        /// <summary>
        /// Applies settings from the current HexWeather state and SeveritySettings to the Particle System modules.
        /// </summary>
        private void ApplyWeatherState()
        {
            // Safety checks, though should be guaranteed by SetWeather logic.
            if (currentWeatherState == null || currentSeveritySettings == null || particleSystem == null) return;

            // --- Renderer --- (Uses top-level HexWeather fields)
            particleRenderer.material = currentWeatherState.particleMaterial;
            particleRenderer.sortingLayerName = currentWeatherState.sortingLayerName;
            particleRenderer.sortingOrder = currentWeatherState.sortingOrder;

            // --- Texture Sheet Animation --- (Uses top-level HexWeather field)
            var texSheet = particleSystem.textureSheetAnimation;
            if (currentWeatherState.particleSprite != null)
            {
                texSheet.enabled = true;
                texSheet.mode = ParticleSystemAnimationMode.Sprites;
                // Ensure previous sprites are cleared if reusing the system
                if(texSheet.spriteCount > 0) texSheet.RemoveSprite(0);
                texSheet.AddSprite(currentWeatherState.particleSprite);
            }
            else
            {
                texSheet.enabled = false; // Disable if no sprite is assigned in the HexWeather SO.
            }

            // --- Main Module --- (Uses SeveritySettings fields, except duration)
            // Duration might be considered a general property of the weather type, not severity-dependent.
            // If you want duration per severity, add it to SeveritySettings in HexWeather.cs.
            // mainModule.duration = currentWeatherState.particleDuration; // Assuming particleDuration exists in HexWeather
            mainModule.duration = 5f; // Using a fixed duration for now. Adjust as needed or add to HexWeather.
            mainModule.loop = true; // Weather effects typically loop.
            mainModule.startLifetime = new ParticleSystem.MinMaxCurve(currentSeveritySettings.minLifetime, currentSeveritySettings.maxLifetime);
            mainModule.startSpeed = new ParticleSystem.MinMaxCurve(currentSeveritySettings.minSpeed, currentSeveritySettings.maxSpeed);
            mainModule.startSize = new ParticleSystem.MinMaxCurve(currentSeveritySettings.minSize, currentSeveritySettings.maxSize);
            mainModule.startColor = currentSeveritySettings.startColor; // Uses Color directly now
            mainModule.maxParticles = CalculateMaxParticles(); // Estimate required particle count based on rate/lifetime.

            // --- Emission Module --- (Uses SeveritySettings field)
            emissionModule.rateOverTime = GetEmissionRateForMap(); // Calculate rate based on map size.

            // --- Shape Module --- (Scale/Position are set later in AdjustEmitterToMap)
            shapeModule.shapeType = ParticleSystemShapeType.Box; // Using Box shape for map coverage.

            // --- Velocity Over Lifetime Module --- (Velocity values applied continuously in UpdateWindDirection)
            velocityModule.enabled = true; // Assume wind effect is always desired when weather is active.

            // --- Noise Module --- (Uses SeveritySettings fields)
            noiseModule.enabled = currentSeveritySettings.noiseEnabled;
            if (noiseModule.enabled)
            {
                noiseModule.strength = currentSeveritySettings.noiseStrength;
                noiseModule.frequency = currentSeveritySettings.noiseFrequency;
                noiseModule.scrollSpeed = currentSeveritySettings.noiseScrollSpeed;
                noiseModule.quality = currentSeveritySettings.noiseQuality;
            }

            // --- Trails Module --- (Explicitly disabled, enable if needed)
            var trailModule = particleSystem.trails;
            trailModule.enabled = false;

            // --- Collision Module --- (Explicitly disabled, enable if needed)
            var collisionModule = particleSystem.collision;
            collisionModule.enabled = false;

            // --- Initial Wind Direction --- (Uses SeveritySettings field, basis for UpdateWindDirection)
            baseWindDirection = currentSeveritySettings.baseWindDirection;
            temporaryWindAdjustment = Vector3.zero; // Reset any temporary adjustments from previous states.
        }

        /// <summary>
        /// Calculates the emission rate based on map size (from GameClient/HexMap) and severity settings.
        /// Falls back to the base rate if map data is unavailable.
        /// </summary>
        /// <returns>The calculated emission rate over time.</returns>
        private float GetEmissionRateForMap()
        {
            // Requires severity settings to get the base rate per hex.
            if (currentSeveritySettings == null) return 0f;

            HexMap hexMap = null;
            // Attempt to get map data from a central GameClient (assumed structure).
            if (GameClient.Instance != null && GameClient.Instance.IsInitialized())
            {
                hexMap = GameClient.Instance.GetHexMap();
            }

            if (hexMap != null)
            {
                Vector2Int mapSize = hexMap.Size;
                // Ensure map size is valid before calculating total hexes.
                if (mapSize.x > 0 && mapSize.y > 0)
                {
                    float totalHexes = mapSize.x * mapSize.y;
                    return currentSeveritySettings.emissionRatePerHex * totalHexes;
                }
                else
                {
                     // Log warning and fallback if map size is invalid.
                     Debug.LogWarning($"[WeatherSystem] GetEmissionRateForMap - HexMap size is zero or negative ({mapSize}). Using base rate per hex.", this);
                     return currentSeveritySettings.emissionRatePerHex;
                }
            }
            else
            {
                // Fallback if map data couldn't be retrieved.
                // Debug.LogWarning("[WeatherSystem] GetEmissionRateForMap - HexMap not available from GameClient. Using base rate per hex.", this);
                return currentSeveritySettings.emissionRatePerHex;
            }
        }


        /// <summary>
        /// Estimates the maximum number of particles needed based on emission rate and lifetime, with a buffer.
        /// Clamps the result within reasonable bounds.
        /// </summary>
        /// <returns>The estimated maximum particles required.</returns>
        private int CalculateMaxParticles()
        {
            if (currentSeveritySettings == null) return 1000; // Default fallback if settings are missing.

            float emissionRate = GetEmissionRateForMap(); // Get the calculated emission rate.
            // Ensure max lifetime is positive to avoid issues in calculation.
            float maxLifetime = currentSeveritySettings.maxLifetime > 0.1f ? currentSeveritySettings.maxLifetime : 0.1f;

            // Estimate: Rate * Lifetime + 50% buffer.
            int calculatedMax = Mathf.CeilToInt(emissionRate * maxLifetime * 1.5f);
            // Clamp within practical limits to prevent excessive allocation.
            return Mathf.Clamp(calculatedMax, 100, 50000); // Adjust max clamp as needed for performance/visuals.
        }

        /// <summary>
        /// Stops all active weather-related coroutines (wind changes, gusts).
        /// </summary>
        private void StopWeatherCoroutines()
        {
            if (windChangeCoroutine != null) StopCoroutine(windChangeCoroutine);
            if (windGustCoroutine != null) StopCoroutine(windGustCoroutine);
            windChangeCoroutine = null;
            windGustCoroutine = null;
        }

        /// <summary>
        /// Calculates and applies the current wind velocity (including turbulence and occlusion)
        /// to the VelocityOverLifetime module. Called from Update().
        /// </summary>
        void UpdateWindDirection()
        {
            // Safety check, though Update() should already verify this.
            if (currentSeveritySettings == null) return;

            // Calculate the effective wind direction including temporary adjustments (if any).
            Vector3 effectiveBaseWind = baseWindDirection + temporaryWindAdjustment;
            // Add Perlin noise-based turbulence to the base wind.
            Vector3 calculatedWind = CalculateWindForce(effectiveBaseWind);
            // Reduce wind strength based on 2D physics raycasts against obstruction layers.
            calculatedWind = CalculateWindOcclusion(calculatedWind);

            // Apply the final calculated wind vector to the particle system module.
            velocityModule.x = calculatedWind.x;
            velocityModule.y = calculatedWind.y;
            // Apply separate noise for Z velocity (up/down flutter effect).
            velocityModule.z = Mathf.Lerp(
                -currentSeveritySettings.zVelocityNoiseAmplitude,
                 currentSeveritySettings.zVelocityNoiseAmplitude,
                 Mathf.PerlinNoise(Time.time * currentSeveritySettings.zVelocityNoiseFrequency, 0f) // Using Time.time for continuous change
            );
        }

        /// <summary>
        /// Calculates wind turbulence using Perlin noise based on severity settings.
        /// </summary>
        /// <param name="currentBaseWind">The base wind direction before adding turbulence.</param>
        /// <returns>The wind vector including turbulence.</returns>
        Vector3 CalculateWindForce(Vector3 currentBaseWind)
        {
            if (currentSeveritySettings == null) return Vector3.zero;

            // Use Perlin noise for smoother, more natural turbulence.
            // Offset sample points in noise space to get less correlated X and Y values.
            float noiseX = (Mathf.PerlinNoise(Time.time * currentSeveritySettings.windFrequency, 100.5f) - 0.5f) * 2f; // Normalize roughly to -1 to 1
            float noiseY = (Mathf.PerlinNoise(50.1f, Time.time * currentSeveritySettings.windFrequency) - 0.5f) * 2f; // Normalize roughly to -1 to 1

            // Scale noise by turbulence strength and apply primarily horizontally.
            Vector3 turbulence = new Vector3(
                noiseX * currentSeveritySettings.windTurbulence,
                noiseY * currentSeveritySettings.windTurbulence,
                0f
            );

            return currentBaseWind + turbulence;
        }


        /// <summary>
        /// Reduces wind strength based on 2D raycasts against obstruction layers defined in the HexWeather state.
        /// </summary>
        /// <param name="windToOcclude">The calculated wind vector before occlusion.</param>
        /// <returns>The potentially reduced wind vector after checking for obstructions.</returns>
        Vector3 CalculateWindOcclusion(Vector3 windToOcclude)
        {
            // Check if occlusion is relevant (state exists, mask is set).
            if (currentWeatherState == null || currentSeveritySettings == null || currentWeatherState.windObstructionMask == 0) // LayerMask 0 means "Nothing".
            {
                return windToOcclude; // Skip occlusion check.
            }

            // Perform a 2D raycast from the emitter's position in the direction of the wind.
            Vector3 rayOrigin = transform.position;
            RaycastHit2D hit = Physics2D.Raycast(
                rayOrigin,
                windToOcclude.normalized, // Cast direction.
                WIND_OCCLUSION_DISTANCE,  // Max distance to check for occlusion.
                currentWeatherState.windObstructionMask // Layers defined in HexWeather SO.
            );

            // If the raycast hits an object on an obstruction layer...
            if (hit.collider != null)
            {
                // Calculate occlusion factor (0 = fully blocked at origin, 1 = not blocked at max distance).
                float occlusionFactor = Mathf.Clamp01(hit.distance / WIND_OCCLUSION_DISTANCE);
                // Linearly interpolate between zero wind and full wind based on the factor.
                return Vector3.Lerp(Vector3.zero, windToOcclude, occlusionFactor);
            }
            else
            {
                // No hit, return the original wind vector.
                return windToOcclude;
            }
        }


        /// <summary>
        /// Coroutine to periodically apply small, random additive changes to the base wind direction vector.
        /// </summary>
        IEnumerator RandomWindChanges()
        {
            while (true) // Loop indefinitely while the weather state is active.
            {
                // Ensure settings are still valid before proceeding.
                if (currentSeveritySettings == null) yield break; // Exit coroutine if weather stopped.

                // Wait for a random interval defined in the severity settings.
                float interval = Random.Range(currentSeveritySettings.windIncreaseIntervalMin, currentSeveritySettings.windIncreaseIntervalMax);
                yield return new WaitForSeconds(interval);

                // Re-check settings after waiting, in case the weather changed during the pause.
                if (currentSeveritySettings == null) yield break;

                // Calculate a random direction vector scaled by the increase strength.
                Vector3 randomIncrease = Random.insideUnitSphere * currentSeveritySettings.windIncreaseStrength;
                randomIncrease.z = 0; // Keep wind changes primarily horizontal.

                // Add the random change to the base wind direction.
                baseWindDirection += randomIncrease;

                // Optional: Clamp the magnitude of baseWindDirection here if needed to prevent runaway wind speeds.
                // float maxWindMagnitude = currentSeveritySettings.baseWindDirection.magnitude * 2.0f; // Example clamp
                // baseWindDirection = Vector3.ClampMagnitude(baseWindDirection, maxWindMagnitude);
            }
        }

        /// <summary>
        /// Coroutine to periodically apply velocity bursts (gusts) to a percentage of active particles.
        /// Uses GetParticles/SetParticles, which can have performance implications with very high particle counts.
        /// </summary>
        IEnumerator WindGustRoutine()
        {
            // Buffer for particle data, reused across iterations.
            ParticleSystem.Particle[] particleBuffer = null;

            while (true) // Loop indefinitely while the weather state is active.
            {
                 // Ensure settings are still valid before proceeding.
                if (currentSeveritySettings == null) yield break; // Exit coroutine if weather stopped.

                // Allocate or resize the particle buffer if needed (e.g., after maxParticles change).
                if (particleBuffer == null || particleBuffer.Length < mainModule.maxParticles)
                {
                    particleBuffer = new ParticleSystem.Particle[mainModule.maxParticles];
                }

                // Wait for a random interval before triggering the next gust.
                float gustDelay = Random.Range(currentSeveritySettings.windGustFrequencyMin, currentSeveritySettings.windGustFrequencyMax);
                yield return new WaitForSeconds(gustDelay);

                // Re-check settings after waiting.
                 if (currentSeveritySettings == null) yield break;

                // Get the current live particles from the system.
                int liveParticleCount = particleSystem.GetParticles(particleBuffer);

                if (liveParticleCount > 0)
                {
                    // Calculate how many particles to affect based on the percentage setting.
                    int affectedCount = Mathf.CeilToInt(liveParticleCount * currentSeveritySettings.windGustAffectedPercentage / 100f);
                    affectedCount = Mathf.Clamp(affectedCount, 0, liveParticleCount); // Ensure count is valid.

                    // Apply gust velocity to randomly selected particles.
                    for (int i = 0; i < affectedCount; i++)
                    {
                        // Simple random index selection. Could be optimized for large counts if needed.
                        int randomIndex = Random.Range(0, liveParticleCount);

                        // Calculate the gust velocity vector.
                        Vector3 gustVector = Random.insideUnitSphere; // Random direction.
                        gustVector.z *= 0.2f; // Reduce vertical component of gust slightly.
                        // Scale by turbulence and the gust strength multiplier.
                        gustVector = gustVector.normalized * currentSeveritySettings.windTurbulence * currentSeveritySettings.windGustStrengthMultiplier;

                        // Add the gust velocity directly to the particle's current velocity.
                        particleBuffer[randomIndex].velocity += gustVector;
                    }

                    // Apply the modified particle data back to the system.
                    particleSystem.SetParticles(particleBuffer, liveParticleCount);
                }

                // Wait for the configured gust duration before the *next* gust can start.
                // This controls the time *between* gusts, not the visual duration of a single gust's effect.
                yield return new WaitForSeconds(currentSeveritySettings.windGustDuration);
            }
        }


        /// <summary>
        /// Adjusts particle emission rate and start speed based on the distance to the main camera,
        /// interpolating between near and far distance values defined in the HexWeather state.
        /// Called from Update().
        /// </summary>
        void UpdateLOD()
        {
            Camera mainCam = Camera.main; // Cache main camera reference.
            // Check prerequisites: main camera, active weather state, and loaded settings.
            if (mainCam == null || currentWeatherState == null || currentSeveritySettings == null)
            {
                 // Cannot calculate LOD without camera or weather data.
                 // Consider logging a warning if mainCam is null repeatedly.
                 return;
            }

            // Calculate the distance from the camera to the particle system emitter.
            float camDistance = Vector3.Distance(mainCam.transform.position, transform.position);

            // Calculate the LOD factor (0 at near distance, 1 at far distance).
            // Uses distances defined directly in the HexWeather SO.
            float lodFactor = Mathf.InverseLerp(currentWeatherState.nearDistance, currentWeatherState.farDistance, camDistance);
            lodFactor = Mathf.Clamp01(lodFactor); // Ensure the factor stays within the 0-1 range.

            // --- Interpolate Emission Rate ---
            // Get the base emission rate calculated for the current map size.
            float baseEmissionRate = GetEmissionRateForMap();
            // Lerp between the base rate and the rate multiplied by the LOD multiplier.
            emissionModule.rateOverTime = Mathf.Lerp(
                baseEmissionRate,
                baseEmissionRate * currentSeveritySettings.lodEmissionRateMultiplier,
                lodFactor
            );

            // --- Interpolate Start Speed ---
            // Get the base min/max speeds from severity settings.
            float minSpeed = currentSeveritySettings.minSpeed;
            float maxSpeed = currentSeveritySettings.maxSpeed;
            // Lerp both min and max speeds based on the LOD factor and multiplier.
            mainModule.startSpeed = new ParticleSystem.MinMaxCurve(
                Mathf.Lerp(minSpeed, minSpeed * currentSeveritySettings.lodStartSpeedMultiplier, lodFactor),
                Mathf.Lerp(maxSpeed, maxSpeed * currentSeveritySettings.lodStartSpeedMultiplier, lodFactor)
            );

             // Other properties like start size or noise strength could also be adjusted here based on LOD factor if desired.
        }

        /// <summary>
        /// Positions the particle emitter transform and configures the Shape module's size/position
        /// based on the strategic map view bounds and main camera position.
        /// Tries to use HexMap logical size for shape scaling if available, otherwise falls back to visual bounds.
        /// Called from SetWeather() after ApplyWeatherState().
        /// </summary>
        void AdjustEmitterToMap()
        {
            // Requires the strategic map view reference to be assigned.
            if (strategicMapView == null)
            {
                 Debug.LogError("[WeatherSystem] AdjustEmitterToMap - strategicMapView is NULL. Cannot position emitter.", this);
                 return;
            }

            // Get the visual bounds of the map view's renderer.
            Bounds mapBounds = strategicMapView.GetRendererBounds();
            // Check if bounds are valid (size > 0). Might be zero if renderer is disabled or not ready.
            if (mapBounds.size == Vector3.zero)
            {
                 Debug.LogWarning("[WeatherSystem] AdjustEmitterToMap - strategicMapView bounds size is zero. Map might not be ready or visible. Emitter position/scale may be incorrect.", this);
                 // Consider returning or using default values if bounds are zero?
            }

            // Get the main camera's Z position for relative emitter placement.
            Camera mainCam = Camera.main;
            // Provide a fallback Z distance if Camera.main is not found.
            float cameraZDistance = (mainCam != null) ? mainCam.transform.position.z : -10f;
            if(mainCam == null)
            {
                 Debug.LogWarning("[WeatherSystem] AdjustEmitterToMap - Camera.main is NULL! Using default Z distance for emitter.", this);
            }


            // --- Position Emitter ---
            // Center the emitter horizontally (X) and vertically (Y) on the map's visual bounds.
            // Place the emitter's Z position halfway between the camera (typically negative Z) and the map plane (Z=0).
            Vector3 emitterPosition = new Vector3(
                mapBounds.center.x,
                mapBounds.center.y, // Centered Y on map bounds
                cameraZDistance / 2f
            );
            transform.position = emitterPosition;

            // --- Configure Shape Module Scale ---
            Vector3 shapeScale;
            HexMap hexMap = null;
            // Attempt to get logical map data for more accurate scaling.
             if (GameClient.Instance != null && GameClient.Instance.IsInitialized())
            {
                hexMap = GameClient.Instance.GetHexMap();
            }

            // If logical map data is available and valid...
            if (hexMap != null && hexMap.Size.x > 0 && hexMap.Size.y > 0)
            {
                 // Calculate approximate world size from hex data.
                 // This calculation is a placeholder and might need adjustment based on your specific hex metrics/layout.
                 // Assumes simple grid size multiplication, replace with HexMetrics if available and needed.
                 float mapWorldWidth = hexMap.Size.x * 1.0f; // Base width from grid size
                 float mapWorldHeight = hexMap.Size.y * 1.0f; // Base height from grid size
                 // Apply a scale multiplier if needed (e.g., currentWeatherState.emitterScaleMultiplier if added)
                 shapeScale = new Vector3(mapWorldWidth, mapWorldHeight, 1f); // Default Z scale for the box shape.
            }
            else // Fallback if logical map data is unavailable or invalid...
            {
                // Use the visual bounds of the map view renderer.
                shapeScale = new Vector3(mapBounds.size.x, mapBounds.size.y, 1f);
                 if(hexMap == null) {
                    // Debug.LogWarning("[WeatherSystem] AdjustEmitterToMap - HexMap not available. Using strategicMapView bounds for shape scale.", this);
                 } else {
                     Debug.LogWarning($"[WeatherSystem] AdjustEmitterToMap - HexMap size is zero or negative ({hexMap.Size}). Using strategicMapView bounds for shape scale.", this);
                 }
            }

            shapeModule.shapeType = ParticleSystemShapeType.Box; // Ensure shape type is Box.
            shapeModule.scale = shapeScale; // Apply the calculated scale.
            shapeModule.position = Vector3.zero; // Center the shape emission volume on the emitter's transform position.

            // --- Emitter Rotation ---
            // Set rotation based on desired emission direction.
            // Quaternion.identity aligns emitter axes with world axes. The Velocity Over Lifetime module
            // should primarily control the particle movement direction (e.g., negative Y for rain/snow).
            transform.rotation = Quaternion.identity; // Align emitter with world axes.
        }
    }

} // End namespace PK