﻿using DG.Tweening;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.Audio;

namespace PK
{
    public class AudioHandler : MonoBehaviour
    {
        private class AmbienceData
        {
            private AudioPreset _preset;
            private AudioSource _source;
            private Tweener _tweener;
            private int _priority;

            public AudioPreset Preset { get { return _preset; } }
            public int Priorty { get { return _priority; } }
            public bool IsPlaying { get { return _source.isPlaying; } }

            public AmbienceData(AudioPreset preset, AudioSource source, int priority)
            {
                _preset = preset;
                _source = source;
                _priority = priority;
            }

            public void FadeIn(float duration)
            {
                Debug.Log(_preset.name + " starts playing");
                if (_tweener != null && _tweener.IsActive() && _tweener.IsPlaying())
                {
                    _tweener.Kill();
                }
                if (!_source.isPlaying)
                {
                    _source.clip = _preset.Clips[Random.Range(0, _preset.Clips.Length)];
                    _source.Play();
                }
                _tweener = _source.DOVolume(_source.volume, 1f * _preset.VolumeMultiplier, duration).SetEase(Ease.Linear).SetUpdate(true);
            }

            public void FadeOut(float duration)
            {
                Debug.Log(_preset.name + " stops playing");
                if (_tweener != null && _tweener.IsActive() && _tweener.IsPlaying())
                {
                    _tweener.Kill();
                }
                _tweener = _source.DOVolume(_source.volume, 0f, duration).SetEase(Ease.Linear).SetUpdate(true).OnComplete(() =>
                {
                    if (_source.isPlaying)
                    {
                        _source.Pause();
                    }
                });
            }
        }

        private const float MUTE_AMBIENCE_TIME = 0.1f;
        private const float UNMUTE_AMBIENCE_TIME = 0.3f;

        private static AudioHandler _instance;

        public static AudioHandler Instance { get { return _instance; } }

        [SerializeField] private AudioMixer _mixer;
        [SerializeField] private AudioMixerGroup _ambienceMixerGroup;
        [SerializeField] private AudioMixerGroup _sfxMixerGroup;
        [SerializeField] private AudioMixerSnapshot _defaultSnapshot;
        [SerializeField] private AudioMixerSnapshot _muteAmbienceSnapshot;

        private Dictionary<AudioPreset, AmbienceData> _mutedAmbiencesDict = new();
        private List<AmbienceData> _currentAmbience = new();
        private Coroutine _muteAmbienceCoroutine;

        private void Awake()
        {
            _instance = this;
        }

        public bool HasPreset(string name)
        {
            return HexDatabase.Instance.GetAudioPreset(name) != null;
        }

        public void PlaySfx(AudioPreset preset, System.Action callback = null)
        {
            if (preset == null || preset.Clips == null || preset.Clips.Length == 0)
            {
                return;
            }

            AudioClip[] clips = preset.Clips;
            AudioClip clip = clips[clips.Length > 1 ? Random.Range(0, clips.Length) : 0];
            AudioSource source = AudioSourcePool.Get();
            source.outputAudioMixerGroup = _sfxMixerGroup;
            source.spatialBlend = 0f;
            source.volume = preset.VolumeMultiplier;
            source.clip = clip;
            source.Play();
            if (preset.MuteAmbience)
            {
                if (_muteAmbienceCoroutine != null)
                {
                    StopCoroutine(_muteAmbienceCoroutine);
                }
                _muteAmbienceCoroutine = StartCoroutine(AmbienceMuteCoroutine(clip.length));
            }
            DOVirtual.DelayedCall(clip.length, () =>
            {
                callback?.Invoke();
                AudioSourcePool.Release(source);
            }).SetUpdate(false);
        }

        public void PlaySfx(string name, System.Action callback = null)
        {
            AudioPreset preset = HexDatabase.Instance.GetAudioPreset(name);
            if (preset != null)
            {
                PlaySfx(preset, callback);
            }
        }

        public void PushAmbience(AudioPreset preset, int priority = 0)
        {
            if (preset != null)
            {
                foreach (AmbienceData data in _currentAmbience.Where((a) => a.Priorty <= priority))
                {
                    if (!_mutedAmbiencesDict.ContainsKey(data.Preset))
                    {
                        if (data.IsPlaying)
                        {
                            data.FadeOut(1f);
                        }
                        _mutedAmbiencesDict.Add(data.Preset, data);
                    }
                }

                if (_mutedAmbiencesDict.TryGetValue(preset, out AmbienceData newData))
                {
                    _mutedAmbiencesDict.Remove(preset);
                }
                else
                {
                    newData = new AmbienceData(preset, CreateAmbienceSource(), priority);
                }
                _currentAmbience.Add(newData);
                if (priority == GetMaxPriority())
                {
                    newData.FadeIn(1f);
                }
            }
        }

        public void PopAmbience(int priority = 0)
        {
            if (_currentAmbience.Count > 0)
            {
                AmbienceData lastData = _currentAmbience.LastOrDefault((a) => a.Priorty == priority);
                if (lastData != null)
                {
                    _currentAmbience.Remove(lastData);
                    if (lastData.IsPlaying)
                    {
                        lastData.FadeOut(1f);
                        if (_currentAmbience.Count > 0)
                        {
                            int maxPriority = GetMaxPriority();
                            AmbienceData nextData = _currentAmbience.Last((a) => a.Priorty == maxPriority);
                            nextData.FadeIn(1f);
                            _mutedAmbiencesDict.Remove(nextData.Preset);
                        }
                    }
                    _mutedAmbiencesDict.Add(lastData.Preset, lastData);
                }
            }
        }

        public void PlayAmbience(AudioPreset preset, int priority = 0)
        {
            if (preset == null)
            {
                foreach (AmbienceData data in _currentAmbience.Where((a) => a.Priorty == priority))
                {
                    if (!_mutedAmbiencesDict.ContainsKey(data.Preset))
                    {
                        if (data.IsPlaying)
                        {
                            data.FadeOut(1f);
                        }
                        _mutedAmbiencesDict.Add(data.Preset, data);
                    }
                }
                _currentAmbience.RemoveAll((a) => a.Priorty == priority);

                int maxPriority = GetMaxPriority();
                if (_currentAmbience.Count > 0 && priority > maxPriority)
                {
                    AmbienceData nextData = _currentAmbience.Last((a) => a.Priorty == maxPriority);
                    nextData.FadeIn(1f);
                    _mutedAmbiencesDict.Remove(nextData.Preset);
                }
            }
            else
            {
                if (IsPresetPlaying(preset))
                {
                    return;
                }

                foreach (AmbienceData data in _currentAmbience.Where((a) => a.Priorty == priority))
                {
                    if (!_mutedAmbiencesDict.ContainsKey(data.Preset))
                    {
                        if (data.IsPlaying)
                        {
                            data.FadeOut(1f);
                        }
                        _mutedAmbiencesDict.Add(data.Preset, data);
                    }
                }
                _currentAmbience.RemoveAll((a) => a.Priorty == priority);

                if (_mutedAmbiencesDict.TryGetValue(preset, out AmbienceData newData))
                {
                    _mutedAmbiencesDict.Remove(preset);
                }
                else
                {
                    newData = new AmbienceData(preset, CreateAmbienceSource(), priority);
                }
                _currentAmbience.Add(newData);
                if (priority >= GetMaxPriority())
                {
                    newData.FadeIn(1f);
                }
            }
        }

        public void PlayAmbience(string name, int priority = 0)
        {
            AudioPreset preset = HexDatabase.Instance.GetAudioPreset(name);
            PlayAmbience(preset, priority);
        }

        public void SetMasterVolume(float value)
        {
            SetVolume("MasterVolume", value);
        }

        private void SetVolume(string parameterName, float value)
        {
            if (_mixer != null)
            {
                _mixer.SetFloat(parameterName, Mathf.Log10(Mathf.Clamp(value, 0.0001f, 1f)) * 20);
            }
            else
            {
                Debug.LogWarning("AudioMixer is not assigned.");
            }
        }

        private bool IsPresetPlaying(AudioPreset preset)
        {
            if (_currentAmbience.Count > 0)
            {
                foreach (AmbienceData data in _currentAmbience)
                {
                    if (data.Preset == preset)
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        private AudioSource CreateAmbienceSource()
        {
            GameObject ambienceObject = new GameObject("Ambience");
            ambienceObject.transform.SetParent(transform, false);
            AudioSource source = ambienceObject.AddComponent<AudioSource>();
            source.outputAudioMixerGroup = _ambienceMixerGroup;
            source.loop = true;
            source.playOnAwake = false;
            source.volume = 0f;
            source.Pause();
            return source;
        }

        private IEnumerator AmbienceMuteCoroutine(float time)
        {
            _muteAmbienceSnapshot.TransitionTo(MUTE_AMBIENCE_TIME);
            yield return new WaitForSeconds(time);
            _defaultSnapshot.TransitionTo(UNMUTE_AMBIENCE_TIME);
        }

        private int GetMaxPriority()
        {
            return _currentAmbience.Count > 0 ? _currentAmbience.Max((a) => a.Priorty) : 0;
        }
    }
}
