﻿using UnityEditor;
using UnityEngine;
using System.Collections.Generic;
using System.Linq;

namespace PK
{
    [CustomEditor(typeof(HexTerrainOverrides))]
    public class HexTerrainOverridesEditor : Editor
    {
        private HexTerrainOverrides _target;
        private Vector2 _scrollPos;
        private int _selectedPairIndex;
        private HexTerrain _newSource;
        private HexTerrain _newTarget;

        private void OnEnable()
        {
            _target = (HexTerrainOverrides)target;
            RefreshTerrainPairs();
        }

        public override void OnInspectorGUI()
        {
            if (_target == null)
            {
                return;
            }

            serializedObject.Update();

            EditorGUILayout.Space();
            DrawCreateNewSection();

            EditorGUILayout.Space();
            DrawConfiguredPairs();

            serializedObject.ApplyModifiedProperties();
        }

        private void DrawCreateNewSection()
        {
            EditorGUILayout.LabelField("Create New Override Set", EditorStyles.boldLabel);

            _newSource = (HexTerrain)EditorGUILayout.ObjectField("Source Terrain", _newSource, typeof(HexTerrain), false);
            _newTarget = (HexTerrain)EditorGUILayout.ObjectField("Target Terrain", _newTarget, typeof(HexTerrain), false);

            var isValid = _newSource != null && _newTarget != null && _newSource != _newTarget;
            var exists = _target.TerrainOverrideExists(_newSource, _newTarget);

            EditorGUI.BeginDisabledGroup(!isValid || exists);
            if (GUILayout.Button("Create Override Set"))
            {
                Undo.RecordObject(_target, "Add Terrain Override");
                _target.AddTerrainOverride(_newSource, _newTarget);
                EditorUtility.SetDirty(_target);
                RefreshTerrainPairs();
            }
            EditorGUI.EndDisabledGroup();

            if (exists)
            {
                EditorGUILayout.HelpBox("This override set already exists", MessageType.Warning);
            }

        }

        private void DrawConfiguredPairs()
        {
            EditorGUILayout.LabelField("Configured Override Sets", EditorStyles.boldLabel);

            var pairs = _target.GetOverrideSets()
                .Where(s => s.sourceTerrain != null && s.targetTerrain != null)
                .ToList();

            if (pairs.Count == 0)
            {
                EditorGUILayout.HelpBox("No override sets configured", MessageType.Info);
                return;
            }

            var pairNames = pairs.Select(p => $"{p.sourceTerrain.name} → {p.targetTerrain.name}").ToArray();
            _selectedPairIndex = EditorGUILayout.Popup("Selected Set", _selectedPairIndex, pairNames);

            var selectedSet = pairs[_selectedPairIndex];
            EditorGUILayout.BeginVertical("Box");

            _scrollPos = EditorGUILayout.BeginScrollView(_scrollPos);
            foreach (var sourceTile in selectedSet.sourceTerrain.Tiles)
            {
                DrawTileOverrides(sourceTile, selectedSet);
            }
            EditorGUILayout.EndScrollView();

            if (GUILayout.Button("Remove This Set"))
            {
                Undo.RecordObject(_target, "Remove Terrain Override");
                _target.RemoveTerrainOverride(selectedSet.sourceTerrain, selectedSet.targetTerrain);
                EditorUtility.SetDirty(_target);
                RefreshTerrainPairs();
            }

            EditorGUILayout.EndVertical();
        }

        private void DrawTileOverrides(HexTile sourceTile, HexTerrainOverrides.TerrainOverrideSet set)
        {
            var overrideData = set.sourceTileOverrides.FirstOrDefault(o => o.sourceTile == sourceTile);
            var isConfigured = overrideData != null && overrideData.overrides.Count > 0;

            EditorGUILayout.BeginHorizontal();
            EditorGUILayout.LabelField(sourceTile.name, GUILayout.Width(150));

            if (isConfigured)
            {
                if (GUILayout.Button("Edit"))
                {
                    EditOverrideWeights(sourceTile, set.targetTerrain);
                }
            }
            else if (GUILayout.Button("Setup Overrides"))
            {
                Undo.RecordObject(_target, "Setup Tile Overrides");

                var weightDict = new Dictionary<HexTile, int>();

                if (set.targetTerrain.Tiles.Length > 0)
                {
                    int baseWeight = 100 / set.targetTerrain.Tiles.Length;
                    foreach (var targetTile in set.targetTerrain.Tiles)
                    {
                        weightDict[targetTile] = baseWeight;
                    }
                }

                _target.SetOverrideRules(sourceTile, set.targetTerrain, weightDict);
                EditorUtility.SetDirty(_target);
            }

            EditorGUILayout.EndHorizontal();
        }
        private void EditOverrideWeights(HexTile sourceTile, HexTerrain targetTerrain)
        {
            var window = EditorWindow.GetWindow<TileOverrideWeightWindow>(true,
                $"Edit {sourceTile.name} → {targetTerrain.name}", true);
            window.Initialize(sourceTile, targetTerrain, _target);
        }

        private class TileOverrideWeightWindow : EditorWindow
        {
            private HexTile _sourceTile;
            private HexTerrain _targetTerrain;
            private HexTerrainOverrides _overrides;
            private Vector2 _scrollPos;

            public void Initialize(HexTile sourceTile, HexTerrain targetTerrain, HexTerrainOverrides overrides)
            {
                _sourceTile = sourceTile;
                _targetTerrain = targetTerrain;
                _overrides = overrides;

                var set = _overrides.GetOrCreateOverrideSet(sourceTile.Terrain, targetTerrain);
                var sourceOverride = _overrides.GetOrCreateSourceTileOverride(set, sourceTile);

                AddMissingTargetTiles(sourceOverride);

                minSize = new Vector2(300, 400);
                Show();
            }

            private void AddMissingTargetTiles(HexTerrainOverrides.SourceTileOverride sourceOverride)
            {
                var existingTiles = new HashSet<HexTile>(sourceOverride.overrides.Select(o => o.targetTile));

                foreach (var targetTile in _targetTerrain.Tiles)
                {
                    if (targetTile != null && !existingTiles.Contains(targetTile))
                    {
                        sourceOverride.overrides.Add(new HexTerrainOverrides.TileOverrideOption
                        {
                            targetTile = targetTile,
                            weight = 0
                        });
                    }
                }

                _overrides.NormalizeWeights(sourceOverride);
            }

            void OnGUI()
            {
                if (_sourceTile == null || _targetTerrain == null)
                {
                    return;
                }

                EditorGUILayout.LabelField($"{_sourceTile.name} Overrides", EditorStyles.boldLabel);
                EditorGUILayout.LabelField($"Target Terrain: {_targetTerrain.name}", EditorStyles.miniBoldLabel);

                var set = _overrides.GetOrCreateOverrideSet(_sourceTile.Terrain, _targetTerrain);
                var sourceOverride = set.sourceTileOverrides
                    .FirstOrDefault(o => o.sourceTile == _sourceTile);

                if (sourceOverride == null)
                {
                    return;
                }

                _scrollPos = EditorGUILayout.BeginScrollView(_scrollPos);

                int totalWeight = 0;
                foreach (var overrideEntry in sourceOverride.overrides)
                {
                    EditorGUILayout.BeginHorizontal();

                    overrideEntry.targetTile = (HexTile)EditorGUILayout.ObjectField(
                        overrideEntry.targetTile,
                        typeof(HexTile),
                        false,
                        GUILayout.Width(150)
                    );

                    var newWeight = EditorGUILayout.IntSlider(
                        overrideEntry.weight,
                        0,
                        100
                    );

                    if (GUILayout.Button("×", GUILayout.Width(20)))
                    {
                        Undo.RecordObject(_overrides, "Remove Override");
                        sourceOverride.overrides.Remove(overrideEntry);
                        _overrides.NormalizeWeights(sourceOverride);
                        EditorUtility.SetDirty(_overrides);
                        GUIUtility.ExitGUI();
                    }

                    if (newWeight != overrideEntry.weight)
                    {
                        overrideEntry.weight = newWeight;
                        _overrides.NormalizeWeights(sourceOverride);
                    }

                    totalWeight += overrideEntry.weight;

                    if (newWeight != overrideEntry.weight)
                    {
                        overrideEntry.weight = newWeight;
                        _overrides.NormalizeWeights(sourceOverride);
                    }

                    EditorGUILayout.EndHorizontal();
                }

                EditorGUILayout.Space();
                EditorGUILayout.LabelField($"Total Weight: {totalWeight}%", EditorStyles.boldLabel);

                if (totalWeight != 100)
                {
                    EditorGUILayout.HelpBox("Total weight must equal 100%", MessageType.Warning);
                    if (GUILayout.Button("Auto-Fix Weights"))
                    {
                        _overrides.NormalizeWeights(sourceOverride);
                    }
                }

                EditorGUILayout.EndScrollView();

                if (GUILayout.Button("Add New Target Override"))
                {
                    Undo.RecordObject(_overrides, "Add Target Override");
                    sourceOverride.overrides.Add(new HexTerrainOverrides.TileOverrideOption
                    {
                        targetTile = _targetTerrain.Tiles.FirstOrDefault(),
                        weight = 0
                    });
                    _overrides.NormalizeWeights(sourceOverride);
                    EditorUtility.SetDirty(_overrides);
                }

                if (GUILayout.Button("Close"))
                {
                    EditorUtility.SetDirty(_overrides);
                    Close();
                }
            }
        }


        private void RefreshTerrainPairs()
        {
            var pairs = _target.GetOverrideSets()
                .Where(s => s.sourceTerrain != null && s.targetTerrain != null)
                .ToList();

            _selectedPairIndex = pairs.Count > 0 ? 0 : -1;
        }
    }
}
