﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    [CreateAssetMenu(menuName = "Hex/Config/Database")]
    public class HexDatabase : ScriptableObject
    {
        private static HexDatabase _instance;

        [SerializeField] private HexMask _mask;
        [SerializeField] private Sprite _proximityTriggerIcon;
        [SerializeField] private Sprite _spawnPointIcon;
        [SerializeField] private Sprite _mapZoneIcon;

        private HexTerrain[] _terrains;
        private HexRoad[] _roads;
        private HexObject[] _objects;
        private HexCreature[] _creatures;
        private HexHero[] _heroes;
        private HexPlayer[] _players;
        private AudioPreset[] _audioPresets;
        private Dictionary<ulong, HexTile> _tilesDict = new();
        private Dictionary<byte, HexRoad> _roadsDict = new();
        private Dictionary<ulong, HexObject> _objectsDict = new();
        private Dictionary<ulong, HexCreature> _creaturesDict = new();
        private Dictionary<ulong, HexHero> _heroesDict = new();
        private Dictionary<string, AudioPreset> _audioPresetsDict = new();
        private bool _isValid = false;

        private static HashSet<HexDatabase> _databases = new();

        public static HexDatabase Instance
        {
            get
            {
                if (_instance == null)
                {
                    HexDatabase[] databases = Resources.LoadAll<HexDatabase>("");
                    _instance = databases[0];
                }
                return _instance;
            }
        }

        public HexTerrain[] Terrains => _terrains;
        public HexObject[] Objects => _objects;
        public HexCreature[] Creatures => _creatures;
        public HexHero[] Heroes => _heroes;
        public HexPlayer[] Players => _players;
        public HexMask Mask => _mask;

        public Sprite ProximityTriggerIcon => _proximityTriggerIcon;
        public Sprite SpawnPointIcon => _spawnPointIcon;
        public Sprite MapZoneIcon => _mapZoneIcon;

        public HexTile[] Tiles 
        { 
            get 
            {
                if (!IsValid())
                {
                    Validate();
                }

                return _tilesDict.Values.ToArray(); 
            }
        }

        public HexRoad[] Roads
        {
            get
            {
                if (!IsValid())
                {
                    Validate();
                }

                return _roadsDict.Values.ToArray();
            }
        }


        private void OnEnable()
        {
            _databases.Add(this);
        }

        private void OnDisable()
        {
            _databases.Remove(this);
        }

        public HexTile GetTile(ulong uid)
        {
            if (!IsValid())
            {
                Validate();
            }

            if (_tilesDict.TryGetValue(uid, out HexTile tile))
            {
                return tile;
            }
            return null;
        }

        public HexObject GetObject(ulong uid)
        {
            if (!IsValid())
            {
                Validate();
            }

            if (_objectsDict.TryGetValue(uid, out HexObject @object))
            {
                return @object;
            }
            return null;
        }

        public HexCreature GetCreature(ulong uid)
        {
            if (!IsValid())
            {
                Validate();
            }

            if (_creaturesDict.TryGetValue(uid, out HexCreature creature))
            {
                return creature;
            }
            return null;
        }

        public HexHero GetHero(ulong uid)
        {
            if (!IsValid())
            {
                Validate();
            }

            if (_heroesDict.TryGetValue(uid, out HexHero hero))
            {
                return hero;
            }
            return null;
        }

        public AudioPreset GetAudioPreset(string name)
        {
            if (!IsValid())
            {
                Validate();
            }

            if (_audioPresetsDict.TryGetValue(name, out AudioPreset preset))
            {
                return preset;
            }
            return null;
        }

        public HexRoad GetRoad(byte type)
        {
            if (!IsValid())
            {
                Validate();
            }

            if (_roadsDict.TryGetValue(type, out HexRoad road))
            {
                return road;
            }
            return null;
        }

        public bool TryValidate()
        {
            if (!_isValid)
            {
                Validate();
                return true;
            }
            return false;
        }

        public static void Invalidate()
        {
            foreach (HexDatabase database in _databases)
            {
                database._isValid = false;
            }
        }

        private void Validate()
        {
            _tilesDict.Clear();
            _roadsDict.Clear();
            _objectsDict.Clear();
            _terrains = Resources.LoadAll<HexTerrain>("");
            _roads = Resources.LoadAll<HexRoad>("");
            _objects = Resources.LoadAll<HexObject>("");
            _creatures = Resources.LoadAll<HexCreature>("");
            _heroes = Resources.LoadAll<HexHero>("");
            _players = Resources.LoadAll<HexPlayer>("");
            _audioPresets = Resources.LoadAll<AudioPreset>("");
            foreach (HexTerrain terrain in _terrains)
            {
                foreach (HexTile tile in terrain.Tiles)
                {
                    _tilesDict[tile.Uid] = tile;
                }
            }
            foreach (HexRoad road in _roads)
            {
                _roadsDict[road.Type] = road;
            }
            foreach (HexObject @object in _objects)
            {
                _objectsDict[@object.Uid] = @object;
            }
            foreach (HexCreature creature in _creatures)
            {
                _creaturesDict[creature.Uid] = creature;
            }
            foreach (HexHero hero in _heroes)
            {
                _heroesDict[hero.Uid] = hero;
            }
            foreach (AudioPreset preset in _audioPresets)
            {
                _audioPresetsDict[preset.name] = preset;
            }
            _isValid = true;
        }

        private bool IsValid()
        {
            return _isValid && _tilesDict.Count > 0;
        }
    }
}
