﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    [CreateAssetMenu(menuName = "Hex/Config/HexTerrainOverrides")]
    public class HexTerrainOverrides : ScriptableObject
    {
        private static HexTerrainOverrides _instance;

        [System.Serializable]
        public class TileOverrideOption
        {
            public HexTile targetTile;
            [Range(0, 100)]
            public int weight;
        }

        [System.Serializable]
        public class SourceTileOverride
        {
            public HexTile sourceTile;
            public List<TileOverrideOption> overrides = new List<TileOverrideOption>();
        }

        [System.Serializable]
        public class TerrainOverrideSet
        {
            public HexTerrain sourceTerrain;
            public HexTerrain targetTerrain;
            public List<SourceTileOverride> sourceTileOverrides = new List<SourceTileOverride>();
        }

        [SerializeField] private List<TerrainOverrideSet> _overrideSets = new List<TerrainOverrideSet>();

        public static HexTerrainOverrides Instance
        {
            get
            {
                if (_instance == null)
                {
                    _instance = Resources.LoadAll<HexTerrainOverrides>("")[0];
                }
                return _instance;
            }
        }

        public List<TerrainOverrideSet> GetOverrideSets() => new List<TerrainOverrideSet>(_overrideSets);

        public void AddDefaultOverride(SourceTileOverride sourceOverride, HexTerrain targetTerrain)
        {
            if (targetTerrain.Tiles.Length == 0)
            {
                return;
            }
            sourceOverride.overrides.Clear();

            foreach (var tile in targetTerrain.Tiles)
            {
                sourceOverride.overrides.Add(new TileOverrideOption
                {
                    targetTile = tile,
                    weight = 100 / targetTerrain.Tiles.Length
                });
            }

            NormalizeWeights(sourceOverride);
        }

        public SourceTileOverride GetOrCreateSourceTileOverride(TerrainOverrideSet set, HexTile sourceTile)
        {
            if (set == null || sourceTile == null)
            {
                return null;
            }

            var sourceOverride = set.sourceTileOverrides.Find(o => o.sourceTile == sourceTile);
            if (sourceOverride == null)
            {
                sourceOverride = new SourceTileOverride { sourceTile = sourceTile };
                set.sourceTileOverrides.Add(sourceOverride);
                AddDefaultOverride(sourceOverride, set.targetTerrain);
            }
            return sourceOverride;
        }


        public void NormalizeWeights(SourceTileOverride sourceOverride)
        {
            if (sourceOverride.overrides.Count == 0)
            {
                return;
            }

            sourceOverride.overrides.RemoveAll(o => o.targetTile == null);

            int totalWeight = sourceOverride.overrides.Sum(o => o.weight);
            if (totalWeight == 0)
            {
                int weight = 100 / sourceOverride.overrides.Count;
                foreach (var entry in sourceOverride.overrides)
                {
                    entry.weight = weight;
                }
                totalWeight = sourceOverride.overrides.Sum(o => o.weight);
            }

            float ratio = 100f / totalWeight;
            foreach (var entry in sourceOverride.overrides)
            {
                entry.weight = Mathf.RoundToInt(entry.weight * ratio);
            }

            int remainder = 100 - sourceOverride.overrides.Sum(o => o.weight);
            if (remainder != 0)
            {
                sourceOverride.overrides[0].weight += remainder;
            }
        }

        public void RemoveTerrainOverride(HexTerrain source, HexTerrain target)
        {
            _overrideSets.RemoveAll(set =>
                set.sourceTerrain == source &&
                set.targetTerrain == target
            );

            foreach (var set in _overrideSets)
            {
                set.sourceTileOverrides.RemoveAll(overrideObj =>
                    overrideObj.sourceTile.Terrain == source && set.targetTerrain == target
                );
            }
        }

        public Dictionary<HexTile, Dictionary<HexTile, int>> GetOverrideRules(HexTerrain source, HexTerrain target)
        {
            var results = new Dictionary<HexTile, Dictionary<HexTile, int>>();
            var set = _overrideSets.Find(s => s.sourceTerrain == source && s.targetTerrain == target);

            if (set != null)
            {
                foreach (var sourceOverride in set.sourceTileOverrides)
                {
                    var tileRules = new Dictionary<HexTile, int>();
                    foreach (var overrideEntry in sourceOverride.overrides)
                    {
                        if (overrideEntry?.targetTile != null)
                        {
                            tileRules[overrideEntry.targetTile] = overrideEntry.weight;
                        }
                    }

                    if (tileRules.Count > 0)
                    {
                        results[sourceOverride.sourceTile] = tileRules;
                    }
                }
            }

            return results;
        }


        public void AddTerrainOverride(HexTerrain source, HexTerrain target)
        {
            if (!TerrainOverrideExists(source, target))
            {
                _overrideSets.Add(new TerrainOverrideSet
                {
                    sourceTerrain = source,
                    targetTerrain = target
                });
            }
        }

        public bool TerrainOverrideExists(HexTerrain source, HexTerrain target)
        {
            return _overrideSets.Exists(s =>
                s.sourceTerrain == source &&
                s.targetTerrain == target
            );
        }

        public void SetOverrideRules(HexTile sourceTile, HexTerrain targetTerrain, Dictionary<HexTile, int> rules)
        {
            var set = GetOrCreateOverrideSet(sourceTile.Terrain, targetTerrain);
            var sourceOverride = GetOrCreateSourceTileOverride(set, sourceTile);

            sourceOverride.overrides.Clear();
            foreach (var rule in rules)
            {
                sourceOverride.overrides.Add(new TileOverrideOption
                {
                    targetTile = rule.Key,
                    weight = rule.Value
                });
            }

            NormalizeWeights(sourceOverride);
        }

        public TerrainOverrideSet GetOrCreateOverrideSet(HexTerrain source, HexTerrain target)
        {
            var set = _overrideSets.Find(s =>
                s.sourceTerrain == source &&
                s.targetTerrain == target
            );

            if (set == null)
            {
                set = new TerrainOverrideSet
                {
                    sourceTerrain = source,
                    targetTerrain = target
                };
                _overrideSets.Add(set);
            }
            return set;
        }

        public HexTile GetOverrideTile(HexTile sourceTile, HexTerrain targetTerrain)
        {
            var set = _overrideSets.Find(s =>
                s.sourceTerrain == sourceTile.Terrain &&
                s.targetTerrain == targetTerrain
            );

            if (set == null)
            {
                return sourceTile;
            }

            var sourceOverride = set.sourceTileOverrides.Find(o => o.sourceTile == sourceTile);
            if (sourceOverride == null || sourceOverride.overrides.Count == 0)
            {
                return sourceTile;
            }

            int totalWeight = sourceOverride.overrides.Sum(o => o.weight);
            int random = Random.Range(0, totalWeight);

            foreach (var entry in sourceOverride.overrides)
            {
                if (random < entry.weight)
                {
                    return entry.targetTile;
                }
                random -= entry.weight;
            }

            return sourceTile;
        }
    }
}
