﻿using TMPro;
using UnityEditor;
using UnityEngine;
using UnityEngine.TextCore;

namespace PK
{
    public static class CreateBitmapFont
    {
        [MenuItem("Assets/Create/TextMeshPro/Bitmap Font", false, 105)]
        public static void Create()
        {
            Object target = Selection.activeObject;

            string path = AssetDatabase.GetAssetPath(target);
            if (target == null || target.GetType() != typeof(TextAsset) || !path.Contains(".fnt"))
            {
                Debug.LogWarning("A Font file must first be selected in order to create a Font Asset.");
                return;
            }
            TextAsset textAsset = target as TextAsset;

            string atlasPath = path.Replace(".fnt", ".png");
            Texture2D atlasTexture = AssetDatabase.LoadAssetAtPath<Texture2D>(atlasPath);
            if (atlasTexture == null)
            {
                Debug.LogWarning("Can't find atlas texture.");
                return;
            }

            string fontAssetPath = path.Replace(".fnt", ".asset");
            TMP_FontAsset fontAsset = AssetDatabase.LoadAssetAtPath<TMP_FontAsset>(fontAssetPath);
            if (fontAsset == null)
            {
                fontAsset = ScriptableObject.CreateInstance<TMP_FontAsset>();
                Texture2D[] atlasTextures = new Texture2D[1];
                fontAsset.atlasTextures = atlasTextures;
                fontAsset.atlasPopulationMode = AtlasPopulationMode.Static;

                AssetDatabase.CreateAsset(fontAsset, fontAssetPath);
                AssetDatabase.SaveAssets();
            }
            else
            {
                fontAsset.characterTable.Clear();
                fontAsset.glyphTable.Clear();
            }

            Material atlasMaterial = AssetDatabase.LoadAssetAtPath<Material>(fontAssetPath);
            Shader shader = Shader.Find("TextMeshPro/Distance Field");
            if (atlasMaterial == null)
            {
                atlasMaterial = new Material(shader);
                fontAsset.material = atlasMaterial;

                AssetDatabase.AddObjectToAsset(atlasMaterial, fontAssetPath);
                AssetDatabase.SaveAssets();
            }
            atlasMaterial.shader = shader;
            atlasMaterial.SetTexture(ShaderUtilities.ID_MainTex, atlasTexture);
            atlasMaterial.name = target.name;

            System.Type type = typeof(TMP_FontAsset);
            type.GetField("m_Version", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic).SetValue(fontAsset, "1.1.0");
            type.GetField("m_AtlasWidth", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic).SetValue(fontAsset, atlasTexture.width);
            type.GetField("m_AtlasHeight", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic).SetValue(fontAsset, atlasTexture.height);

            string text = textAsset.text;
            FntParser fnt = FntParser.GetFntParse(ref text);

            FaceInfo newFaceInfo = fontAsset.faceInfo;
            newFaceInfo.baseline = 0;
            newFaceInfo.lineHeight = fnt.lineHeight;
            newFaceInfo.ascentLine = fnt.lineBaseHeight;
            newFaceInfo.descentLine = -(fnt.lineHeight - fnt.lineBaseHeight);
            newFaceInfo.pointSize = fnt.fontSize;
            type.GetProperty("faceInfo").SetValue(fontAsset, newFaceInfo);

            fontAsset.atlasTextures[0] = atlasTexture;

            for (int i = 0; i < fnt.charInfos.Length; i++)
            {
                CharacterInfo charInfo = fnt.charInfos[i];
                GlyphMetrics metrics = new GlyphMetrics(charInfo.glyphWidth, charInfo.glyphHeight, charInfo.bearing, charInfo.maxY, charInfo.advance);
                RawCharacterInfo rawCharInfo = fnt.rawCharInfos[i];
                GlyphRect rect = new GlyphRect(rawCharInfo.X, atlasTexture.height - rawCharInfo.Y - rawCharInfo.Height, rawCharInfo.Width, rawCharInfo.Height);
                Glyph glyph = new Glyph((uint)i, metrics, rect);
                TMP_Character character = new TMP_Character((uint)charInfo.index, glyph);
                fontAsset.glyphTable.Add(glyph);
                fontAsset.characterTable.Add(character);
            }

            EditorUtility.SetDirty(fontAsset);
            AssetDatabase.SaveAssets();
        }
    }
}
