﻿
using System.Linq;
using UnityEngine;

namespace PK.Strategic
{
    public partial class GameController
    {
        private class TryInteractCommand : Command
        {
            private ulong _heroId;
            private ulong _entityId;

            public TryInteractCommand(ulong heroId, ulong entityId)
            {
                _heroId = heroId;
                _entityId = entityId;
            }

            protected override CommandResult ExecuteBeforeChilds(IContext context)
            {
                HexHeroModel hero = context.MapData.Map.GetHero(_heroId);
                HexEntityModel entity = context.MapData.Map.GetEntity(_entityId);

                if (hero.Position != entity.Position)
                {
                    int mask = context.MapData.Map.GetInteractionMask(entity.Position.x, entity.Position.y);
                    bool[] tilesData = HexObjectMaskHelper.GetTilesData(mask, true);
                    int sideIndex = System.Array.IndexOf(HexHelper.AXIAL_DIRECTIONS, HexHelper.OddToAxial(entity.Position) - HexHelper.OddToAxial(hero.Position));
                    if (sideIndex == -1 || !tilesData[(sideIndex + 3) % 6]) // Flip side
                    {
                        return CommandResult.Complete;
                    }
                }

                if (hero != null && entity != null && entity is HexInteractableEntityModel interactableEntity && entity.Enabled)
                {
                    if (interactableEntity.Interaction is ActionInteraction actionInteraction)
                    {
                        ActionGraph graph = actionInteraction.Graph;
                        if (graph != null)
                        {
                            PushSubCommand(new ExecuteActionGraphCommand(graph, hero.Player, entity, hero));
                        }
                    }
                    else if (interactableEntity.Interaction is CapturableBuildingInteraction capturableBuildingInteraction)
                    {
                        capturableBuildingInteraction.Player = hero.Player;
                        context.EventManager.Get<MapEvents>().CallOnEntityChange(entity);
                    }
                    else if (interactableEntity.Interaction is StartBattleInteraction startBattleInteraction)
                    {
                        HexMap map = startBattleInteraction.Map;
                        if (map != null && !map.IsStrategic)
                        {
                            context.EventManager.Get<StrategicEvents>().CallOnStartBattle(map, hero.Id, interactableEntity.Id, startBattleInteraction.Enemies.Select((e) => new HexCreatureSquadModel(e.Creature.Uid, 0, Vector2Int.zero, hero.Player, Mathf.Min(e.Count, e.Creature.SoldiersPerUnit), e.Creature.Health)).ToArray(), true, Player.Red, (BattleData result) =>
                            {
                                if (result.Winner == hero.Player)
                                {
                                    if (context.MapData.Map.GetEntity(result.EnemyId) is HexCreatureModel creature)
                                    {
                                        creature.Enabled = false;
                                        context.EventManager.Get<MapEvents>().CallOnEntitySetActive(creature.Id, false);
                                        context.MapData.RefreshObstacleMask();
                                        context.EventManager.Get<MapDataEvents>().CallOnRefreshObstacleMask();
                                    }
                                }
                            });
                        }
                    }
                    context.EventManager.Get<StrategicEvents>().CallOnHeroInteract(hero.Id, interactableEntity.Id);
                }
                return CommandResult.Complete;
            }
        }
    }
}
