﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Strategic
{
    public partial class GameController : BaseController<GameController.IGameContext>
    {
        private Savefile _savefile;
        private EventManager _eventManager = new();
        private ControllerLocker _locker = new();
        private GameContext _context = new();

        public IEventManager EventManager { get { return _eventManager; } }
        public IControllerLocker Locker { get { return _locker; } }
        public Player SelfPlayer { get { return Player.Red; } }
        public Player CurrentPlayer { get { return Context.Player; } }
        private IGameContext Context { get { return _context; } }

        public GameController()
        {
            RegisterEvents();
        }

        public void Initialize(Savefile savefile)
        {
            _savefile = savefile;
            _context.Init(savefile, 1);
            InitContext(savefile.StrategicMap, savefile, _eventManager, _locker, _context);
        }

        public bool Simulate()
        {
            return Execute();
        }

        public void GetAllSquads(ulong id, List<ICreatureSquadModelForView> resultSquads)
        {
            HexHeroModel hero = _savefile.StrategicMap.Map.GetEnumerable<HexHeroModel>().Where((h) => h.Id == id).FirstOrDefault();
            if (hero != null)
            {
                Savefile.HeroData data = _savefile.GetHeroData(hero);
                if (data.Squads != null)
                {
                    resultSquads.AddRange(data.Squads.Select((s) => s as ICreatureSquadModelForView));
                }
            }
        }

        public (IReadOnlyList<(Vector2Int, int)> path, int stepsLeft) GetHeroPath(ulong id, Vector2Int target)
        {
            HexHeroModel hero = _savefile.StrategicMap.Map.GetEnumerable<HexHeroModel>().Where((h) => h.Id == id).FirstOrDefault();
            if (hero != null)
            {
                List<(Vector2Int, int)> path = PathfindingHelper.FindPath(hero.Position, target, _savefile.StrategicMap.ObstacleMask, _savefile.StrategicMap.MovementCostMask, _savefile.StrategicMap.Map.InteractionMask, _savefile.StrategicMap.Map.Size, new List<Vector2Int>() { target });
                return (path, hero.StepsLeft);
            }
            return default;
        }

        public void GetReachableByHeroArea(ulong id, List<Vector2Int> resultPositions)
        {
            if (_savefile.StrategicMap.Map.GetEntity(id) is HexHeroModel hero)
            {
                PathfindingHelper.FindArea(hero.Position, hero.StepsLeft, _savefile.StrategicMap.ObstacleMask, _savefile.StrategicMap.MovementCostMask, _savefile.StrategicMap.Map.InteractionMask, _savefile.StrategicMap.Map.Size, resultPositions);
            }
        }

        public IHeroModelForView GetAnyPlayerHero()
        {
            return _savefile.StrategicMap.Map.GetEnumerable<HexHeroModel>().FirstOrDefault((h) => h.Player == SelfPlayer);
        }

        public IHeroModelForView TrySelectHero(Vector2Int position)
        {
            HexHeroModel hero = _savefile.StrategicMap.Map.GetEnumerable<HexHeroModel>().Where((h) => h.Player == CurrentPlayer && h.Position == position).FirstOrDefault();
            if (hero != null)
            {
                return hero;
            }
            return null;
        }

        public bool MoveHero(ulong id, Vector2Int target)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new WalkHeroCommand(id, target));
                turn.Push(new TryToEndTurnCommand());
            });
        }

        public bool EndPlayerTurn()
        {
            if (CurrentPlayer != SelfPlayer)
            {
                return false;
            }

            // TODO AI turns
            return StartNewTurn();
        }

        public bool StartGame()
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new StartGameCommand());
            });
        }

        public bool StartNewTurn()
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new TrySwitchDayCommand());
                turn.Push(new SwitchTurnCommand());
            });
        }

        public void ContinueGame()
        {
            BaseContext.EventManager.Get<StrategicEvents>().CallOnStartGame();
            BaseContext.EventManager.Get<StrategicEvents>().CallOnUpdateFogOfWar(_savefile.StrategicMap.FogOfWarMask, _savefile.StrategicMap.Map.Size);
        }

        private void RegisterEvents()
        {
            _eventManager.Register<IMapEvents, MapEvents>(new MapEvents());
            _eventManager.Register<IMapDataEvents, MapDataEvents>(new MapDataEvents());
            _eventManager.Register<IStrategicEvents, StrategicEvents>(new StrategicEvents());
            _eventManager.Register<IInteractionEvents, InteractionEvents>(new InteractionEvents());
        }
    }
}
