﻿using DG.Tweening;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Strategic
{
    public class StrategicAudioHandler : MonoBehaviour, IEventReceiver
    {
        private const float SWITCH_AMBIENCE_DELAY = 2f;

        [SerializeField] private StrategicMapView _view;

        private string _terrainAmbience;
        private Tween _switchAmbienceTween;

        void IEventReceiver.AddEvents()
        {
            IStrategicEvents events = StrategicGameMediator.Instance.EventManager.Get<IStrategicEvents>();
            events.OnStartBattle += OnStartBattle;
            events.OnStartDay += OnStartDay;
            events.OnHeroWalk += OnHeroWalk;
            events.OnHeroEndWalk += OnHeroEndWalk;
            events.OnHeroInteract += OnHeroInteract;

            IMapEvents mapEvents = StrategicGameMediator.Instance.EventManager.Get<IMapEvents>();
            mapEvents.OnEntitySetActive += OnEntitySetActive;

            _view.OnUpdateSelectedHero += OnUpdateSelectedHero;
        }

        void IEventReceiver.RemoveEvents()
        {
            IStrategicEvents events = StrategicGameMediator.Instance.EventManager.Get<IStrategicEvents>();
            events.OnStartBattle -= OnStartBattle;
            events.OnStartDay -= OnStartDay;
            events.OnHeroWalk -= OnHeroWalk;
            events.OnHeroEndWalk -= OnHeroEndWalk;
            events.OnHeroInteract -= OnHeroInteract;

            IMapEvents mapEvents = StrategicGameMediator.Instance.EventManager.Get<IMapEvents>();
            mapEvents.OnEntitySetActive -= OnEntitySetActive;

            _view.OnUpdateSelectedHero -= OnUpdateSelectedHero;
        }

        private void OnStartBattle(HexMap map, ulong heroId, ulong enemyId, HexCreatureSquadModel[] enemySquads, bool canRetry, Player beginer, System.Action<BattleData> callback)
        {
            _terrainAmbience = null;
            if (_switchAmbienceTween != null && _switchAmbienceTween.IsActive() && _switchAmbienceTween.IsPlaying())
            {
                _switchAmbienceTween.Kill();
            }
        }

        private void OnStartDay(int day)
        {
            if (day > 1)
            {
                string sfxName;
                if (day % 28 == 0)
                {
                    sfxName = "NewMonthSfx";
                }
                else if (day % 7 == 0)
                {
                    sfxName = "NewWeekSfx";
                }
                else
                {
                    sfxName = "NewDaySfx";
                }
                AudioHandler.Instance.PlaySfx(sfxName);
            }
        }

        private void OnHeroWalk(ulong id, Vector2Int start, Vector2Int end)
        {
            if (id == _view.SelectedHero.Id)
            {
                HexTerrain terrain = _view.TerrainView.GetTerrain(end);
                string sfxName = $"HeroMove{terrain.name}Sfx";
                if (!AudioHandler.Instance.HasPreset(sfxName))
                {
                    sfxName = "HeroMoveDefaultSfx";
                }
                AudioHandler.Instance.PlaySfx(sfxName);
            }
        }

        private void OnHeroEndWalk(ulong id, Player player)
        {
            if (id == _view.SelectedHero.Id)
            {
                UpdateTerrainAmbience(_view.SelectedHero.Position);
            }
        }

        private void OnHeroInteract(ulong heroId, ulong id)
        {
            if (_view.Entities.TryGetValue(id, out HexEntityView view) && view is HexObjectView)
            {
                HexObject @object = HexDatabase.Instance.GetObject(view.Uid);
                if (@object.InteractionSfx != null)
                {
                    AudioHandler.Instance.PlaySfx(@object.InteractionSfx);
                }
            }
        }

        private void OnEntitySetActive(ulong id, bool active)
        {
            if (!active && _view.Entities.TryGetValue(id, out HexEntityView view) && view is HexObjectView)
            {
                HexObject @object = HexDatabase.Instance.GetObject(view.Uid);
                if (@object.DisableSfx != null)
                {
                    AudioHandler.Instance.PlaySfx(@object.DisableSfx);
                }
            }
        }

        private void OnUpdateSelectedHero(IHeroModelForView hero)
        {
            UpdateTerrainAmbience(hero.Position);
        }

        private void UpdateTerrainAmbience(Vector2Int position)
        {
            List<HexTerrain> nearbyTerrains = new();
            Vector3Int cubePosition = HexHelper.OddToCube(position);
            int radius = 2;
            for (int i = -radius; i <= radius; i++)
            {
                for (int j = -radius; j <= radius; j++)
                {
                    for (int k = -radius; k <= radius; k++)
                    {
                        if (i + j + k == 0)
                        {
                            Vector2Int nextPosition = HexHelper.CubeToOdd(cubePosition + new Vector3Int(i, j, k));
                            if (_view.Map.Model.IsInBounds(nextPosition))
                            {
                                HexTerrain nextTerrain = _view.TerrainView.GetTerrain(nextPosition);
                                if (nextTerrain != null)
                                {
                                    nearbyTerrains.Add(nextTerrain);
                                }
                            }
                        }
                    }
                }
            }
            HexTerrain terrain = nearbyTerrains.GroupBy((t) => t).OrderByDescending((g) => g.Count()).First().Key;
            string terainAmbience = $"{terrain.name}Ambience";

            if (terainAmbience != _terrainAmbience)
            {
                if (_switchAmbienceTween != null && _switchAmbienceTween.IsActive() && _switchAmbienceTween.IsPlaying())
                {
                    _switchAmbienceTween.Kill();
                }
                _terrainAmbience = terainAmbience;
                _switchAmbienceTween = DOVirtual.DelayedCall(SWITCH_AMBIENCE_DELAY, () =>
                {
                    AudioHandler.Instance.PlayAmbience(terainAmbience);
                }).SetUpdate(false);
            }
        }
    }
}
