﻿using DG.Tweening;
using System;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Animations;
using UnityEngine.EventSystems;

namespace PK.Strategic
{
    public partial class StrategicMapView
    {
        private static string LOCK = "StrategicMapViewLock";
        
        [SerializeField] private Camera _camera;

        [Space]
        [Header("Player path")]
        [SerializeField] private Sprite _greenDot;
        [SerializeField] private Sprite _greenCross;
        [SerializeField] private Sprite _greenAttack;
        [SerializeField] private Sprite _greenHorse;
        [SerializeField] private Sprite _greenHorseMove;

        [SerializeField] private Sprite _redDot;
        [SerializeField] private Sprite _redCross;
        [SerializeField] private Sprite _redAttack;
        [SerializeField] private Sprite _redHorse;
        [SerializeField] private Sprite _redHorseMove;

        private SelectionLogic _selectionLogic;
        private IHeroModelForView _selectedHero;
        private List<Vector2Int> _reachableArea = new();
        private List<IObjectModelForView> _reachableObjects = new();
        private IObjectModelForView _hoveredReachableObject;
        private ulong _walkingHeroId;

        public IHeroModelForView SelectedHero { get { return _selectedHero; } }
        public event Action<IHeroModelForView> OnUpdateSelectedHero;

        private void AddPlayerEvents()
        {
            IStrategicEvents events = StrategicGameMediator.Instance.EventManager.Get<IStrategicEvents>();
            events.OnStartGame += OnStartGame;
            events.OnStartTurn += OnStartTurn;
            events.OnHeroStartWalk += OnHeroStartWalk;
            events.OnHeroWalk += OnHeroWalk;
            events.OnHeroEndWalk += OnHeroEndWalk;
        }

        private void RemovePlayerEvents()
        {
            IStrategicEvents events = StrategicGameMediator.Instance.EventManager.Get<IStrategicEvents>();
            events.OnStartGame -= OnStartGame;
            events.OnStartTurn -= OnStartTurn;
            events.OnHeroStartWalk -= OnHeroStartWalk;
            events.OnHeroWalk -= OnHeroWalk;
            events.OnHeroEndWalk -= OnHeroEndWalk;

            ClearReachableArea();
        }

        private void UpdateSelection()
        {
            DrawReachableArea();

            bool canInteract = StrategicGameMediator.Instance.CurrentPlayer == StrategicGameMediator.Instance.SelfPlayer && !InputLock.IsLock;
            bool canInteractWithMap = canInteract && !EventSystem.current.IsPointerOverGameObject();
            
            if (canInteract)
            {
                if (_reachableArea.Count == 0)
                {
                    FillReachableArea();
                }
            }
            else
            {
                if (_reachableArea.Count > 0)
                {
                    ClearReachableArea();
                }
            }
            
            if (canInteractWithMap)
            {
                Vector3 cursorWorldPosition = GetCursorWorldPosition();
                (HexContent content, Vector2Int position) = GetContent(cursorWorldPosition);

                IHeroModelForView playerHero = StrategicGameMediator.Instance.TrySelectHero(position);
                if (playerHero == null || playerHero == _selectedHero)
                {
                    bool reachable = false;
                    if (_reachableArea.Contains(position) || content.Type == HexContent.ContentType.Interactable && content.Id == _hoveredReachableObject.Id)
                    {
                        reachable = true;
                    }

                    Sprite cursorSprite = GetCursorSprite(content, reachable);
                    if (cursorSprite != null)
                    {
                        ImmediateSpriteRenderer.DrawSprite(cursorWorldPosition, cursorSprite, SortingOrder.CURSOR);
                    }
                }

                // Avoid double click
                if (_selectionLogic != null)
                {
                    _selectionLogic.Update();
                    if (_selectionLogic == null)
                    {
                        return;
                    }
                }

                if (_selectionLogic == null && _selectedHero != null)
                {
                    if (ControlHandler.Instance.WasLeftClicked && ClickCooldown.TryClick())
                    {
                        if (playerHero != null && playerHero != _selectedHero)
                        {
                            SelectHero(playerHero);
                        }
                        else if (content.Type != HexContent.ContentType.Obstacle && content.Type != HexContent.ContentType.Hidden)
                        {
                            if (position != _selectedHero.Position)
                            {
                                _selectionLogic = new SelectPathLogic(this, position);
                            }
                            else if (content.Type == HexContent.ContentType.Interactable)
                            {
                                StrategicGameMediator.Instance.MoveHero(_selectedHero.Id, position);
                            }
                        }
                    }

                    if (Input.GetKeyDown(KeyCode.Space))
                    {
                        content = StrategicGameMediator.Instance.GetContent(_selectedHero.Position);
                        if (content.Type == HexContent.ContentType.Interactable)
                        {
                            StrategicGameMediator.Instance.MoveHero(_selectedHero.Id, _selectedHero.Position);
                        }
                    }
                }
            }
            else
            {
                // Stop hero
                if (Input.anyKeyDown && ClickCooldown.TryClick(2.0f) && _selectedHero != null && _selectedHero.Id == _walkingHeroId)
                {
                    _selectedHero.StopRequested = true;
                }
            }
        }

        private void UpdateCamera()
        {
            if (!InputLock.IsLock)
            {
                Vector2 delta = Vector2.zero;
                float screenBorderThreshold = 3f;
                Vector3 mousePosition = Input.mousePosition;

                if (Input.GetKey(KeyCode.D) || Input.GetKey(KeyCode.RightArrow) || mousePosition.x >= Screen.width - screenBorderThreshold)
                {
                    delta.x = 1;
                }
                else if (Input.GetKey(KeyCode.A) || Input.GetKey(KeyCode.LeftArrow) || mousePosition.x <= screenBorderThreshold)
                {
                    delta.x = -1;
                }

                if (Input.GetKey(KeyCode.W) || Input.GetKey(KeyCode.UpArrow) || mousePosition.y >= Screen.height - screenBorderThreshold)
                {
                    delta.y = 1;
                }
                else if (Input.GetKey(KeyCode.S) || Input.GetKey(KeyCode.DownArrow) || mousePosition.y <= screenBorderThreshold)
                {
                    delta.y = -1;
                }

                bool needsClamping = delta.sqrMagnitude > 0;
                if (needsClamping)
                {
                    Vector3 targetPosition = _camera.transform.position;
                    if (delta.sqrMagnitude > 0)
                    {
                        targetPosition += (Vector3)delta.normalized * Time.deltaTime * Preferences.MapScrollingSpeed;
                    }

                    Rect mapBounds = HexHelper.GetMapWorldBounds(_map.Model.Width, _map.Model.Height);
                    targetPosition.x = Mathf.Clamp(targetPosition.x, mapBounds.xMin, mapBounds.xMax);
                    targetPosition.y = Mathf.Clamp(targetPosition.y, mapBounds.yMin, mapBounds.yMax);
                    _camera.transform.position = targetPosition;
                }
            }
        }

        private Vector3 GetCursorWorldPosition()
        {
            Vector3 cursorWorldPosition = _camera.ScreenToWorldPoint(Input.mousePosition);
            cursorWorldPosition.z = 0;
            return cursorWorldPosition;
        }

        private (HexContent, Vector2Int) GetContent(Vector3 cursorWorldPosition)
        {
            Vector2Int position = HexHelper.GetHexPosition(cursorWorldPosition);
            return (StrategicGameMediator.Instance.GetContent(position), position);
        }

        private void OnStartGame()
        {
            SelectHero(StrategicGameMediator.Instance.GetAnyHero());
            Vector3 playerPosition = HexHelper.GetTilePosition(_selectedHero.Position);
            Vector3 targetPosition = new Vector3(playerPosition.x, playerPosition.y, _camera.transform.position.z);
            _camera.transform.position = targetPosition;
        }

        private void OnStartTurn(Player player)
        {
            if (!_selectedHero.Enabled)
            {
                SelectHero(StrategicGameMediator.Instance.GetAnyHero());
            }

            if (_selectionLogic != null)
            {
                _selectionLogic.Refresh();
            }

            if (player == StrategicGameMediator.Instance.SelfPlayer && _selectedHero != null)
            {
                Vector3 playerPosition = HexHelper.GetTilePosition(_selectedHero.Position);
                Vector3 targetPosition = new Vector3(playerPosition.x, playerPosition.y, _camera.transform.position.z);

                _locker.Lock(LOCK);
                float time = (targetPosition - _camera.transform.position).magnitude * Time.deltaTime * Preferences.MapScrollingSpeed;
                _camera.transform.DOMove(targetPosition, time).OnComplete(() =>
                {
                    _locker.Unlock(LOCK);
                });
            }
        }

        private void OnHeroStartWalk(ulong id, Player player)
        {
            if (player == StrategicGameMediator.Instance.SelfPlayer)
            {
                HexHeroView heroView = _entities[id] as HexHeroView;
                _locker.Lock(LOCK);
                Vector3 target = new Vector3(heroView.Position.x, heroView.Position.y, _camera.transform.position.z);
                float time = (target - _camera.transform.position).magnitude * Time.deltaTime * Preferences.MapScrollingSpeed;
                _camera.transform.DOMove(target, time).OnComplete(() =>
                {
                    if (!_camera.gameObject.TryGetComponent(out PositionConstraint constraint))
                    {
                        constraint = _camera.gameObject.AddComponent<PositionConstraint>();
                    }
                    else
                    {
                        for (int i = constraint.sourceCount - 1; i >= 0; i--)
                        {
                            constraint.RemoveSource(i);
                        }
                    }
                    constraint.AddSource(new ConstraintSource() { sourceTransform = heroView.transform, weight = 1 });
                    constraint.translationAxis = Axis.X | Axis.Y;
                    constraint.constraintActive = true;

                    _locker.Unlock(LOCK);
                });
            }
        }

        private void OnHeroWalk(ulong id, Vector2Int from, Vector2Int to)
        {
            if (_entities.TryGetValue(id, out HexEntityView view) && view is HexHeroView heroView)
            {
                _walkingHeroId = id;
                _locker.Lock(LOCK);
                heroView.Walk(from, to, () =>
                {
                    _locker.Unlock(LOCK);
                    _walkingHeroId = 0;
                });
            }
        }

        private void OnHeroEndWalk(ulong id, Player player)
        {
            if (player == StrategicGameMediator.Instance.SelfPlayer)
            {
                Destroy(_camera.gameObject.GetComponent<PositionConstraint>());
                HexHeroView heroView = _entities[id] as HexHeroView;
                _camera.transform.position = new Vector3(heroView.Position.x, heroView.Position.y, _camera.transform.position.z);
            }
        }

        private Sprite GetCursorSprite(HexContent content, bool reachable)
        {
            switch (content.Type)
            {
                case HexContent.ContentType.Empty:
                    return reachable ? _greenHorseMove : _redHorseMove;
                case HexContent.ContentType.Creature:
                    return reachable ? _greenAttack : _redAttack;
                case HexContent.ContentType.Hero:
                    if (content.Player != StrategicGameMediator.Instance.SelfPlayer)
                    {
                        return reachable ? _greenAttack : _redAttack;
                    }
                    else
                    {
                        return null;
                    }
                case HexContent.ContentType.Obstacle:
                    return null;
                case HexContent.ContentType.Interactable:
                    return reachable ? _greenHorse : _redHorse;
                default:
                    return null;
            }
        }

        private void SelectHero(IHeroModelForView hero)
        {
            _selectedHero = hero;
            OnUpdateSelectedHero?.Invoke(hero);
            DOVirtual.DelayedCall(1f / 60f, () =>
            {
                FillReachableArea();
            }).SetUpdate(false);
        }

        private void ClearReachableArea()
        {
            _reachableArea.Clear();
            foreach (IObjectModelForView reachableObject in _reachableObjects)
            {
                if (_entities.TryGetValue(reachableObject.Id, out HexEntityView view) && view is HexObjectView objectView)
                {
                    objectView.SetSelectionState(ObjectSelectionState.None);
                }
            }
            if (_hoveredReachableObject != null && _entities.TryGetValue(_hoveredReachableObject.Id, out HexEntityView hoveredView) && hoveredView is HexObjectView hoveredObjectView)
            {
                hoveredObjectView.SetSelectionState(ObjectSelectionState.None);
                _hoveredReachableObject = null;
            }
            _reachableObjects.Clear();
        }

        private void FillReachableArea()
        {
            if (_selectedHero != null)
            {
                ClearReachableArea();
                if (_selectedHero.StepsLeft > 0)
                {
                    StrategicGameMediator.Instance.GetReachableByHeroArea(_selectedHero.Id, _reachableArea);
                    _reachableArea.Add(_selectedHero.Position);
                    // TODO refactor, really slow method and should not access map models inside view
                    foreach (HexObjectModel @object in _map.Model.GetEnumerable<HexObjectModel>())
                    {
                        if (@object.IsInteractable)
                        {
                            foreach (Vector2Int position in _reachableArea)
                            {
                                if (HexHelper.AxialDistance(HexHelper.OddToAxial(position), HexHelper.OddToAxial(@object.Position)) <= 1)
                                {
                                    _reachableObjects.Add(@object);
                                    if (_entities.TryGetValue(@object.Id, out HexEntityView view) && view is HexObjectView objectView)
                                    {
                                        objectView.SetSelectionState(ObjectSelectionState.Reachable);
                                    }
                                    break;
                                }
                            }
                        }
                    }
                    //foreach (Vector2Int position in _reachableArea)
                    //{
                    //    ulong id = _map.Model.GetInteractionMaskId(position.x, position.y);
                    //    if (id != 0 && _map.Model.GetEntity(id) is IObjectModelForView objectModel)
                    //    {
                    //        _reachableObjects.Add(objectModel);
                    //        if (_entities.TryGetValue(objectModel.Id, out HexEntityView view) && view is HexObjectView objectView)
                    //        {
                    //            objectView.SetSelectionState(ObjectSelectionState.Reachable);
                    //        }
                    //    }
                    //}
                }
            }
        }

        private void DrawReachableArea()
        {
            //Sprite sprite = HexDatabase.Instance.Mask.GetGroundSprite();
            //Color color = new Color(0, 0, 0, 0.2f);
            Sprite sprite = HexDatabase.Instance.Mask.GetSelectionSprite();
            foreach (Vector2Int reachablePosition in _reachableArea)
            {
                ImmediateSpriteRenderer.DrawSprite(HexHelper.GetTilePosition(reachablePosition), sprite, SortingOrder.ENTITY - 2, _terrain.GetTerrain(reachablePosition).SelectionTint);
            }

            (HexContent content, Vector2Int position) = GetContent(GetCursorWorldPosition());
            IObjectModelForView newObject = null;
            foreach (IObjectModelForView reachableObject in _reachableObjects)
            {
                if (reachableObject.Position == position)
                {
                    newObject = reachableObject;
                    break;
                }
            }
            if (newObject != _hoveredReachableObject)
            {
                if (_hoveredReachableObject != null && _entities.TryGetValue(_hoveredReachableObject.Id, out HexEntityView previousView) && previousView is HexObjectView previousObjectView)
                {
                    previousObjectView.SetSelectionState(ObjectSelectionState.Reachable);
                }
                _hoveredReachableObject = newObject;
                if (newObject != null && _entities.TryGetValue(newObject.Id, out HexEntityView view) && view is HexObjectView objectView)
                {
                    objectView.SetSelectionState(ObjectSelectionState.Hovered);
                }
            }
        }

        private abstract class SelectionLogic
        {
            protected StrategicMapView _view;

            public SelectionLogic(StrategicMapView view)
            {
                _view = view;
            }

            public abstract void Refresh();
            public abstract void Update();

            public virtual void Disable()
            {
                _view._selectionLogic = null;
            }
        }

        private class SelectPathLogic : SelectionLogic
        {
            private (IReadOnlyList<(Vector2Int, int)> path, int stepsLeft) _data;
            private List<SpriteRenderer> _pathRenderers = new();
            private Vector2Int _target;

            public SelectPathLogic(StrategicMapView view, Vector2Int target) : base(view)
            {
                _target = target;
                Refresh();
            }

            public override void Refresh()
            {
                _data = StrategicGameMediator.Instance.GetHeroPath(_view._selectedHero.Id, _target);
                if (_data.path == null)
                {
                    Disable();
                    return;
                }

                ClearPath();
                IReadOnlyList<(Vector2Int, int)> path = _data.path;
                for (int i = 1; i < path.Count; i++)
                {
                    SpriteRenderer renderer = SpriteRendererPool.Get();
                    renderer.transform.position = HexHelper.GetTilePosition(path[i].Item1);
                    renderer.sprite = (i < path.Count - 1) ? (path[i].Item2 <= _data.stepsLeft ? _view._greenDot : _view._redDot) : (path[i].Item2 <= _data.stepsLeft ? _view._greenCross : _view._redCross);
                    renderer.sortingOrder = SortingOrder.PATH;
                    _pathRenderers.Add(renderer);
                }
            }

            public override void Update()
            {
                (HexContent content, Vector2Int position) = _view.GetContent(_view.GetCursorWorldPosition());

                if (ControlHandler.Instance.WasLeftClicked && ClickCooldown.TryClick())
                {
                    if (_target == position)
                    {
                        StrategicGameMediator.Instance.MoveHero(_view._selectedHero.Id, _target);
                    }
                    Disable();
                }

                if (Input.GetKeyDown(KeyCode.Space))
                {
                    StrategicGameMediator.Instance.MoveHero(_view._selectedHero.Id, _target);
                    Disable();
                }

                if (ControlHandler.Instance.WasRightClicked)
                {
                    Disable();
                }
            }

            public override void Disable()
            {
                base.Disable();
                ClearPath();
            }

            private void ClearPath()
            {
                _pathRenderers.ForEach((s) => SpriteRendererPool.Release(s));
                _pathRenderers.Clear();
            }
        }
    }
}
