﻿using System.Collections.Generic;
using System.Linq;

namespace PK.Tactical
{
    public partial class GameController
    {
        private class MeleeAttackSquadsCommand : Command
        {
            private ulong _id;
            private ulong _targetId;

            public MeleeAttackSquadsCommand(ulong id, ulong targetId)
            {
                _id = id;
                _targetId = targetId;
            }

            protected override CommandResult ExecuteBeforeChilds(IContext context)
            {
                List<HexCreatureSquadModel> squads = new();
                List<HexCreatureSquadModel> targetSquads = new();
                HexMapModel map = context.MapData.Map;
                SquadHelper.GetAllSquads(_id, map, squads);
                SquadHelper.GetAllSquads(_targetId, map, targetSquads);

                if (!SquadHelper.AreSquadsNearby(squads, targetSquads))
                {
                    return CommandResult.Complete;
                }

                HexCreature creature = HexDatabase.Instance.GetCreature(squads[0].Uid);

                HashSet<HexCreatureSquadModel> attackingSquads = new();
                HashSet<HexCreatureSquadModel> attackedSquads = new();

                foreach (HexCreatureSquadModel targetSquad in targetSquads)
                {
                    foreach (HexCreatureSquadModel squad in squads)
                    {
                        if (HexHelper.AxialDistance(HexHelper.OddToAxial(squad.Position), HexHelper.OddToAxial(targetSquad.Position)) == 1)
                        {
                            attackingSquads.Add(squad);
                            attackedSquads.Add(targetSquad);
                        }
                    }
                }
                float damage = 0f;
                foreach (HexCreatureSquadModel squad in attackingSquads)
                {
                    if (creature is HexCavalryCreature cavalryCreature && squad.StepsMoved >= cavalryCreature.MinChargeSteps)
                    {
                        damage += cavalryCreature.ChargeDamage * squad.Units.Count;
                    }
                    else
                    {
                        damage += creature.Damage * squad.Units.Count;
                    }
                }
                PushSubCommand(new AttackSubCommand(squads, attackingSquads, attackedSquads));
                damage /= attackedSquads.Count;
                if (damage > 0)
                {
                    PushSubCommand(new DamageSquadsCommand(attackedSquads.ToList(), damage));
                }
                context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsStartAttack(targetSquads.Select((s) => s.Id).ToList());
                return CommandResult.Complete;
            }

            private class AttackSubCommand : Command
            {
                private List<HexCreatureSquadModel> _squads;
                private HashSet<HexCreatureSquadModel> _attackingSquads;
                private HashSet<HexCreatureSquadModel> _attackedSquads;

                public AttackSubCommand(List<HexCreatureSquadModel> squads, HashSet<HexCreatureSquadModel> attackingSquads, HashSet<HexCreatureSquadModel> attackedSquads)
                {
                    _squads = squads;
                    _attackingSquads = attackingSquads;
                    _attackedSquads = attackedSquads;
                }

                protected override CommandResult ExecuteBeforeChilds(IContext context)
                {
                    context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsAttackMelee(_attackingSquads.Select((s) => s.Id).ToList(), _attackedSquads.Select((s) => s.Id).ToList());
                    foreach (HexCreatureSquadModel squad in _squads)
                    {
                        squad.HasAction = false;
                        if (squad.ParentSquadId == 0)
                        {
                            context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadUpdate(squad.Id, false);
                        }
                    }
                    return CommandResult.Complete;
                }
            }
        }
    }
}
