﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public partial class GameController
    {
        private class RangedAttackAOECommand : Command
        {
            private ulong _id;
            private Vector2Int _target;
            private int _radius;

            public RangedAttackAOECommand(ulong id, Vector2Int target, int radius)
            {
                _id = id;
                _target = target;
                _radius = radius;
            }

            protected override CommandResult ExecuteBeforeChilds(IContext context)
            {
                List<HexCreatureSquadModel> attackingSquads = new();
                HexMapModel map = context.MapData.Map;
                SquadHelper.GetAllSquads(_id, map, attackingSquads);
                HexRangedCreature creature = HexDatabase.Instance.GetCreature(attackingSquads[0].Uid) as HexRangedCreature;
                Vector2Int axialTarget = HexHelper.OddToAxial(_target);

                List<Vector2Int> targets = new();
                Vector3Int cubeTarget = HexHelper.OddToCube(_target);
                for (int i = -_radius; i <= _radius; i++)
                {
                    for (int j = -_radius; j <= _radius; j++)
                    {
                        for (int k = -_radius; k <= _radius; k++)
                        {
                            if (i + j + k == 0)
                            {
                                targets.Add(HexHelper.CubeToOdd(cubeTarget + new Vector3Int(i, j, k)));
                            }
                        }
                    }
                }

                float damage = 0f;
                foreach (HexCreatureSquadModel squad in attackingSquads)
                {
                    damage += creature.RangedDamage * squad.Units.Count;
                }
                damage /= (targets.Count * 2);

                IEnumerable<HexCreatureSquadModel> allSquads = map.GetEnumerable<HexCreatureSquadModel>();
                Dictionary<ulong, List<HexCreatureSquadModel>> squadsToAttack = new();
                foreach (Vector2Int target in targets)
                {
                    foreach (HexCreatureSquadModel squad in allSquads)
                    {
                        if (squad.Position == target)
                        {
                            ulong id = squad.ParentSquadId == 0 ? squad.Id : squad.ParentSquadId;
                            if (!squadsToAttack.TryGetValue(id, out List<HexCreatureSquadModel> squads))
                            {
                                squads = new List<HexCreatureSquadModel>();
                                squadsToAttack[id] = squads;
                            }
                            squads.Add(squad);
                        }
                    }
                }
                PushSubCommand(new AttackSubCommand(attackingSquads, targets));
                foreach (KeyValuePair<ulong, List<HexCreatureSquadModel>> pair in squadsToAttack)
                {
                    PushSubCommand(new DamageSquadsCommand(pair.Value, damage));
                }
                context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsStartAttack(squadsToAttack.First().Value.Select((s) => s.Id).ToList());
                return CommandResult.Complete;
            }

            private class AttackSubCommand : Command
            {
                private List<HexCreatureSquadModel> _attackingSquads;
                private List<Vector2Int> _targets;

                public AttackSubCommand(List<HexCreatureSquadModel> attackingSquads, List<Vector2Int> targets)
                {
                    _attackingSquads = attackingSquads;
                    _targets = targets;
                }

                protected override CommandResult ExecuteBeforeChilds(IContext context)
                {
                    context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadsAttackRanged(_attackingSquads.Select((s) => s.Id).ToList(), _targets);

                    foreach (HexCreatureSquadModel squad in _attackingSquads)
                    {
                        squad.HasAction = false;
                        if (squad.ParentSquadId == 0)
                        {
                            context.EventManager.Get<TacticalEvents>().CallOnCreatureSquadUpdate(squad.Id, false);
                        }
                    }
                    return CommandResult.Complete;
                }
            }
        }
    }
}
