﻿using System.Collections.Generic;
using UnityEngine;

namespace PK.Tactical
{
    public partial class GameController : BaseController<GameController.IGameContext>
    {
        private Savefile _savefile;
        private EventManager _eventManager = new();
        private ControllerLocker _locker = new();
        private GameContext _context = new();

        public Player SelfPlayer { get { return Player.Red; } }
        public Player EnemyPlayer { get { return Player.Blue; } }
        public Player CurrentPlayer { get { return Context.Player; } }
        public IEventManager EventManager { get { return _eventManager; } }
        public IControllerLocker Locker { get { return _locker; } }
        private IGameContext Context { get { return _context; } }

        public GameController()
        {
            RegisterEvents();
        }

        public void Initialize(Savefile savefile)
        {
            _savefile = savefile;
            _context.Init();
            InitContext(savefile.TacticalMap, savefile, _eventManager, _locker, _context);
        }

        public bool PlaceSquadsRandomly(HexCreatureSquadModel[] redSquads, HexCreatureSquadModel[] blueSquads)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new PlaceCreatureSquadsRandomly(redSquads, Player.Red));
                turn.Push(new PlaceCreatureSquadsRandomly(blueSquads, Player.Blue));
            });
        }

        public bool StartGame(Player beginer)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new StartGameCommand());
                turn.Push(new SwitchTurnCommand(beginer == Player.Red ? 1 : 2));
            });
        }

        public bool StartNewTurn()
        {
            if (!Context.BattleInProgress)
            {
                return false;
            }
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new SwitchTurnCommand());
            });
        }

        public bool IsReachableBySquad(ulong id, Vector2Int position, List<Vector2Int> squadsPositions)
        {
            if (_savefile.TacticalMap.Map.GetEntity(id) is HexCreatureSquadModel squad)
            {
                HexCreature creature = HexDatabase.Instance.GetCreature(squad.Uid);
                List<(Vector2Int, int)> path = PathfindingHelper.FindPath(squad.Position, position, _savefile.TacticalMap.ObstacleMask, _savefile.TacticalMap.MovementCostMask, _savefile.TacticalMap.Map.InteractionMask, _savefile.TacticalMap.Map.Size, squadsPositions);
                if (path != null && path[path.Count - 1].Item2 <= creature.StepsPerTurn)
                {
                    return true;
                }
            }
            return false;
        }

        public void GetReachableBySquadArea(ulong id, List<Vector2Int> resultPositions, List<Vector2Int> squadsPositions)
        {
            if (_savefile.TacticalMap.Map.GetEntity(id) is HexCreatureSquadModel squad)
            {
                HexCreature creature = HexDatabase.Instance.GetCreature(squad.Uid);
                PathfindingHelper.FindArea(squad.Position, creature.StepsPerTurn, _savefile.TacticalMap.ObstacleMask, _savefile.TacticalMap.MovementCostMask, _savefile.TacticalMap.Map.InteractionMask, _savefile.TacticalMap.Map.Size, resultPositions, squadsPositions);
            }
        }

        public bool AreSquadsNearby(ulong id, ulong targetId)
        {
            return SquadHelper.AreSquadsNearby(id, targetId, _savefile.TacticalMap.Map);
        }

        public void GetAllSquads(Player player, List<ICreatureSquadModelForView> resultSquads)
        {
            foreach (HexCreatureSquadModel squad in _savefile.TacticalMap.Map.GetEnumerable<HexCreatureSquadModel>())
            {
                if (squad.ParentSquadId == 0 && squad.Player == player)
                {
                    resultSquads.Add(squad);
                }
            }
        }

        public void GetAllSquads(ulong id, List<ICreatureSquadModelForView> resultSquads)
        {
            SquadHelper.GetAllSquads(id, _savefile.TacticalMap.Map, resultSquads);
        }

        public bool MoveSquads(Dictionary<ulong, Vector2Int> targets)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new MoveSquadsCommand(targets));
                turn.Push(new TryToEndTurnCommand());
            });
        }

        public bool AttackSquadsMelee(ulong id, ulong targetId)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new MeleeAttackSquadsCommand(id, targetId));
                turn.Push(new TryToEndTurnCommand());
                turn.Push(new TryToEndBattleCommand());
            });
        }

        public bool AttackSquadsRanged(ulong id, ulong targetId)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new RangedAttackSquadsCommand(id, targetId));
                turn.Push(new TryToEndTurnCommand());
                turn.Push(new TryToEndBattleCommand());
            });
        }

        public bool AttackSquadsRangedAOE(ulong id, Vector2Int target, int radius)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new RangedAttackAOECommand(id, target, radius));
                turn.Push(new TryToEndTurnCommand());
                turn.Push(new TryToEndBattleCommand());
            });
        }

        public bool ExecuteAITask(AITask task)
        {
            return CreateAndExecuteTurn((ITurn turn) =>
            {
                turn.Push(new ExecuteAITaskCommand(task));
                turn.Push(new TryToEndTurnCommand());
                turn.Push(new TryToEndBattleCommand());
            });
        }

        public bool EndTurn(Player player)
        {
            if (CurrentPlayer != player || !Context.BattleInProgress)
            {
                return false;
            }

            return StartNewTurn();
        }

        public bool Simulate()
        {
            return Execute();
        }

        private void RegisterEvents()
        {
            _eventManager.Register<ITacticalEvents, TacticalEvents>(new TacticalEvents());
            _eventManager.Register<IMapEvents, MapEvents>(new MapEvents());
            _eventManager.Register<IMapDataEvents, MapDataEvents>(new MapDataEvents());
            _eventManager.Register<IInteractionEvents, InteractionEvents>(new InteractionEvents());
        }
    }
}
