﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public class TacticalAudioHandler : MonoBehaviour, IEventReceiver
    {
        [SerializeField] private TacticalMapView _view;

        void IEventReceiver.AddEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnStartGame += OnStartGame;
            events.OnEndBattle += OnEndBattle;
            events.OnCreatureSquadsMove += OnCreatureSquadsMove;
            events.OnCreatureSquadsAttackMelee += OnCreatureSquadsAttackMelee;
            events.OnCreatureSquadsAttackRanged += OnCreatureSquadsAttackRanged;
            events.OnCreatureSquadsHit += OnCreatureSquadsHit;
            events.OnCreatureSquadsSurrender += OnCreatureSquadsSurrender;

            _view.OnCreatureSquadSelected += OnCreatureSquadSelected;
        }

        void IEventReceiver.RemoveEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnStartGame -= OnStartGame;
            events.OnEndBattle -= OnEndBattle;
            events.OnCreatureSquadsMove -= OnCreatureSquadsMove;
            events.OnCreatureSquadsAttackMelee -= OnCreatureSquadsAttackMelee;
            events.OnCreatureSquadsAttackRanged -= OnCreatureSquadsAttackRanged;
            events.OnCreatureSquadsHit -= OnCreatureSquadsHit;
            events.OnCreatureSquadsSurrender -= OnCreatureSquadsSurrender;

            _view.OnCreatureSquadSelected -= OnCreatureSquadSelected;
        }

        private void OnStartGame(Vector2Int spawnAreaPosition)
        {
            AudioHandler.Instance.PlayAmbience(string.Empty);
            AudioHandler.Instance.PlaySfx("StartBattleSfx", () =>
            {
                AudioHandler.Instance.PlayAmbience("BattleAmbience");
            });
        }

        private void OnEndBattle(Player player)
        {
            AudioHandler.Instance.PlayAmbience(string.Empty);
            string sfxName = player == TacticalGameMediator.Instance.SelfPlayer ? "WinBattleSfx" : "LoseBattleSfx";
            AudioHandler.Instance.PlaySfx(sfxName);
        }

        private void OnCreatureSquadsMove(List<(ICreatureSquadModelForView squad, Vector2Int start, Vector2Int end)> data)
        {
            if (_view.Entities.TryGetValue(data[0].squad.Id, out HexEntityView view))
            {
                HexCreature creature = HexDatabase.Instance.GetCreature(view.Uid);
                AudioHandler.Instance.PlaySfx(creature.MoveSfx);
            }
        }

        private void OnCreatureSquadsAttackMelee(List<ulong> ids, List<ulong> targetIds)
        {
            if (_view.Entities.TryGetValue(ids[0], out HexEntityView view))
            {
                HexCreature creature = HexDatabase.Instance.GetCreature(view.Uid);
                AudioHandler.Instance.PlaySfx(creature.MeleeAttackSfx);
            }
        }

        private void OnCreatureSquadsAttackRanged(List<ulong> ids, List<Vector2Int> targets)
        {
            if (_view.Entities.TryGetValue(ids[0], out HexEntityView view))
            {
                HexCreature creature = HexDatabase.Instance.GetCreature(view.Uid);
                if (creature is HexRangedCreature rangedCreature)
                {
                    AudioHandler.Instance.PlaySfx(rangedCreature.RangedAttackSfx);
                }
            }
        }

        private void OnCreatureSquadsHit(List<ICreatureSquadModelForView> targets, List<ICreatureSquadModelForView> allTargets)
        {
            HexCreature creature = HexDatabase.Instance.GetCreature(targets[0].Uid);
            if (targets.Any((s) => s.Health > 0))
            {
                AudioHandler.Instance.PlaySfx(creature.HitSfx);
            }
            else
            {
                AudioHandler.Instance.PlaySfx(creature.DeathSfx);
            }
        }

        private void OnCreatureSquadsSurrender(List<(ICreatureSquadModelForView, Vector2Int)> data)
        {
            if (_view.Entities.TryGetValue(data[0].Item1.Id, out HexEntityView view))
            {
                if (data.Any((s) => s.Item1.Health > 0))
                {
                    HexCreature creature = HexDatabase.Instance.GetCreature(view.Uid);
                    AudioHandler.Instance.PlaySfx(creature.SurrenderSfx);
                }
            }
        }

        private void OnCreatureSquadSelected(ulong id)
        {
            if (_view.Entities.TryGetValue(id, out HexEntityView view))
            {
                HexCreature creature = HexDatabase.Instance.GetCreature(view.Uid);
                AudioHandler.Instance.PlaySfx(creature.SelectSfx);
            }
        }
    }
}
