﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK.Tactical
{
    public interface ITacticalGameMediator
    {
        public Player SelfPlayer { get; }
        public Player EnemyPlayer { get; }
        public Player CurrentPlayer { get; }
        public IEventManager EventManager { get; }
        public IControllerLocker Locker { get; }

        public HexContent GetContent(Vector2Int position);
        public bool IsVisible(Vector2Int position);
        public bool IsReachableBySquad(ulong id, Vector2Int position, List<Vector2Int> squadsPositions);
        public void GetReachableBySquadArea(ulong id, List<Vector2Int> resultPositions, List<Vector2Int> squadsPositions);
        public bool AreSquadsNearby(ulong id, ulong targetId);
        public void GetAllSquads(Player player, List<ICreatureSquadModelForView> resultSquads);
        public void GetAllSquads(ulong id, List<ICreatureSquadModelForView> resultSquads);
        public void MoveSquads(Dictionary<ulong, Vector2Int> targets);
        public void AttackSquadsMelee(ulong id, ulong targetId);
        public void AttackSquadsRanged(ulong id, ulong targetId);
        public void AttackSquadsRangedAOE(ulong id, Vector2Int target, int radius);
        public void MoveAndAttackSquadsMelee(Dictionary<ulong, Vector2Int> targets, ulong targetId);

        public void ExecuteAITask(AITask task, Action callback);

        public void EndTurn(Player player);
    }

    public class TacticalGameMediator : MonoBehaviour, ITacticalGameMediator
    {
        const string LOCKER = "TacticalGameMediator";

        private static TacticalGameMediator _instance;

        public static ITacticalGameMediator Instance { get { return _instance; } }

        [SerializeField] private TacticalMapView _view;

        private Savefile _savefile;
        private GameController _controller;

        Player ITacticalGameMediator.SelfPlayer { get { return _controller.SelfPlayer; } }
        Player ITacticalGameMediator.EnemyPlayer { get { return _controller.EnemyPlayer; } }
        Player ITacticalGameMediator.CurrentPlayer { get { return _controller.CurrentPlayer; } }
        IEventManager ITacticalGameMediator.EventManager { get { return _controller.EventManager; } }
        IControllerLocker ITacticalGameMediator.Locker { get { return _controller.Locker; } }

        private void Awake()
        {
            _instance = this;
            _controller = new GameController();
        }

        public void SetViewActive(bool active)
        {
            for (int i = 0; i < transform.childCount; i++)
            {
                GameObject child = transform.GetChild(i).gameObject;
                child.SetActive(active);
            }
            foreach (IEventReceiver receiver in GetComponentsInChildren<IEventReceiver>(true))
            {
                if (active)
                {
                    receiver.AddEvents();
                }
                else
                {
                    receiver.RemoveEvents();
                }
            }
        }

        public void Initialize(Savefile savefile, HexMap map)
        {
            _savefile = savefile;
            _controller.Initialize(savefile);
            _view.SetMap(map, _controller.EventManager.Get<IMapEvents>(), _controller.Locker);
        }

        public void StartGame(ulong heroId, HexCreatureSquadModel[] enemySquads, Player beginer)
        {
            Savefile.HeroData heroData = _savefile.GetHeroData(_savefile.StrategicMap.Map.GetHero(heroId));
            // Clone squads to avoid modyfing the original ones
            string data = JsonUtility.ToJson(heroData);
            List<HexCreatureSquadModel> playerSquads = JsonUtility.FromJson<Savefile.HeroData>(data).Squads;
            if (_controller.PlaceSquadsRandomly(playerSquads.ToArray(), enemySquads))
            {
                Simulate(() =>
                {
                    if (_controller.StartGame(beginer))
                    {
                        Simulate();
                    }
                });
            }
        }

        HexContent ITacticalGameMediator.GetContent(Vector2Int position)
        {
            return _savefile.TacticalMap.GetContent(position);
        }

        bool ITacticalGameMediator.IsVisible(Vector2Int position)
        {
            return _savefile.TacticalMap.IsVisible(position);
        }

        bool ITacticalGameMediator.IsReachableBySquad(ulong id, Vector2Int position, List<Vector2Int> squadsPositions)
        {
            return _controller.IsReachableBySquad(id, position, squadsPositions);
        }

        void ITacticalGameMediator.GetReachableBySquadArea(ulong id, List<Vector2Int> resultPositions, List<Vector2Int> squadsPositions)
        {
            _controller.GetReachableBySquadArea(id, resultPositions, squadsPositions);
        }

        bool ITacticalGameMediator.AreSquadsNearby(ulong id, ulong targetId)
        {
            return _controller.AreSquadsNearby(id, targetId);
        }

        void ITacticalGameMediator.GetAllSquads(Player player, List<ICreatureSquadModelForView> resultSquads)
        {
            _controller.GetAllSquads(player, resultSquads);
        }

        void ITacticalGameMediator.GetAllSquads(ulong id, List<ICreatureSquadModelForView> resultSquads)
        {
            _controller.GetAllSquads(id, resultSquads);
        }

        void ITacticalGameMediator.MoveSquads(Dictionary<ulong, Vector2Int> targets)
        {
            if (_controller.MoveSquads(targets))
            {
                Simulate();
            }
        }

        void ITacticalGameMediator.AttackSquadsMelee(ulong id, ulong targetId)
        {
            if (_controller.AttackSquadsMelee(id, targetId))
            {
                Simulate();
            }
        }

        void ITacticalGameMediator.AttackSquadsRanged(ulong id, ulong targetId)
        {
            if (_controller.AttackSquadsRanged(id, targetId))
            {
                Simulate();
            }
        }

        void ITacticalGameMediator.AttackSquadsRangedAOE(ulong id, Vector2Int target, int radius)
        {
            if (_controller.AttackSquadsRangedAOE(id, target, radius))
            {
                Simulate();
            }
        }

        void ITacticalGameMediator.MoveAndAttackSquadsMelee(Dictionary<ulong, Vector2Int> targets, ulong targetId)
        {
            if (_controller.MoveSquads(targets))
            {
                Simulate(() =>
                {
                    if (_controller.AttackSquadsMelee(targets.Keys.First(), targetId))
                    {
                        Simulate();
                    }
                });
            }
        }

        void ITacticalGameMediator.ExecuteAITask(AITask task, Action callback)
        {
            if (_controller.ExecuteAITask(task))
            {
                Simulate(callback);
            }
        }

        void ITacticalGameMediator.EndTurn(Player player)
        {
            if (_controller.EndTurn(player))
            {
                Simulate();
            }
        }

        private void Simulate(Action callback = null)
        {
            StartCoroutine(SimulationCoroutine(callback));
        }

        private IEnumerator SimulationCoroutine(Action callback)
        {
            InputLock.Lock(LOCKER);
            while (true)
            {
                bool result = _controller.Simulate();

                while (_controller.Locker.IsLock)
                {
                    yield return null;
                }

                if (!result)
                {
                    break;
                }
            }
            InputLock.Unlock(LOCKER);
            callback?.Invoke();
        }
    }
}
