﻿using System.Collections.Generic;
using UnityEngine;

namespace PK.Tactical
{
    public class CreatureSquadBar : MonoBehaviour, IEventReceiver
    {
        [SerializeField] private TacticalMapView _view;
        [SerializeField] private CreatureSquadBarIcon _iconPrefab;
        [SerializeField] private CanvasGroup _canvasGroup;

        private List<CreatureSquadBarIcon> _icons = new();
        private List<ICreatureSquadModelForView> _squads = new();
        private List<ICreatureSquadModelForView> _temporarySquads = new();

        void IEventReceiver.AddEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnStartTurn += OnStartTurn;
            events.OnCreatureSquadUpdate += OnCreatureSquadUpdate;

            _view.OnCreatureSquadHovered += OnCreatureSquadHovered;
            _view.OnCreatureSquadSelected += OnCreatureSquadSelected;
        }

        void IEventReceiver.RemoveEvents()
        {
            ITacticalEvents events = TacticalGameMediator.Instance.EventManager.Get<ITacticalEvents>();
            events.OnStartTurn -= OnStartTurn;
            events.OnCreatureSquadUpdate -= OnCreatureSquadUpdate;

            _view.OnCreatureSquadHovered -= OnCreatureSquadHovered;
            _view.OnCreatureSquadSelected -= OnCreatureSquadSelected;
        }

        private void OnStartTurn(Player player)
        {
            Refresh();
        }

        private void OnCreatureSquadUpdate(ulong id, bool hasAction)
        {
            Refresh();
        }

        private void OnCreatureSquadHovered(ulong id)
        {
            foreach (CreatureSquadBarIcon icon in _icons)
            {
                icon.SetSelectionState(icon.IsSquad(id) ? SquadSelectionState.Hovered : SquadSelectionState.None);
            }
        }

        private void OnCreatureSquadSelected(ulong id)
        {
            foreach (CreatureSquadBarIcon icon in _icons)
            {
                icon.SetSelectionState(icon.IsSquad(id) ? SquadSelectionState.Selected : SquadSelectionState.None);
            }
        }

        private void Refresh()
        {
            _squads.Clear();
            TacticalGameMediator.Instance.GetAllSquads(TacticalGameMediator.Instance.SelfPlayer, _squads);
            _canvasGroup.interactable = TacticalGameMediator.Instance.CurrentPlayer == TacticalGameMediator.Instance.SelfPlayer;
            for (int i = 0; i < _squads.Count; i++)
            {
                ICreatureSquadModelForView squad = _squads[i];
                CreatureSquadBarIcon icon;
                if (i < _icons.Count)
                {
                    icon = _icons[i];
                }
                else
                {
                    icon = Instantiate(_iconPrefab, transform);
                    _icons.Add(icon);
                }
                icon.SetActive(true);
                _temporarySquads.Clear();
                TacticalGameMediator.Instance.GetAllSquads(squad.Id, _temporarySquads);
                icon.Refresh(_temporarySquads, false, (CreatureSquadBarIcon.Result result) =>
                {
                    if (result == CreatureSquadBarIcon.Result.Click)
                    {
                        _view.SelectSquads(squad.Id);
                    }
                    else
                    {
                        _view.HoverSquads(result == CreatureSquadBarIcon.Result.Enter ? squad.Id : 0);
                    }
                });
            }
            for (int i = _squads.Count; i < _icons.Count; i++)
            {
                _icons[i].SetActive(false);
            }
        }
    }
}
