﻿using UnityEditor;
using UnityEditor.Localization;
using UnityEngine.Localization;
using UnityEngine.Localization.Tables;
using UnityEngine.UIElements;

namespace PK
{
    [EditorNode(typeof(ActionDialogMessageNode), "Dialogs/Message")]
    class ActionDialogMessageEditorNode : ActionDefaultEditorNode
    {
        public ActionDialogMessageEditorNode(SerializedProperty property) : base(property)
        {
            title = "Dialog Message";
            SetTitleColor(Styles.MessageTitleColor);
            InitializeDefaultPorts(true, true);

            SerializedProperty characterProperty = property.FindPropertyRelative("_character");
            SerializedProperty backgroundProperty = property.FindPropertyRelative("_background");
            SerializedProperty lineProperty = property.FindPropertyRelative("_line");

            Label label = new Label();
            label.style.whiteSpace = WhiteSpace.Normal;
            label.style.maxWidth = 200f;
            RefreshLabel(lineProperty, label);
            object value = lineProperty.boxedValue;

            IMGUIContainer propertiesField = new IMGUIContainer(() =>
            {
                EditorGUI.BeginChangeCheck();
                PropertyHelper.PropertyField(characterProperty);
                PropertyHelper.PropertyField(backgroundProperty);
                if (EditorGUI.EndChangeCheck())
                {
                    _property.serializedObject.ApplyModifiedProperties();
                }

                EditorGUILayout.PropertyField(lineProperty);
                if (value != lineProperty.boxedValue)
                {
                    value = lineProperty.boxedValue;
                    _property.serializedObject.ApplyModifiedProperties();
                    RefreshLabel(lineProperty, label);
                }
            });
            extensionContainer.Add(propertiesField);
            extensionContainer.Add(label);

            Refresh();
        }

        private void RefreshLabel(SerializedProperty property, Label label)
        {
            LocalizedString str = (property.boxedValue as LocalizedString);
            if (str == null || str.IsEmpty)
            {
                label.text = string.Empty;
                return;
            }
            StringTableCollection collection = LocalizationEditorSettings.GetStringTableCollection(str.TableReference);
            StringTable table = collection.GetTable("en") as StringTable;
            StringTableEntry entry = table.GetEntryFromReference(str.TableEntryReference);
            if (entry == null)
            {
                label.text = "Missing";
            }
            else
            {
                label.text = entry.LocalizedValue;
            }
        }
    }
}
