﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    public interface ICreatureSquadModelForView : IEntityModelForView
    {
        public ulong ParentSquadId { get; }
        public Player Player { get; }
        public List<HexCreatureSquadModel.UnitData> Units { get; }
        public float Health { get; }
        public float MaxHealth { get; }
        public int AliveUnitCount { get; }
        public bool HasAction { get; }
    }

    [System.Serializable]
    public class HexCreatureSquadModel : HexEntityModel, ICreatureSquadModelForView
    {
        [System.Serializable]
        public class UnitData
        {
            [SerializeField] private float _health;

            public float Health { get { return _health; } set { _health = value; } }

            public UnitData()
            {
            }

            public UnitData(float health)
            {
                _health = health;
            }
        }

        [SerializeField] private ulong _parentSquadId;
        [SerializeField] private List<UnitData> _units = new();
        [SerializeField] private Player _player;
        [SerializeField] private float _moral;
        [SerializeField] private float _unitHealth;
        [SerializeField] private bool _hasAction;
        [SerializeField] private int _stepsMoved;

        public override EntityType Type { get { return EntityType.CreatureSquad; } }
        public ulong ParentSquadId { get { return _parentSquadId; } set { _parentSquadId = value; } }
        public List<UnitData> Units { get { return _units; } }
        public float Moral { get { return _moral; } set { _moral = value; } }
        public float Health { get { return _units.Where((u) => u != null).Sum((u) => u.Health); } }
        public int AliveUnitCount { get { return _units.Count((u) => u != null && u.Health > 0); } }
        public float MaxHealth { get { return _units.Count * _unitHealth; } }
        public bool HasAction { get { return _hasAction; } set { _hasAction = value; } }
        public int StepsMoved { get { return _stepsMoved; } set { _stepsMoved = value; } }
        public Player Player { get { return _player; } set { _player = value; } }

        public HexCreatureSquadModel()
        {
        }

        public HexCreatureSquadModel(ulong uid, ulong parentSquadId, Vector2Int position, Player player, int unitCount, float health)
        {
            _uid = uid;
            _parentSquadId = parentSquadId;
            _position = position;
            _player = player;
            _unitHealth = health;
            for (int i = 0; i < unitCount; i++)
            {
                _units.Add(new UnitData(health));
            }
        }

        private HexCreatureSquadModel(ulong uid)
        {
            _uid = uid;
        }

        public HexCreatureSquadModel[] Split(int soldierPerSquad)
        {
            int count = (_units.Count + (soldierPerSquad - 1)) / soldierPerSquad;
            HexCreatureSquadModel[] result = new HexCreatureSquadModel[count];
            int targetSoldiersPerSquad = Mathf.CeilToInt(_units.Count / (float)count);
            for (int i = 0; i < _units.Count; i += targetSoldiersPerSquad)
            {
                HexCreatureSquadModel model = new HexCreatureSquadModel(_uid);
                model._unitHealth = _unitHealth;
                model.Units.AddRange(_units.GetRange(i, Mathf.Min(targetSoldiersPerSquad, _units.Count - i)));
                result[i / targetSoldiersPerSquad] = model;
            }
            return result;
        }

        public static HexCreatureSquadModel Merge(params HexCreatureSquadModel[] squads)
        {
            HexCreatureSquadModel result = new HexCreatureSquadModel();
            foreach (HexCreatureSquadModel squad in squads)
            {
                if (squad.ParentSquadId == 0)
                {
                    squad.Clone(result);
                }
                result._units.AddRange(squad.Units.Where((u) => u.Health > 0));
            }
            return result;
        }

        private void Clone(HexCreatureSquadModel target)
        {
            target._uid = _uid;
            target._id = _id;
            target._player = _player;
            target._moral = _moral;
            target._unitHealth = _unitHealth;
            target._hasAction = _hasAction;
            target._stepsMoved = _stepsMoved;
        }
    }
}
