﻿using System;
using System.Collections.Generic;
using System.Reflection;
using UnityEditor;
using UnityEditor.Searcher;
using UnityEngine;
using UnityEngine.UIElements;

namespace PK
{
    public struct SearchContextElement
    {
        private object _target;
        private string _title;

        public object Target { get { return _target; } }
        public string Title { get { return _title; } }

        public SearchContextElement(object target, string title)
        {
            _target = target;
            _title = title;
        }
    }

    public class BaseGraphSearchWindowProvider : ScriptableObject
    {
        private Dictionary<Type, NodeEntry> _nodeEntries = new();
        private BaseGraphView _graphView;

        public BaseGraphView GraphView { set { _graphView = value; } }

        public Type GetEditorNodeType(Type nodeType)
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            if (_nodeEntries.TryGetValue(nodeType, out NodeEntry entry))
            {
                return entry.EditorNodeType;
            }

            return null;
        }

        public Searcher LoadSearchWindow()
        {
            if (_nodeEntries.Count == 0)
            {
                GenerateEntries();
            }

            List<SearcherItem> root = new();
            NodeEntry dummyEntry = new NodeEntry();

            foreach (NodeEntry nodeEntry in _nodeEntries.Values)
            {
                SearcherItem parent = null;
                string[] entryTitle = nodeEntry.MenuItem.Split('/');

                for (int i = 0; i < entryTitle.Length; i++)
                {
                    var pathEntry = entryTitle[i];
                    var children = parent != null ? parent.Children : root;
                    var item = children.Find(x => x.Name == pathEntry);

                    if (item == null)
                    {
                        //if we don't have slot entries and are at a leaf, add userdata to the entry
                        if (i == entryTitle.Length - 1)
                            item = new SearchNodeItem(pathEntry, nodeEntry);

                        //if we aren't a leaf, don't add user data
                        else
                            item = new SearchNodeItem(pathEntry, dummyEntry);

                        if (parent != null)
                        {
                            parent.AddChild(item);
                        }
                        else
                        {
                            children.Add(item);
                        }
                    }

                    parent = item;

                    if (parent.Depth == 0 && !root.Contains(parent))
                        root.Add(parent);
                }
            }

            SearcherDatabase nodeDatabase = SearcherDatabase.Create(root, string.Empty, false);

            return new Searcher(nodeDatabase, new SearchWindowAdapter("Create Node"));
        }

        public bool OnSelectEntry(SearcherItem selectedEntry, Vector2 mousePosition)
        {
            if (selectedEntry == null || ((selectedEntry as SearchNodeItem).NodeEntry.NodeType == null && (selectedEntry as SearchNodeItem).NodeEntry.EditorNodeType == null))
            {
                return false;
            }

            NodeEntry nodeEntry = (selectedEntry as SearchNodeItem).NodeEntry;
            Vector2 windowMousePosition = _graphView.Window.rootVisualElement.ChangeCoordinatesTo(_graphView.Window.rootVisualElement.parent, mousePosition);
            Vector2 graphMousePosition = _graphView.contentViewContainer.WorldToLocal(windowMousePosition);

            BaseNode node = (BaseNode)Activator.CreateInstance(nodeEntry.NodeType);
            node.Position = new Rect(graphMousePosition, new Vector2());
            _graphView.AddNode(node);

            return true;
        }

        private void GenerateEntries()
        {
            _nodeEntries.Clear();
            foreach (Type type in TypeCache.GetTypesDerivedFrom<BaseGraphEditorNode>())
            {
                if (type.GetCustomAttribute<EditorNodeAttribute>() is EditorNodeAttribute attribute)
                {
                    _nodeEntries.Add(attribute.NodeType, new NodeEntry(attribute.NodeType, type, attribute.MenuItem));
                }
            }
        }

        public class SearchWindowAdapter : SearcherAdapter
        {
            public override bool HasDetailsPanel => false;

            public SearchWindowAdapter(string title) : base(title)
            {
            }
        }

        public class SearchNodeItem : SearcherItem
        {
            private NodeEntry _nodeEntry;

            public NodeEntry NodeEntry { get { return _nodeEntry; } }

            public SearchNodeItem(string name, NodeEntry nodeEntry, string help = "", List<SearchNodeItem> children = null) : base(name)
            {
                _nodeEntry = nodeEntry;
            }
        }

        public struct NodeEntry : IEquatable<NodeEntry>
        {
            private Type _nodeType;
            private Type _editorNodeType;
            private string _menuItem;

            public Type NodeType { get { return _nodeType; } }
            public Type EditorNodeType { get { return _editorNodeType; } }
            public string MenuItem { get { return _menuItem; } }

            public NodeEntry(Type nodeType, Type editorNodeType, string menuItem)
            {
                _nodeType = nodeType;
                _editorNodeType = editorNodeType;
                _menuItem = menuItem;
            }

            public bool Equals(NodeEntry other)
            {
                return Equals(_menuItem, other._menuItem) && _nodeType == other._nodeType;
            }
        }


        //private BaseGraphView _graphView;
        //private VisualElement _target;

        //private static List<SearchContextElement> _elements;

        //public BaseGraphView GraphView { set { _graphView = value; } }
        //public VisualElement Target { set { _target = value; } }

        //public List<SearchTreeEntry> CreateSearchTree(SearchWindowContext context)
        //{
        //    List<SearchTreeEntry> tree = new();
        //    tree.Add(new SearchTreeGroupEntry(new GUIContent("Nodes"), 0));

        //    _elements = new();
        //    Assembly[] assemblies = AppDomain.CurrentDomain.GetAssemblies();
        //    foreach (Assembly assembly in assemblies)
        //    {
        //        foreach (Type type in assembly.GetTypes())
        //        {
        //            if (type.CustomAttributes.Count() > 0)
        //            {
        //                Attribute attribute = type.GetCustomAttribute(typeof(NodeInfoAttribute));
        //                if (attribute != null)
        //                {
        //                    NodeInfoAttribute nodeInfoAttribute = (NodeInfoAttribute)attribute;
        //                    object node = Activator.CreateInstance(type);
        //                    if (string.IsNullOrEmpty(nodeInfoAttribute.MenuItem))
        //                    {
        //                        continue;
        //                    }
        //                    _elements.Add(new SearchContextElement(node, nodeInfoAttribute.MenuItem));
        //                }
        //            }
        //        }
        //    }

        //    List<string> groups = new();

        //    foreach (SearchContextElement element in _elements)
        //    {
        //        string[] entryTitle = element.Title.Split('/');
        //        string groupName = "";

        //        for (int i = 0; i < entryTitle.Length - 1; i++)
        //        {
        //            groupName += entryTitle[i];
        //            if (!groups.Contains(groupName))
        //            {
        //                tree.Add(new SearchTreeGroupEntry(new GUIContent(entryTitle[i]), i + 1));
        //                groups.Add(groupName);
        //            }
        //            groupName += "/";
        //        }

        //        SearchTreeEntry entry = new SearchTreeEntry(new GUIContent(entryTitle.Last()));
        //        entry.level = entryTitle.Length;
        //        entry.userData = new SearchContextElement(element.Target, element.Title);
        //        tree.Add(entry);
        //    }

        //    return tree;
        //}

        //public bool OnSelectEntry(SearchTreeEntry SearchTreeEntry, SearchWindowContext context)
        //{
        //    Vector2 windowMousePosition = _graphView.ChangeCoordinatesTo(_graphView, context.screenMousePosition - _graphView.Window.position.position);
        //    Vector2 graphMousePosition = _graphView.contentViewContainer.WorldToLocal(windowMousePosition);

        //    SearchContextElement element = (SearchContextElement)SearchTreeEntry.userData;

        //    BaseNode node = (BaseNode)element.Target;
        //    node.Position = new Rect(graphMousePosition, new Vector2());
        //    _graphView.AddNode(node);
        //    return true;
        //}
    }
}
