﻿using System;
using System.Collections.Generic;
using System.Linq;
using UnityEditor;
using UnityEditor.Experimental.GraphView;
using UnityEditor.Searcher;
using UnityEngine;
using UnityEngine.UIElements;

namespace PK
{
    public class GraphData : ScriptableSingleton<GraphData>
    {
        [SerializeField] private BaseGraph _graph;

        public BaseGraph Graph { get { return _graph; } set { _graph = value; } }
    }

    public class BaseGraphView : GraphView
    {
        protected static string UNDO_MESSAGE = "Edited nodes";

        private SerializedProperty _graphProperty;
        private SerializedObject _serializedData;
        private SerializedProperty _nodesListProperty;
        private SerializedProperty _connectionsListProperty;
        private ActionGraphEditorWindow _window;

        private List<BaseGraphEditorNode> _graphNodes = new();
        private List<Edge> _graphEdges = new();
        private Dictionary<string, BaseGraphEditorNode> _nodeDict = new();

        private BaseGraphSearchWindowProvider _searchProvider;

        public bool IsEmpty { get { return _graphNodes.Count == 0; } }
        public ActionGraphEditorWindow Window { get { return _window; } }
        public BaseGraph Graph { get { return GraphData.instance.Graph; } }
        public event Action OnChanged;

        public BaseGraphView(UnityEngine.Object @object, string propertyPath, ActionGraphEditorWindow window)
        {
            SerializedObject serializedObject = new SerializedObject(@object);
            GraphData.instance.Graph = serializedObject.FindProperty(propertyPath).boxedValue as BaseGraph;
            serializedObject.Dispose();

            _serializedData = new SerializedObject(GraphData.instance);
            _graphProperty = _serializedData.FindProperty("_graph");
            _nodesListProperty = _graphProperty.FindPropertyRelative("_nodes");
            _connectionsListProperty = _graphProperty.FindPropertyRelative("_connections");
            _window = window;
            _searchProvider = ScriptableObject.CreateInstance<BaseGraphSearchWindowProvider>();
            _searchProvider.GraphView = this;
            this.nodeCreationRequest += OnCreateSearchWindow;
            this.graphViewChanged += OnGraphViewChanged;
            this.serializeGraphElements += OnSerializeGraphElements;
            this.unserializeAndPaste += OnUnserializeAndPaste;

            StyleSheet styleSheet = (StyleSheet)EditorGUIUtility.Load("StyleSheets/GraphView/GraphView.uss");
            styleSheets.Add(styleSheet);

            GridBackground background = new GridBackground();
            Insert(0, background);
            background.StretchToParentSize();

            SetupZoom(0.05f, 8f);
            this.AddManipulator(new ContentDragger());
            this.AddManipulator(new SelectionDragger());
            this.AddManipulator(new RectangleSelector());
            this.AddManipulator(new ClickSelector());

            UpdateView();
            Undo.undoRedoPerformed += OnUndoRedo;
            EditorApplication.delayCall += Validate;
        }

        public void Dispose()
        {
            _serializedData.Dispose();
            Undo.ClearUndo(GraphData.instance);
            Undo.undoRedoPerformed -= OnUndoRedo;
        }

        private void OnUndoRedo()
        {
            EditorApplication.delayCall += UpdateView;
        }

        private void OnCreateSearchWindow(NodeCreationContext context)
        {
            SearcherWindow.Show(_window, _searchProvider.LoadSearchWindow(),
                    item => _searchProvider.OnSelectEntry(item, context.screenMousePosition - _window.position.position),
                    context.screenMousePosition - _window.position.position, null);
        }

        private GraphViewChange OnGraphViewChanged(GraphViewChange graphViewChange)
        {
            if (graphViewChange.movedElements != null)
            {
                BaseGraphEditorNode[] editorNodes = graphViewChange.movedElements.OfType<BaseGraphEditorNode>().ToArray();
                if (editorNodes.Length > 0)
                {
                    RecordUndo();
                    foreach (BaseGraphEditorNode node in editorNodes)
                    {
                        node.UpdatePosition();
                    }
                }
            }

            if (graphViewChange.elementsToRemove != null)
            {
                RecordUndo();
                BaseGraphEditorNode[] editorNodes = graphViewChange.elementsToRemove.OfType<BaseGraphEditorNode>().ToArray();
                if (editorNodes.Length > 0)
                {
                    foreach (BaseGraphEditorNode editorNode in editorNodes)
                    {
                        RemoveNode(editorNode);
                    }
                }
                Edge[] edges = graphViewChange.elementsToRemove.OfType<Edge>().ToArray();
                if (edges.Length > 0)
                {
                    foreach (Edge edge in edges)
                    {
                        RemoveEdge(edge);
                    }
                }
            }

            if (graphViewChange.edgesToCreate != null)
            {
                if (graphViewChange.edgesToCreate.Count > 0)
                {
                    RecordUndo();
                    foreach(Edge edge in graphViewChange.edgesToCreate)
                    {
                        AddEdge(edge);
                    }
                }
            }

            OnChanged?.Invoke();
            EditorApplication.delayCall += ValidateNodes;

            return graphViewChange;
        }

        private string OnSerializeGraphElements(IEnumerable<GraphElement> elements)
        {
            IEnumerable<BaseGraphEditorNode> nodes = elements.OfType<BaseGraphEditorNode>();
            IEnumerable<Edge> edges = elements.OfType<Edge>();
            CopyPasteGraph graph = new CopyPasteGraph(nodes, edges);
            return JsonUtility.ToJson(graph);
        }

        private void OnUnserializeAndPaste(string operationName, string data)
        {
            CopyPasteGraph graph = JsonUtility.FromJson<CopyPasteGraph>(data);
            graph.UpdateGuids();
            foreach (BaseNode node in graph.Nodes)
            {
                _nodesListProperty.InsertArrayElementAtIndex(_nodesListProperty.arraySize);
                SerializedProperty nodeProperty = _nodesListProperty.GetArrayElementAtIndex(_nodesListProperty.arraySize - 1);
                nodeProperty.managedReferenceValue = node;
            }
            foreach (NodeConnection connection in graph.Connections)
            {
                if (!string.IsNullOrEmpty(connection.InputPort.NodeGuid) && !string.IsNullOrEmpty(connection.OutputPort.NodeGuid))
                {
                    _connectionsListProperty.InsertArrayElementAtIndex(_connectionsListProperty.arraySize);
                    _connectionsListProperty.GetArrayElementAtIndex(_connectionsListProperty.arraySize - 1).boxedValue = connection;
                }
            }
            _nodesListProperty.serializedObject.ApplyModifiedProperties();
            UpdateView();
            ClearSelection();
            foreach (Edge edge in _graphEdges)
            {
                if (graph.Connections.Contains(new NodeConnection(new NodeConnectionPort(edge.input.node.viewDataKey, edge.input.viewDataKey), new NodeConnectionPort(edge.output.node.viewDataKey, edge.output.viewDataKey))))
                {
                    AddToSelection(edge);
                }
            }
            foreach (BaseGraphEditorNode node in _graphNodes)
            {
                if (graph.Nodes.Contains(node.Node))
                {
                    AddToSelection(node);
                }
            }
        }

        public override List<Port> GetCompatiblePorts(Port startPort, NodeAdapter nodeAdapter)
        {
            List<Port> allPorts = new();
            List<Port> ports = new();

            foreach (BaseGraphEditorNode node in _graphNodes)
            {
                allPorts.AddRange(node.Ports.Values);
            }

            foreach (Port port in allPorts)
            {
                if (port == startPort || port.node == startPort.node || port.direction == startPort.direction)
                {
                    continue;
                }
                if (port.portType == startPort.portType)
                {
                    ports.Add(port);
                }
            }

            return ports;
        }

        public void AddNode(BaseNode node)
        {
            RecordUndo();

            _nodesListProperty.InsertArrayElementAtIndex(_nodesListProperty.arraySize);
            SerializedProperty nodeProperty = _nodesListProperty.GetArrayElementAtIndex(_nodesListProperty.arraySize - 1);
            node.Guid = GUID.Generate().ToString();
            nodeProperty.managedReferenceValue = node;
            _nodesListProperty.serializedObject.ApplyModifiedProperties();

            AddNodeToView(node, nodeProperty);
            ValidateNodes();
            OnChanged?.Invoke();
        }

        public void RemoveNode(BaseGraphEditorNode editorNode)
        {
            for (int i = 0; i < _nodesListProperty.arraySize; i++)
            {
                SerializedProperty element = _nodesListProperty.GetArrayElementAtIndex(i);
                if (element.managedReferenceValue == editorNode.Node)
                {
                    element.managedReferenceValue = null;
                    break;
                }
            }
            _graphProperty.serializedObject.ApplyModifiedProperties();

            _graphNodes.Remove(editorNode);
            _nodeDict.Remove(editorNode.viewDataKey);
        }

        private void AddNodeToView(BaseNode node, SerializedProperty property)
        {
            if (node == null)
            {
                return;
            }

            Type editorNodeType = _searchProvider.GetEditorNodeType(node.GetType());
            if (editorNodeType != null)
            {
                BaseGraphEditorNode editorNode = (BaseGraphEditorNode)Activator.CreateInstance(editorNodeType, property);
                editorNode.View = this;
                editorNode.SetPosition(node.Position);
                _graphNodes.Add(editorNode);
                _nodeDict.Add(node.Guid, editorNode);
                editorNode.viewDataKey = node.Guid;

                AddElement(editorNode);
            }
        }

        private void AddEdge(Edge edge)
        {
            NodeConnection connection = new NodeConnection(edge.input.node.viewDataKey, edge.input.viewDataKey, edge.output.node.viewDataKey, edge.output.viewDataKey);
            _connectionsListProperty.InsertArrayElementAtIndex(_connectionsListProperty.arraySize);
            _connectionsListProperty.GetArrayElementAtIndex(_connectionsListProperty.arraySize - 1).boxedValue = connection;
            _connectionsListProperty.serializedObject.ApplyModifiedProperties();
            _graphEdges.Add(edge);
        }

        private void RemoveEdge(Edge edge)
        {
            for (int i = 0; i < _connectionsListProperty.arraySize; i++)
            {
                NodeConnection connection = (NodeConnection)_connectionsListProperty.GetArrayElementAtIndex(i).boxedValue;
                if (edge.input.node?.viewDataKey == connection.InputPort.NodeGuid && edge.output.node?.viewDataKey == connection.OutputPort.NodeGuid && connection.InputPort.PortGuid == edge.input.viewDataKey && connection.OutputPort.PortGuid == edge.output.viewDataKey)
                {
                    _connectionsListProperty.DeleteArrayElementAtIndex(i);
                    break;
                }
            }
            _graphProperty.serializedObject.ApplyModifiedProperties();
            _graphEdges.Remove(edge);
        }

        public void UpdateView()
        {
            graphViewChanged -= OnGraphViewChanged;
            _serializedData.Update();
            DeleteElements(_graphNodes);
            DeleteElements(_graphEdges);
            _nodeDict.Clear();
            _graphNodes.Clear();
            _graphEdges.Clear();
            DrawNodes();
            DrawConnections();
            ValidateNodes();
            graphViewChanged += OnGraphViewChanged;
        }

        public void RemovePort(Port port)
        {
            List<Edge> edgesToRemove = new();
            for (int i = _graphEdges.Count - 1; i >= 0; i--)
            {
                Edge edge = _graphEdges[i];
                if (edge.input == port || edge.output == port)
                {
                    for (int j = _connectionsListProperty.arraySize - 1; j >= 0; j--)
                    {
                        NodeConnection connection = (NodeConnection)_connectionsListProperty.GetArrayElementAtIndex(j).boxedValue;
                        if (connection.InputPort.PortGuid == port.viewDataKey || connection.OutputPort.PortGuid == port.viewDataKey)
                        {
                            _connectionsListProperty.DeleteArrayElementAtIndex(j);
                        }
                    }
                    _graphEdges.RemoveAt(i);
                    edgesToRemove.Add(edge);
                }
            }
            _connectionsListProperty.serializedObject.ApplyModifiedProperties();
            RemoveElement(port);
            DeleteElements(edgesToRemove);
        }

        private void DrawNodes()
        {
            for (int i = 0; i < _nodesListProperty.arraySize; i++)
            {
                SerializedProperty property = _nodesListProperty.GetArrayElementAtIndex(i);
                AddNodeToView((BaseNode)property.managedReferenceValue, property);
            }
        }

        private void DrawConnections()
        {
            for (int i = 0; i < _connectionsListProperty.arraySize; i++)
            {
                DrawConnection((NodeConnection)_connectionsListProperty.GetArrayElementAtIndex(i).boxedValue);
            }
        }

        private void DrawConnection(NodeConnection connection)
        {
            if (_nodeDict.TryGetValue(connection.InputPort.NodeGuid, out BaseGraphEditorNode inputNode) && _nodeDict.TryGetValue(connection.OutputPort.NodeGuid, out BaseGraphEditorNode outputNode))
            {
                try
                {
                    Port inputPort = inputNode.Ports[connection.InputPort.PortGuid];
                    Port outputPort = outputNode.Ports[connection.OutputPort.PortGuid];
                    Edge edge = inputPort.ConnectTo(outputPort);
                    AddElement(edge);
                    _graphEdges.Add(edge);
                }
                catch
                {
                    Debug.Log($"Can't create edge {connection.InputPort.NodeGuid} {connection.OutputPort.NodeGuid}");
                }
            }
        }

        private void ValidateNodes()
        {
            foreach (BaseGraphEditorNode node in _graphNodes)
            {
                node.Validate();
            }
        }

        private void RecordUndo()
        {
            SerializedObject obj = _graphProperty.serializedObject;
            if (obj != null)
            {
                UnityEngine.Object @object = obj.targetObject;
                Undo.RecordObject(@object, UNDO_MESSAGE);
            }
        }

        private void Validate()
        {
            for (int i = _connectionsListProperty.arraySize - 1; i >= 0; i--)
            {
                NodeConnection connection = (NodeConnection)_connectionsListProperty.GetArrayElementAtIndex(i).boxedValue;
                if (!Graph.Nodes.Any((n) => n.Guid == connection.InputPort.NodeGuid) || !Graph.Nodes.Any((n) => n.Guid == connection.OutputPort.NodeGuid))
                {
                    Debug.LogWarning("Removed invalid connection.");
                    _connectionsListProperty.DeleteArrayElementAtIndex(i);
                }
            }
            if (_serializedData.hasModifiedProperties)
            {
                _serializedData.ApplyModifiedProperties();
                OnChanged?.Invoke();
            }
        }
    }
}
