﻿using System;
using System.Linq;
using System.Collections.Generic;
using UnityEngine;

namespace PK
{
    public static class HexHelper
    {
        // https://www.redblobgames.com/grids/hexagons/
        // For 90x72 tile there are different constants calculated manually
        // Constant sqrt(3) is 2.5
        // Constant 3/2 is 1.61111 (vertical distance between cells is not 3 half sizes)

        private static float PIXELS_PER_UNIT = 100f;
        private static Vector2 TIP_SIZE = new Vector2(0, 14 / PIXELS_PER_UNIT);
        private static Vector2 TILE_SIZE = new Vector2(90 / PIXELS_PER_UNIT, 72 / PIXELS_PER_UNIT);
        private static float HALF_SIZE = TILE_SIZE.y / 2.0f;

        private static Matrix2x2 AXIAL_HEX_TO_WORLD = new Matrix2x2(2.5f, 2.5f / 2.0f, 0, 1.61111f);
        private static Matrix2x2 AXIAL_WORLD_TO_HEX = AXIAL_HEX_TO_WORLD.Inverse();

        // Counter-clockwise from south-west
        public static Vector2Int[] AXIAL_DIRECTIONS =
        {
            new Vector2Int(0, -1),
            new Vector2Int(1, -1),
            new Vector2Int(1, 0),
            new Vector2Int(0, 1),
            new Vector2Int(-1, 1),
            new Vector2Int(-1, 0)
        };

        public static Vector2Int INVALID_POSITION = new Vector2Int(-1, -1);

        public static Vector3[] TILE_CORNERS = new Vector3[]
        {
            new Vector3(0, (TILE_SIZE.y) / 2),
            new Vector3(TILE_SIZE.x / 2, TILE_SIZE.y / 2 - TIP_SIZE.y),
            new Vector3(TILE_SIZE.x / 2, -TILE_SIZE.y / 2 + TIP_SIZE.y),
            new Vector3(0, (-TILE_SIZE.y) / 2),
            new Vector3(-TILE_SIZE.x / 2, -TILE_SIZE.y / 2 + TIP_SIZE.y),
            new Vector3(-TILE_SIZE.x / 2, TILE_SIZE.y / 2 - TIP_SIZE.y),
        };

        private static System.Random _random = new System.Random();

        public static Vector3 GetTilePosition(Vector2Int odd)
        {
            return AXIAL_HEX_TO_WORLD * OddToAxial(odd) * HALF_SIZE;
        }

        public static Vector3 GetTilePosition(Vector2Int odd, Vector2 offset)
        {
            Vector3 position = GetTilePosition(odd);
            return position + (Vector3)(offset * TILE_SIZE);
        }

        public static Vector3[] GetTileCorners(Vector3 position, float scale = 1)
        {
            Vector3[] result = new Vector3[7];
            for (int i = 0; i < 6; i++)
            {
                result[i] = position + TILE_CORNERS[i] * scale;
            }
            result[6] = result[0];
            return result;
        }

        public static Rect GetMapWorldBounds(int mapWidth, int mapHeight)
        {
            if (mapWidth <= 0 || mapHeight <= 0) return Rect.zero;

            // Get world positions of corner tiles (adjust if your grid origin isn't 0,0)
            // This assumes GetTilePosition returns the center of the hex.
            Vector2 bottomLeft = GetTilePosition(new Vector2Int(0, 0));
            Vector2 topRight = GetTilePosition(new Vector2Int(mapWidth - 1, mapHeight - 1));
            Vector2 topLeft = GetTilePosition(new Vector2Int(0, mapHeight - 1));
            Vector2 bottomRight = GetTilePosition(new Vector2Int(mapWidth - 1, 0));

            // Find min/max world coordinates based on tile centers
            float minX = Mathf.Min(bottomLeft.x, topLeft.x);
            float maxX = Mathf.Max(bottomRight.x, topRight.x);
            float minY = Mathf.Min(bottomLeft.y, bottomRight.y);
            float maxY = Mathf.Max(topLeft.y, topRight.y);

            // Add half tile dimensions to extend bounds to tile edges
            float hexEdgeHorizontal = TILE_SIZE.x * 0.5f;
            float hexEdgeVertical = TILE_SIZE.y * 0.5f;

            return new Rect(
                minX - hexEdgeHorizontal,
                minY - hexEdgeVertical,
                (maxX - minX) + 2 * hexEdgeHorizontal,
                (maxY - minY) + 2 * hexEdgeVertical
            );
        }

        public static Vector2Int GetHexPosition(Vector3 localPosition)
        {
            Vector2 pos = AXIAL_WORLD_TO_HEX * localPosition / HALF_SIZE;
            pos = CubeToAxial(RoundCube(AxialToCube(pos)));
            return AxialToOdd(new Vector2Int(Mathf.RoundToInt(pos.x), Mathf.RoundToInt(pos.y)));
        }

        public static Vector2 GetOffset(Vector3 tilePosition, Vector3 localPosition)
        {
            localPosition.x = Mathf.Round(localPosition.x * PIXELS_PER_UNIT) / PIXELS_PER_UNIT;
            localPosition.y = Mathf.Round(localPosition.y * PIXELS_PER_UNIT) / PIXELS_PER_UNIT;
            return (localPosition - tilePosition) / TILE_SIZE;
        }

        public static Vector3 AxialToCube(Vector2 axial)
        {
            return new Vector3(axial.x, axial.y, -axial.x - axial.y);
        }

        public static Vector3Int AxialToCube(Vector2Int axial)
        {
            return new Vector3Int(axial.x, axial.y, -axial.x - axial.y);
        }

        public static Vector2Int CubeToAxial(Vector3Int cube)
        {
            return new Vector2Int(cube.x, cube.y);
        }

        public static Vector2Int CubeToOdd(Vector3Int cube)
        {
            return AxialToOdd(new Vector2Int(cube.x, cube.y));
        }

        public static Vector2Int AxialToOdd(Vector2Int axial)
        {
            return new Vector2Int(axial.x + (axial.y - (axial.y & 1)) / 2, axial.y);
        }

        public static Vector2Int OddToAxial(Vector2Int odd)
        {
            return new Vector2Int(odd.x - (odd.y - (odd.y & 1)) / 2, odd.y);
        }

        public static Vector3Int OddToCube(Vector2Int odd)
        {
            return AxialToCube(OddToAxial(odd));
        }

        public static int AxialDistance(Vector2Int axialStart, Vector2Int axialEnd)
        {
            Vector2Int axialSubtract = new Vector2Int(axialStart.x - axialEnd.x, axialStart.y - axialEnd.y);
            int distance = (Mathf.Abs(axialSubtract.x) + Mathf.Abs(axialSubtract.x + axialSubtract.y) + Mathf.Abs(axialSubtract.y)) / 2;
            return distance;
        }

        private static Vector3Int RoundCube(Vector3 cube)
        {
            Vector3Int roundCube = new Vector3Int(Mathf.RoundToInt(cube.x), Mathf.RoundToInt(cube.y), Mathf.RoundToInt(cube.z));
            Vector3 diff = new Vector3(Mathf.Abs(cube.x - roundCube.x), Mathf.Abs(cube.y - roundCube.y), Mathf.Abs(cube.z - roundCube.z));

            if (diff.x > diff.y && diff.x > diff.z)
            {
                roundCube.x = -roundCube.y - roundCube.z;
            }
            else if (diff.y > diff.z)
            {
                roundCube.y = -roundCube.x - roundCube.z;
            }
            else
            {
                roundCube.z = -roundCube.x - roundCube.y;
            }
            return roundCube;
        }

        public static ulong GenerateUID()
        {
            ulong uid = (ulong)_random.Next();
            uid |= (ulong)_random.Next() << 32;
            return uid;
        }

        public static bool IsUidUnique(ulong uid, HexUniqueScriptableObject[] objects)
        {
            if (uid == 0)
            {
                return false;
            }
            return objects.Count((o) => o.Uid == uid) < 2;
        }

        public static List<Vector2Int> GetNeighbourList(Vector2Int center, int radius)
        {
            List<Vector2Int> result = new List<Vector2Int>();
            if (radius == 0)
            {
                return result;
            }

            Vector3Int cubePosition = OddToCube(center);
            for (int i = -radius; i <= radius; i++)
            {
                for (int j = -radius; j <= radius; j++)
                {
                    for (int k = -radius; k <= radius; k++)
                    {
                        if (i + j + k == 0)
                        {
                            result.Add(CubeToOdd(cubePosition + new Vector3Int(i, j, k)));
                        }
                    }
                }
            }
            return result;
        }

        public static Vector3[] GetMapCornerPositions(int mapWidth, int mapHeight)
        {
            if (mapWidth <= 0 || mapHeight <= 0)
            {
                return null;
            }

            Vector3[] corners = new Vector3[4];
            corners[0] = GetTilePosition(new Vector2Int(0, 0));
            corners[1] = GetTilePosition(new Vector2Int(mapWidth - 1, 0));
            corners[2] = GetTilePosition(new Vector2Int(0, mapHeight - 1));
            corners[3] = GetTilePosition(new Vector2Int(mapWidth - 1, mapHeight - 1));
            return corners;
        }
    }
}