using System.Collections.Generic;
using UnityEditor;
using UnityEditor.ShortcutManagement;
using UnityEngine;

namespace PK
{
    public class SaveMapProcessor : AssetModificationProcessor
    {
        public static string[] OnWillSaveAssets(string[] paths)
        {
            HexMapEditorWindow window = EditorWindow.GetWindow<HexMapEditorWindow>("Map Editor", false);
            if (window != null && window.hasFocus && window.hasUnsavedChanges)
            {
                window.SaveChanges();
            }
            return paths;
        }
    }

    public class HexMapEditorWindow : PreviewSceneView
    {
        private enum Section
        {
            Terrain = 0,
            Roads = 10,
            Entities = 20,
            GlobalTriggers = 30,
            Remover = 40,
            Settings = 50,
        }

        private enum Tool
        {
            PaintTerrain = 0,
            RemoveTerrain = 1,
            PaintRoads = 10,
            RemoveRoads = 11,
            PlaceObjects = 20,
            PlaceCreatures = 21,
            PlaceHeroes = 22,
            PlaceUtilities = 23,
            EditObjects = 24,
            GlobalVariables = 30,
            GlobalTriggers = 40,
            Remover = 50,
            Settings = 60,
        }

        private static string UNDO_MESSAGE = "Edited map";
        private static int TILE_MAP_VIEW_EDITOR_HASH = "HexMapViewEditor".GetHashCode();

        private static string[] SECTION_TEXTS = new string[]
        {
            "Terrain",
            "Roads",
            "Entities",
            "Variables",
            "Triggers",
            "Remover",
            "Settings",
        };

        private static string[] TERRAIN_TEXTS = new string[]
        {
            "Place",
            "Remove"
        };

        private static string[] ROADS_TEXTS = new string[]
        {
            "Place",
            "Remove"
        };

        private static string[] ENTITIES_TEXTS = new string[]
        {
            "Objects",
            "Creatures",
            "Heroes",
            "Utilities",
            "Edit"
        };

        private static HexMapEditorWindow _instance;

        [SerializeField] private HexMap _map;

        private PreviewSceneView _sceneVisualElement;
        private HexMapWritableView _view;

        [SerializeField] private Section _selectedSection = Section.Entities;
        [SerializeField] private Tool _selectedTool = Tool.EditObjects;
        private Dictionary<Tool, MapEditorTool> _tools = new();
        private MapEditorTool _currentTool;
        private Dictionary<Sprite, Texture2D> _icons = new();
        private bool _isMouseDown;
        private Vector3[] _gridLines;

        public static HexMapEditorWindow Instance { get { return _instance; } }

        public HexMap Map { get { return _map; } }

        [MenuItem("PK/Map Editor")]
        public static void Open()
        {
            Open(null);
        }

        public static void Open(HexMap map)
        {
            HexMapEditorWindow window = (HexMapEditorWindow)GetWindow(typeof(HexMapEditorWindow), false, "Map Editor");
            window.Show();
            window.Initialize(map);
        }

        [UnityEditor.Callbacks.OnOpenAsset(1)]
        public static bool OpenAsset(int instanceID, int line)
        {
            Object obj = EditorUtility.InstanceIDToObject(instanceID);
            if (obj is HexMap map)
            {
                Open(map);
                return true;
            }
            return false;
        }

        protected override void OnEnable()
        {
            base.OnEnable();
            _instance = this;

            GameObject viewObject = new GameObject();
            _view = viewObject.AddComponent<HexMapWritableView>();
            Add(viewObject);

            AddEditorTool(Tool.PaintTerrain, new PaintTerrainTool());
            AddEditorTool(Tool.RemoveTerrain, new RemoveTerrainTool());
            AddEditorTool(Tool.PaintRoads, new PaintRoadsTool());
            AddEditorTool(Tool.RemoveRoads, new RemoveRoadsTool());
            AddEditorTool(Tool.PlaceObjects, new PlaceObjectsTool());
            AddEditorTool(Tool.PlaceCreatures, new PlaceCreaturesTool());
            AddEditorTool(Tool.PlaceHeroes, new PlaceHeroesTool());
            AddEditorTool(Tool.PlaceUtilities, new PlaceUtilitiesTool());
            AddEditorTool(Tool.EditObjects, new EditEntitiesTool());
            AddEditorTool(Tool.GlobalVariables, new GlobalVariablesTool());
            AddEditorTool(Tool.GlobalTriggers, new GlobalTriggersTool());
            AddEditorTool(Tool.Remover, new RemoverTool());
            AddEditorTool(Tool.Settings, new SettingsTool());

            SelectTool(_selectedTool);

            if (_map != null)
            {
                Initialize(_map);
            }

            Undo.undoRedoPerformed += Repaint;
            EditorApplication.playModeStateChanged += playModeStateChanged;
        }
        
        protected override void OnDisable()
        {
            base.OnDisable();
            if (_view != null)
            {
                DestroyImmediate(_view.gameObject);
                _view = null;
            }
            foreach (Texture2D icon in _icons.Values)
            {
                DestroyImmediate(icon);
            }
            _icons.Clear();

            Undo.undoRedoPerformed -= Repaint;
            EditorApplication.playModeStateChanged -= playModeStateChanged;
        }

        private void playModeStateChanged(PlayModeStateChange stateChange)
        {
            // It behaves weird in play mode
            if (_map != null)
            {
                if (stateChange == PlayModeStateChange.EnteredPlayMode)
                {
                    SaveChanges();
                    _view.Clear();
                }
                if (stateChange == PlayModeStateChange.EnteredEditMode)
                {
                    _view.SetMap(_map);
                }
            }
        }

        private void OnDestroy()
        {
            if (_instance == this)
            {
                _instance = null;
            }
        }

        protected override void OnDrawHandles()
        {
            if (_map == null)
            {
                return;
            }

            if (SettingsTool.DrawInteractionMask)
            {
                DrawInteractionMask();
            }
            if (SettingsTool.DrawGrid)
            {
                DrawGrid();
            }
            DrawSelections();

            _currentTool?.OnDrawHandle();
        }

        protected override void OnDrawGUI()
        {
            if (_map == null)
            {
                _map = (HexMap)EditorGUILayout.ObjectField(_map, typeof(HexMap), false);
                if (_map != null)
                {
                    Initialize(_map);
                }
            }
            else
            {
                int controlId = GUIUtility.GetControlID(TILE_MAP_VIEW_EDITOR_HASH, FocusType.Keyboard);
                switch (Event.current.GetTypeForControl(controlId))
                {
                    case EventType.MouseMove:
                        _currentTool?.OnMouseMove();
                        Event.current.Use();
                        break;
                    case EventType.MouseDrag:
                    case EventType.MouseDown:
                        _currentTool?.OnMouseMove();
                        _currentTool?.OnMouseDown();
                        if (Event.current.type == EventType.Used)
                        {
                            Repaint();
                        }
                        if (_isMouseDown == false)
                        {
                            _currentTool?.OnMouseClick();
                            _isMouseDown = true;
                        }
                        break;
                    case EventType.MouseUp:
                        _isMouseDown = false;
                        break;
                    case EventType.KeyDown:
                        _currentTool?.OnKeyDown();
                        break;
                }
            }
        }

        private void Initialize(HexMap map)
        {
            foreach (MapEditorTool tool in _tools.Values)
            {
                tool.OnClear();
            }
            _map = map;
            _view.SetMap(map);
            EditorPrefs.SetString("EditorMap", AssetDatabase.AssetPathToGUID(AssetDatabase.GetAssetPath(map)));

            // Frame the map view
            Vector3[] corners = HexHelper.GetMapCornerPositions(_map.Model.Width, _map.Model.Height);
            Bounds mapBounds = new Bounds(corners[0], Vector3.one * 0.1f);
            mapBounds.Encapsulate(corners[1]);
            mapBounds.Encapsulate(corners[2]);
            mapBounds.Encapsulate(corners[3]);
            mapBounds.Expand(5.0f);
            this.Frame(mapBounds);
        }

        private void DrawGrid()
        {
            HexMapModel model = _view.Map.Model;
            using (new Handles.DrawingScope(Color.gray, _view.transform.localToWorldMatrix))
            {
                int count = model.Width * model.Height * 6 * 2;
                if (_gridLines == null || _gridLines.Length != count)
                {
                    _gridLines = new Vector3[count];
                    int offset = 0;
                    for (int i = 0; i < model.Width; i++)
                    {
                        for (int j = 0; j < model.Height; j++)
                        {
                            Vector3 position = HexHelper.GetTilePosition(new Vector2Int(i, j));

                            _gridLines[offset] = position + HexHelper.TILE_CORNERS[0];
                            _gridLines[offset + 1] = position + HexHelper.TILE_CORNERS[1];
                            _gridLines[offset + 2] = position + HexHelper.TILE_CORNERS[1];
                            _gridLines[offset + 3] = position + HexHelper.TILE_CORNERS[2];
                            _gridLines[offset + 4] = position + HexHelper.TILE_CORNERS[2];
                            _gridLines[offset + 5] = position + HexHelper.TILE_CORNERS[3];
                            _gridLines[offset + 6] = position + HexHelper.TILE_CORNERS[3];
                            _gridLines[offset + 7] = position + HexHelper.TILE_CORNERS[4];
                            _gridLines[offset + 8] = position + HexHelper.TILE_CORNERS[4];
                            _gridLines[offset + 9] = position + HexHelper.TILE_CORNERS[5];
                            _gridLines[offset + 10] = position + HexHelper.TILE_CORNERS[5];
                            _gridLines[offset + 11] = position + HexHelper.TILE_CORNERS[0];

                            offset += 12;
                        }
                    }
                }
                Handles.DrawLines(_gridLines);
            }
        }

        private void DrawInteractionMask()
        {
            Color previousColor = Handles.color;
            HexMapModel model = _view.Map.Model;
            using (new Handles.DrawingScope(_view.transform.localToWorldMatrix))
            {
                for (int i = 0; i < model.Width; i++)
                {
                    for (int j = 0; j < model.Height; j++)
                    {
                        int type = (int)HexObjectMaskHelper.GetType(model.GetInteractionMask(i, j));
                        if (type > 0)
                        {
                            Handles.color = HexObjectEditorWindow.MASK_COLORS[type];
                            Handles.DrawAAConvexPolygon(HexHelper.GetTileCorners(HexHelper.GetTilePosition(new Vector2Int(i, j))));
                        }
                    }
                }
            }
            Handles.color = previousColor;
        }

        private void DrawSelections()
        {
            if (_currentTool is EditEntitiesTool editEntitiesTool)
            {
                using (new Handles.DrawingScope(_view.transform.localToWorldMatrix))
                {
                    editEntitiesTool.OnDrawSelection();
                }
            }
        }

        public void OnToolsGUI()
        {
            if (Application.isPlaying)
            {
                return;
            }

            if (_map == null)
            {
                return;
            }

            GUILayout.BeginHorizontal();
            GUILayout.FlexibleSpace();

            Section newSelectedSection = (Section)(GUILayout.Toolbar((int)_selectedSection / 10, SECTION_TEXTS) * 10);
            if (newSelectedSection != _selectedSection)
            {
                _selectedSection = newSelectedSection;
                SelectTool((Tool)newSelectedSection);
            }

            GUILayout.FlexibleSpace();
            GUILayout.EndHorizontal();

            DrawInspectorPanel();
        }

        private void DrawInspectorPanel()
        {
            switch (_selectedSection)
            {
                case Section.Terrain:
                    DrawSubPanel(Section.Terrain, TERRAIN_TEXTS);
                    break;
                case Section.Roads:
                    DrawSubPanel(Section.Roads, ROADS_TEXTS);
                    break;
                case Section.Entities:
                    DrawSubPanel(Section.Entities, ENTITIES_TEXTS);
                    break;
            }
            _currentTool?.OnInspectorPanel();
        }

        private void DrawSubPanel(Section section, string[] texts)
        {
            GUILayout.BeginHorizontal();
            GUILayout.FlexibleSpace();

            Tool newSelectedTool = (Tool)(GUILayout.Toolbar((int)_selectedTool - (int)section, texts) + (int)section);
            if (newSelectedTool != _selectedTool)
            {
                SelectTool(newSelectedTool);
            }

            GUILayout.FlexibleSpace();
            GUILayout.EndHorizontal();
        }

        private void AddEditorTool(Tool tool, MapEditorTool editorTool)
        {
            editorTool.Initialize(this, _view);
            _tools.Add(tool, editorTool);
        }

        private void SelectTool(Tool tool)
        {
            _selectedTool = tool;
            _currentTool = _tools[tool];
        }

        public Texture2D GetIcon(Sprite sprite)
        {
            if (sprite == null)
            {
                return null;
            }

            if (!_icons.TryGetValue(sprite, out Texture2D icon))
            {
                if (sprite != null)
                {
                    icon = new Texture2D(128, 128);
                    IconRenderer.Render(sprite, icon, Vector2.one * 0.5f);
                    _icons.Add(sprite, icon);
                }
            }
            return icon;
        }

        public void StartEditingEntity(ulong entitiyInstanceId)
        {
            Focus();
            SelectTool(Tool.EditObjects);
            EditEntitiesTool editEntitiesTool = _currentTool as EditEntitiesTool;
            editEntitiesTool.SetSelectedInstanceId(entitiyInstanceId);
            Repaint();
        }

        public void RecordUndo()
        {
            EditorUtility.SetDirty(_view.Map);
            Undo.RegisterCompleteObjectUndo(_view, UNDO_MESSAGE);
            Undo.RegisterCompleteObjectUndo(_view.Map, UNDO_MESSAGE);
            hasUnsavedChanges = true;
        }

        public override void SaveChanges()
        {
            base.SaveChanges();
            EditorUtility.SetDirty(_view.Map);
            AssetDatabase.SaveAssets();
            AssetDatabase.Refresh();
            hasUnsavedChanges = false;
        }
    }
}
