﻿using UnityEditor;
using UnityEditor.AnimatedValues;
using UnityEditor.SceneManagement;
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEngine.UIElements;

namespace PK
{
    public class PreviewSceneView : EditorWindow
    {
        public const float DEFAULT_VIEW_SIZE = 2f;
        public const float MIN_VIEW_SIZE = 0.5f;
        public const float MAX_VIEW_SIZE = 100f;

        public static readonly Color BACKGROUND_COLOR = new Color(0.278431f, 0.278431f, 0.278431f, 1);

        private VisualElement _cameraVisualElement;
        private Camera _camera;
        private RenderTexture _cameraTexture;
        private Scene _scene;

        [SerializeField] private AnimVector3 _position = new AnimVector3(new Vector3(0, 0, -10));
        private AnimFloat _size = new AnimFloat(DEFAULT_VIEW_SIZE);

        public Camera Camera { get { return _camera; } }
        public Vector3 Pivot { get { return _position.value; } set { _position.value = value; } }
        public float Size { get { return _size.value; } set { _size.value = value; } }
        public float CameraDistance
        {
            get
            {
                float res = Size * 2f;

                // clamp to allowed range in case scene view size was huge
                return Mathf.Clamp(res, MIN_VIEW_SIZE, MAX_VIEW_SIZE);
            }
        }

        protected virtual void OnEnable()
        {
            wantsMouseMove = true;
            wantsLessLayoutEvents = true;
            wantsMouseEnterLeaveWindow = true;

            _scene = EditorSceneManager.NewPreviewScene();
            if (_cameraVisualElement == null)
            {
                _cameraVisualElement = CreateCameraVisualElement();
                rootVisualElement.Add(_cameraVisualElement);
            }

            GameObject cameraObject = new GameObject();
            cameraObject.transform.position = _position.value;
            _camera = cameraObject.AddComponent<Camera>();
            _camera.enabled = false;
            _camera.cameraType = CameraType.Game;
            _camera.scene = _scene;
            _camera.orthographic = true;
            _camera.backgroundColor = BACKGROUND_COLOR;
            _camera.orthographicSize = _size.value;
            SceneManager.MoveGameObjectToScene(cameraObject, _scene);

            _position.valueChanged.AddListener(Repaint);
            _size.valueChanged.AddListener(Repaint);
        }

        protected virtual void OnDisable()
        {
            EditorSceneManager.ClosePreviewScene(_scene);

            _position.valueChanged.RemoveListener(Repaint);
            _size.valueChanged.RemoveListener(Repaint);
        }

        protected void Add(GameObject go)
        {
            SceneManager.MoveGameObjectToScene(go, _scene);
        }

        private VisualElement CreateCameraVisualElement()
        {
            VisualElement root = new IMGUIContainer(OnSceneGUI)
            {
                name = "scene-editor-camera-rect",
                pickingMode = PickingMode.Position,
                viewDataKey = name
            };

            root.style.overflow = Overflow.Hidden;
            root.style.flexGrow = 1;

            return root;
        }

        private void OnSceneGUI()
        {
            if (Event.current.type == EventType.MouseMove)
            {
                Repaint();
            }

            Rect windowSpaceCameraRect = _cameraVisualElement.contentRect;

            GUI.BeginGroup(windowSpaceCameraRect);

            Rect groupSpaceCameraRect = new Rect(0, 0, windowSpaceCameraRect.width, windowSpaceCameraRect.height);
            Rect groupSpaceCameraRectInPixels = EditorGUIUtility.PointsToPixels(groupSpaceCameraRect);
            Handles.SetCamera(groupSpaceCameraRectInPixels, _camera);

            if (Event.current.type == EventType.Repaint)
            {
                Handles.BeginGUI();
                PrepareCameraTexture(groupSpaceCameraRectInPixels);
                SetupCamera();
                DrawCamera(windowSpaceCameraRect, groupSpaceCameraRect);
                Handles.EndGUI();
            }

            OnDrawHandles();

            GUI.EndGroup();

            OnDrawGUI();

            PreviewSceneViewMotion.HandleMotion(this);
        }

        protected virtual void OnDrawHandles() { }
        protected virtual void OnDrawGUI() { }

        private void PrepareCameraTexture(Rect cameraRect)
        {
            if (_cameraTexture == null || _cameraTexture.width != cameraRect.width || _cameraTexture.height != cameraRect.height)
            {
                if (_cameraTexture != null)
                {
                    _cameraTexture.Release();
                }
                _cameraTexture = new RenderTexture((int)cameraRect.width, (int)cameraRect.height, 0);
            }
        }

        private void SetupCamera()
        {
            _camera.transform.position = GetTransformPosition();
            _camera.orthographicSize = GetVerticalOrthoSize();
        }

        private void DrawCamera(Rect windowSpaceCameraRect, Rect groupSpaceCameraRect)
        {
            _camera.pixelRect = new Rect(0, 0, _cameraTexture.width, _cameraTexture.height);
            _camera.targetTexture = _cameraTexture;
            _camera.Render();
            GUI.DrawTexture(groupSpaceCameraRect, _cameraTexture);
        }

        private Vector3 GetTransformPosition()
        {
            return _position.value + _camera.transform.rotation * new Vector3(0, 0, -CameraDistance);
        }

        private float GetVerticalOrthoSize()
        {
            float res = Size;
            if (_camera.aspect < 1.0)
            {
                res /= _camera.aspect;
            }
            return res;
        }

        public virtual void Frame(Bounds bounds)
        {
            if (_camera == null)
            {
                return;
            }
            _position.value = new Vector3(bounds.center.x, bounds.center.y, _position.value.z);
            _size.value = Mathf.Clamp(bounds.size.y * 0.5f, MIN_VIEW_SIZE, MAX_VIEW_SIZE);;
            Repaint();
        }
    }
}
