﻿using System;
using UnityEditor;
using UnityEngine;

namespace PK
{
    public class EntityEditor
    {
        public HexMapWritableView View { get; set; }
        public HexMapEditorWindow Editor { get; set; }
        public HexMapModel Map { get; set; }
        public HexEntityModel Target { get; set; }
        public SerializedProperty SerializedProperty { get; set; }
        public Vector2Int HoveredTile { get; set; }

        public virtual void OnInspectorPanel() 
        {
            DrawProperties();
            DrawInteraction();
        }
        public virtual void OnDrawSelection() { }

        public virtual void OnMouseDown() { }
        public virtual void OnMouseClick() { }

        protected void DrawProperties()
        {
            SerializedProperty.serializedObject.UpdateIfRequiredOrScript();
            EditorGUI.BeginChangeCheck();
            if (SerializedProperty.hasVisibleChildren)
            {
                PropertyHelper.BeginIndent();
                PropertyHelper.ChildrenProperties(SerializedProperty);
                PropertyHelper.EndIndent();
            }
            if (EditorGUI.EndChangeCheck())
            {
                SerializedProperty.serializedObject.ApplyModifiedProperties();
                View.UpdateEntity(Target);
            }
        }

        protected void DrawInteraction()
        {
            SerializedObject serializedObject = SerializedProperty.serializedObject;
            ulong id = Target.Id;
            if (Map.IsInteractable(id))
            {
                if (!Map.HasInteraction(id))
                {
                    if (GUILayout.Button("Add interaction"))
                    {
                        Undo.RecordObject(serializedObject.targetObject, "Added interaction");
                        Map.AddInteraction(id);
                        serializedObject.UpdateIfRequiredOrScript();
                    }
                }
                else
                {
                    int interactionIndex = Map.GetInteractionIndex(id);
                    if (interactionIndex != -1)
                    {
                        EditorGUI.BeginChangeCheck();
                        SerializedProperty interactions = serializedObject.FindProperty("_model").FindPropertyRelative("_interactions");
                        SerializedProperty interactionPairProperty = interactions.GetArrayElementAtIndex(interactionIndex);
                        SerializedProperty interactionProperty = interactionPairProperty.FindPropertyRelative("_value");

                        if (interactionProperty.isExpanded = EditorGUILayout.Foldout(interactionProperty.isExpanded, "Interaction"))
                        {
                            PropertyHelper.BeginIndent();
                            PropertyHelper.ReferenceProperty<BaseInteraction>(interactionProperty);
                            PropertyHelper.ChildrenProperties(interactionProperty);

                            if (GUILayout.Button("Remove interaction"))
                            {
                                Undo.RecordObject(serializedObject.targetObject, "Removed interaction");
                                Map.RemoveInteraction(id);
                                serializedObject.Update();
                            }

                            PropertyHelper.EndIndent();
                        }
                        if (EditorGUI.EndChangeCheck())
                        {
                            serializedObject.ApplyModifiedProperties();
                        }
                    }
                }
            }
        }
    }

    public class CustomEntityEditorAttribute : Attribute
    {
        private Type _entityType;

        public Type EntityType { get { return _entityType; } }

        public CustomEntityEditorAttribute(Type entityType)
        {
            _entityType = entityType;
        }
    }
}
