﻿using UnityEngine;
using System.Linq;
using System.Collections.Generic;
using UnityEditor;
using Unity.VisualScripting;
using UnityEngine.UIElements;
using Unity.Collections.LowLevel.Unsafe;

namespace PK
{
    public class RemoverTool : MapPaintingTool
    {
        private static string REMOVER_BRUSH_SIZE = "RemoverBrushSize";
        private static string REMOVER_BRUSH_DENSITY = "RemoverBrushDensity";

        private static string REMOVER_TOGGLE_TERRAIN = "RemoverToggleTerrain";
        private static string REMOVER_TOGGLE_ROADS = "RemoverToggleRoads";
        private static string REMOVER_TOGGLE_ENTITIES = "RemoverToggleEntities";
        private static string REMOVER_TOGGLE_NAMED_ENTITIES = "RemoverToggleNamedEntities";

        protected override int BrushSize
        {
            get
            {
                return EditorPrefs.GetInt(REMOVER_BRUSH_SIZE);
            }
            set
            {
                EditorPrefs.SetInt(REMOVER_BRUSH_SIZE, value);
            }
        }

        protected override int BrushDensity
        {
            get
            {
                return EditorPrefs.GetInt(REMOVER_BRUSH_DENSITY);
            }
            set
            {
                EditorPrefs.SetInt(REMOVER_BRUSH_DENSITY, value);
            }
        }

        public static bool RemoverToggleTerrain { get { return EditorPrefs.GetBool(REMOVER_TOGGLE_TERRAIN); } set { EditorPrefs.SetBool(REMOVER_TOGGLE_TERRAIN, value); } }
        public static bool RemoverToggleRoads { get { return EditorPrefs.GetBool(REMOVER_TOGGLE_ROADS); } set { EditorPrefs.SetBool(REMOVER_TOGGLE_ROADS, value); } }
        public static bool RemoverToggleEntities { get { return EditorPrefs.GetBool(REMOVER_TOGGLE_ENTITIES); } set { EditorPrefs.SetBool(REMOVER_TOGGLE_ENTITIES, value); } }
        public static bool RemoverToggleNamedEntities { get { return EditorPrefs.GetBool(REMOVER_TOGGLE_NAMED_ENTITIES); } set { EditorPrefs.SetBool(REMOVER_TOGGLE_NAMED_ENTITIES, value); } }

        protected override Color PaintingToolGridColor => Color.red;

        public override bool IsToolReady()
        {
            return true;
        }

        public override void OnInspectorPanel()
        {
            EditorGUI.BeginChangeCheck();

            PaintingToolInspectorPanel();

            RemoverToggleTerrain = GUILayout.Toggle(RemoverToggleTerrain, "Remove terrain");
            RemoverToggleRoads = GUILayout.Toggle(RemoverToggleRoads, "Remove roads");
            RemoverToggleEntities = GUILayout.Toggle(RemoverToggleEntities, "Remove entities");
            RemoverToggleNamedEntities = GUILayout.Toggle(RemoverToggleNamedEntities, "Remove named entities");
            if (EditorGUI.EndChangeCheck())
            {
                SceneView.RepaintAll();
            }
        }

        public override void OnMouseDown()
        {
            if (Event.current.button != 0)
            {
                return;
            }
            _view.PauseEditorUpdates();
            RecordUndo();

            List<Vector2Int> neighbours = GetPaintedHexes();
            foreach (Vector2Int neighbour in neighbours)
            {
                if (RemoverToggleTerrain)
                {
                    _view.RemoveTile(neighbour);
                }
                if (RemoverToggleRoads)
                {
                    _view.RemoveRoad(neighbour);
                }
                if (RemoverToggleEntities)
                {
                    List<HexObjectModel> hexObjects = _view.Map.Model.GetEnumerable<HexObjectModel>(true).Where((o) => o.Position.Equals(neighbour)).ToList<HexObjectModel>();
                    foreach (HexObjectModel hexObject in hexObjects)
                    {
                        if (hexObject.Name.Length > 0 && RemoverToggleNamedEntities == false)
                        {
                            continue;
                        }
                        _view.RemoveObject(hexObject.Id);
                    }
                }
            }
            _view.ResumeEditorUpdates();
            Event.current.Use();
        }
    }
}
