﻿using UnityEngine;

namespace PK
{
    [ExecuteAlways]
    public class HexMapWritableView : HexMapView
    {
#if UNITY_EDITOR
        private MapEvents _writableEvents = new();

        public void SetMap(HexMap map)
        {
            _map = map;
            _events = _writableEvents;
            UpdateAll();
        }

        public bool CanReplaceTile(Vector2Int position, HexTile newTile)
        {
            return _map.Model.CanReplaceTerrainTile(position, newTile.Uid);
        }

        public void PlaceTile(Vector2Int position, HexTile newTile)
        {
            _map.Model.ReplaceTerrainTile(position, newTile.Uid);
            UpdateTerrain();
            // Some objects may change sprite because of tile change
            UpdateObjects();
        }

        public void RemoveTile(Vector2Int position)
        {
            _map.Model.ReplaceTerrainTile(position, 0);
            UpdateTerrain();
        }

        public void PlaceRoad(Vector2Int position, byte type, byte index = 0)
        {
            _map.Model.ReplaceRoad(position, type, index);
            UpdateTerrain();
        }

        public void RemoveRoad(Vector2Int position)
        {
            _map.Model.ReplaceRoad(position, 0, 0);
            UpdateTerrain();
        }

        public void DisplaceMap(Vector2Int direction)
        {
            _map.Model.MoveTerrain(direction);
            foreach (HexEntityModel entity in _map.Model.GetEnumerable(true))
            {
                //Vector2Int position = HexHelper.AxialToOdd(HexHelper.OddToAxial(entity.Position) + direction);
                Vector2Int position = entity.Position + direction;
                Vector2 offset = Vector2.zero;
                if (entity is HexObjectModel @object)
                {
                    offset = @object.Offset;
                }
                _map.Model.MoveEntity(position, offset, entity.Id);
                _writableEvents.CallOnEntityMove(entity.Id, position, offset);
            }
            UpdateTerrain();
        }

        public bool CanPlaceObject(Vector2Int position)
        {
            return _map.Model.CanPlaceObject(position);
        }

        public void PlaceObject(Vector2Int position, Vector2 offset, HexObject @object)
        {
            offset = @object.IsAligned ? Vector2.zero : offset;
            ulong uid = @object.Uid;
            ulong id = _map.Model.PlaceObject(position, offset, uid);
            _writableEvents.CallOnEntityCreate(_map.Model.GetEntity(id));
        }

        public void MoveObject(Vector2Int position, Vector2 offset, ulong id)
        {
            HexObject prop = HexDatabase.Instance.GetObject(_entities[id].Uid);
            if (prop != null)
            {
                offset = prop.IsAligned ? Vector2.zero : offset;
            }
            _map.Model.MoveEntity(position, offset, id);
            _writableEvents.CallOnEntityMove(id, position, offset);
        }

        public void RemoveObject(ulong id)
        {
            _map.Model.RemoveEntity(id);
            _writableEvents.CallOnEntityDestroy(id);
        }

        public ulong AddEntity(HexEntityModel entity)
        {
            ulong id = _map.Model.AddEntity(entity);
            _writableEvents.CallOnEntityCreate(entity);
            return id;
        }

        public void UpdateEntity(HexEntityModel entity)
        {
            _writableEvents.CallOnEntityChange(entity);
        }

        public bool CanPlaceCreature(Vector2Int position)
        {
            return _map.Model.CanPlaceCreature(position);
        }

        public void PlaceCreature(Vector2Int position, HexCreature creature)
        {
            ulong id = _map.Model.PlaceCreature(position, creature.Uid);
            _writableEvents.CallOnEntityCreate(_map.Model.GetEntity(id));
        }

        public bool CanPlaceHero(Vector2Int position)
        {
            return _map.Model.CanPlaceHero(position);
        }

        public void PlaceHero(Vector2Int position, HexHero hero)
        {
            ulong id = _map.Model.PlaceHero(position, hero.Uid);
            _writableEvents.CallOnEntityCreate(_map.Model.GetEntity(id));
        }

        public void PlaceProximityTrigger(Vector2Int position)
        {
            ulong id = _map.Model.PlaceProximityTrigger(position);
            _writableEvents.CallOnEntityCreate(_map.Model.GetEntity(id));
        }

        public void PlaceSpawnPoint(Vector2Int position)
        {
            ulong id = _map.Model.PlaceSpawnPoint(position);
            _writableEvents.CallOnEntityCreate(_map.Model.GetEntity(id));
        }

        public void PlaceMapZone(Vector2Int position)
        {
            ulong id = _map.Model.PlaceMapZone(position);
            _writableEvents.CallOnEntityCreate(_map.Model.GetEntity(id));
        }

        public void PauseEditorUpdates()
        {
            _updates_paused = true;
        }

        public void ResumeEditorUpdates()
        {
            _updates_paused = false;
            UpdateAll();
        }
#endif
    }
}
