﻿using System.Collections.Generic;
using System.Linq;
using UnityEngine;

namespace PK
{
    public class HexTerrainView : MonoBehaviour
    {
        private static int _mainTexPropertyId = Shader.PropertyToID("_MainTex");

        private HexMap _map;
        private HexTerrain[] _terrainMask;

        private Renderer _renderer;
        private MeshFilter _meshFilter;
        private Mesh _mesh;

        public Bounds Bounds { get { return _renderer.bounds; } }

        public HexTerrain GetTerrain(Vector2Int position)
        {
            return _terrainMask[position.x + _map.Size.x * position.y];
        }

        public void Initialize(HexMap map)
        {
            _map = map;
            Initialize();
        }

        public void Clear()
        {
            if (_mesh != null)
            {
                _mesh.Clear();
            }
        }

        public void UpdateTiles()
        {
            if (_mesh == null || _map == null)
            {
                return;
            }
            Dictionary<Vector2Int, HexTile> tileDatas = new();
            for (int i = 0; i < _map.Size.x; i++)
            {
                for (int j = 0; j < _map.Size.y; j++)
                {
                    ulong tileUid = _map.Model.GetTerrainTile(i, j);
                    if (tileUid != 0)
                    {
                        HexTile tile = HexDatabase.Instance.GetTile(tileUid);
                        if (tile != null)
                        {
                            tileDatas.Add(new Vector2Int(i, j), tile);
                            _terrainMask[i + _map.Size.x * j] = tile.Terrain;
                        }
                        else
                        {
                            Debug.LogError($"Tile {tileUid} is missing. Tile count: {HexDatabase.Instance.Tiles.Length}");
                        }
                    }
                }
            }

            SpriteHelper helper = new();
            List<Material> oldMaterials = _renderer.sharedMaterials.Where((m) => m != null).ToList();
            List<(Material material, int triangles)> submeshes = new();
            GenerateTiles(helper, oldMaterials, submeshes, tileDatas);
            GenerateMask(helper, oldMaterials, submeshes, tileDatas);
            GenerateRoads(helper, oldMaterials, submeshes);
            helper.FillMesh(_mesh);
            _mesh.subMeshCount = submeshes.Count;
            for (int i = 0, offset = 0; i < submeshes.Count; i++)
            {
                int triangles = submeshes[i].triangles;
                _mesh.SetSubMesh(i, new UnityEngine.Rendering.SubMeshDescriptor(offset, triangles));
                offset += triangles;
            }
            _renderer.SetSharedMaterials(submeshes.Select((s) => s.material).ToList());
        }

        private void GenerateTiles(SpriteHelper helper, List<Material> oldMaterials, List<(Material material, int triangles)> submeshes, Dictionary<Vector2Int, HexTile> tileDatas)
        {
            Texture atlasTexture = null;
            int currentIndexCount = helper.IndexCount;

            if (tileDatas.Count > 0)
            {
                atlasTexture = tileDatas.First().Value.Sprite.texture;
                foreach (KeyValuePair<Vector2Int, HexTile> pair in tileDatas)
                {
                    Sprite sprite = pair.Value.Sprite;
                    if (sprite.texture != atlasTexture)
                    {
                        Debug.LogError("Tiles are not in a single atlas.");
                        return;
                    }
                    helper.Add(HexHelper.GetTilePosition(pair.Key), sprite);
                }
            }
            Material material = GetMaterial(oldMaterials, HexMaterials.Instance.TileMaterial);
            material.SetTexture(_mainTexPropertyId, atlasTexture);
            material.SetFloat("_StencilComp", 8); // Disable mask
            submeshes.Add((material, helper.IndexCount - currentIndexCount));
        }

        private void GenerateMask(SpriteHelper helper, List<Material> oldMaterials, List<(Material material, int triangles)> submeshes, Dictionary<Vector2Int, HexTile> tileDatas)
        {
            Texture atlasTexture = null;
            Dictionary<Vector2Int, HexTile[]> maskTiles = new();

            if (tileDatas.Count > 0)
            {
                foreach (KeyValuePair<Vector2Int, HexTile> pair in tileDatas)
                {
                    HexTile currentTile = pair.Value;
                    Vector2Int axialPos = HexHelper.OddToAxial(pair.Key);
                    for (int i = 0; i < 6; i++)
                    {
                        Vector2Int nearbyPos = HexHelper.AxialToOdd(axialPos + HexHelper.AXIAL_DIRECTIONS[i]);
                        if (tileDatas.TryGetValue(nearbyPos, out HexTile tile))
                        {
                            if (tile.Terrain.Priority > currentTile.Terrain.Priority)
                            {
                                if (!maskTiles.TryGetValue(pair.Key, out HexTile[] existingMaskTiles))
                                {
                                    existingMaskTiles = new HexTile[6];
                                    maskTiles[pair.Key] = existingMaskTiles;
                                }
                                existingMaskTiles[i] = tile;
                                atlasTexture = tile.Sprite.texture;
                            }
                        }
                    }
                }
            }

            (SpriteHelper maskHelper, SpriteHelper tileHelper)[] helpers = new (SpriteHelper maskHelper, SpriteHelper tileHelper)[6];
            for (int i = 0; i < 6; i++)
            {
                helpers[i] = (new SpriteHelper(), new SpriteHelper());
            }

            foreach (KeyValuePair<Vector2Int, HexTile[]> pair in maskTiles)
            {
                Vector3 back = Vector3.back * 0.01f;
                Vector3 position = HexHelper.GetTilePosition(pair.Key) + back;
                int helperOffset = 0;
                for (int i = 0; i < 6; i++)
                {
                    HexTile tile = pair.Value[i];
                    bool foundAny = false;
                    for (int j = 0; j < 6; j++)
                    {
                        HexTile tile1 = pair.Value[j];
                        if (tile != null && tile1 != null && tile.Terrain == tile1.Terrain)
                        {
                            (SpriteHelper maskHelper, SpriteHelper tileHelper) = helpers[helperOffset];
                            tileHelper.Add(position + helperOffset * back, tile.Sprite);
                            maskHelper.Add(position + helperOffset * back, HexDatabase.Instance.Mask.GetSprite((HexDirection)(1 << j)));
                            pair.Value[j] = null;
                            foundAny = true;
                        }
                    }
                    if (foundAny)
                    {
                        ++helperOffset;
                    }
                }
            }

            for (int i = 0; i < 6; i++)
            {
                (SpriteHelper maskHelper, SpriteHelper tileHelper) = helpers[i];
                if (maskHelper.IndexCount > 0)
                {
                    int currentIndexCount = helper.IndexCount;
                    helper.Combine(maskHelper);
                    Material material = GetMaterial(oldMaterials, HexMaterials.Instance.TileMaskMaterial);
                    material.SetTexture(_mainTexPropertyId, atlasTexture);
                    material.SetFloat("_StencilRef", 1 << i);
                    submeshes.Add((material, helper.IndexCount - currentIndexCount));

                    currentIndexCount = helper.IndexCount;
                    helper.Combine(tileHelper);
                    material = GetMaterial(oldMaterials, HexMaterials.Instance.TileMaterial);
                    material.SetTexture(_mainTexPropertyId, atlasTexture);
                    material.SetFloat("_StencilRef", 1 << i);
                    material.SetFloat("_StencilComp", 3);
                    submeshes.Add((material, helper.IndexCount - currentIndexCount));
                }
            }
        }

        private void GenerateRoads(SpriteHelper helper, List<Material> oldMaterials, List<(Material material, int triangles)> submeshes)
        {
            Texture atlasTexture = null;
            int currentIndexCount = helper.IndexCount;

            for (int i = 0; i < _map.Size.x; i++)
            {
                for (int j = 0; j < _map.Size.y; j++)
                {
                    Vector2Int position = new Vector2Int(i, j);
                    (byte type, byte index) = _map.Model.GetRoad(position);
                    if (type > 0)
                    {
                        HexDirection direction = 0;
                        for (int k = 0; k < 6; k++)
                        {
                            Vector2Int nearbyPos = HexHelper.AxialToOdd(HexHelper.OddToAxial(position) + HexHelper.AXIAL_DIRECTIONS[k]);
                            if (nearbyPos.x < 0 || nearbyPos.y < 0 || nearbyPos.x >= _map.Size.x || nearbyPos.y >= _map.Size.y)
                            {
                                continue;
                            }
                            if (_map.Model.HasRoad(nearbyPos))
                            {
                                direction |= (HexDirection)(1 << k);
                            }
                        }
                        HexTerrain terrain = null;
                        ulong tileUid = _map.Model.GetTerrainTile(i, j);
                        if (tileUid != 0)
                        {
                            HexTile tile = HexDatabase.Instance.GetTile(tileUid);
                            if (tile != null)
                            {
                                terrain = tile.Terrain;
                            }
                        }
                        Sprite sprite = HexDatabase.Instance.GetRoad(type).GetSprite(direction, index, terrain);
                        if (sprite != null)
                        {
                            helper.Add(HexHelper.GetTilePosition(position), sprite);
                            atlasTexture = sprite.texture;
                        }
                    }
                }
            }

            Material material = GetMaterial(oldMaterials, HexMaterials.Instance.TileMaterial);
            material.SetFloat("_StencilComp", 8); // Disable mask
            material.SetTexture(_mainTexPropertyId, atlasTexture);
            submeshes.Add((material, helper.IndexCount - currentIndexCount));
        }

        private Material GetMaterial(List<Material> oldMaterials, Material newMaterial)
        {
            Material material = oldMaterials.Find((m) => m.shader == newMaterial.shader);
            if (material == null)
            {
                material = Instantiate(newMaterial);
            }
            else
            {
                oldMaterials.Remove(material);
            }
            return material;
        }

        private void Initialize()
        {
            if (_terrainMask == null || _terrainMask.Length != _map.Size.x * _map.Size.y)
            {
                _terrainMask = new HexTerrain[_map.Size.x * _map.Size.y];
            }

            if (_renderer == null)
            {
                hideFlags = HideFlags.HideInInspector | HideFlags.DontSave;

                _renderer = gameObject.AddComponent<MeshRenderer>();
                _renderer.hideFlags = HideFlags.HideInInspector | HideFlags.DontSave;

                _meshFilter = gameObject.AddComponent<MeshFilter>();
                _meshFilter.hideFlags = HideFlags.HideInInspector | HideFlags.DontSave;

                _mesh = new Mesh();
                _mesh.hideFlags = HideFlags.DontSave;
                _meshFilter.mesh = _mesh;
            }
        }
    }
}
