#!/bin/bash
set -e
set -o pipefail

readonly API_PREFIX="https://build-api.cloud.unity3d.com"
readonly SLEEP_FOR_SECS=100

mkdir -p ${OUTPUT_DIR?}

echo "Cancelling already running build to avoid errors..."
curl \
   -s \
  -X DELETE \
  -H "Content-Type: application/json" \
  -H "Authorization: Basic ${UNITY_CLOUD_API_KEY?}" \
  ${API_PREFIX}/api/v1/orgs/${UNITY_CLOUD_ORG_ID?}/projects/${UNITY_CLOUD_PROJECT_ID?}/buildtargets/_all/builds

echo "Starting builds..."

declare BUILD_START_RESPONSE=$(curl \
   -s \
  -X POST \
  -d "{\"clean\": true, \"delay\": 30, \"branch\": \"${CI_COMMIT_BRANCH?}\"}" \
  -H "Content-Type: application/json" \
  -H "Authorization: Basic ${UNITY_CLOUD_API_KEY?}" \
  ${API_PREFIX}/api/v1/orgs/${UNITY_CLOUD_ORG_ID?}/projects/${UNITY_CLOUD_PROJECT_ID?}/buildtargets/_all/builds
)
# must fail the build if cannot be parsed
# set -o pipefail for this reason

declare -a BUILD_LINKS_LEFT=()

while read -r BUILD; do
    declare BUILD_NUMBER=$(echo "$BUILD" | jq -rc ".build")
    declare BUILD_TARGET_ID=$(echo "$BUILD" | jq -rc ".buildtargetid")
    declare LINK=$(echo "$BUILD" | jq -rc ".links.self.href")
    echo "Started Unity Cloud Build ${BUILD_TARGET_ID} #${BUILD_NUMBER}"
    echo "$LINK"
    BUILD_LINKS_LEFT+=("$LINK")
done < <(echo "$BUILD_START_RESPONSE" | jq -rc '.[]')

declare EXIT_CODE=0
while true 
do
    declare NEW_BUILD_LINKS_LEFT
    
    NEW_BUILD_LINKS_LEFT=()

    for LINK in ${BUILD_LINKS_LEFT[@]}; do
        echo "Checking status: ${LINK}"        
        declare BUILD_INFO_RESPONSE=$(curl \
            -s \
            -X GET \
            -H "Authorization: Basic ${UNITY_CLOUD_API_KEY?}" \
            ${API_PREFIX}${LINK}                
        )
        declare BUILD_STATUS=$(echo $BUILD_INFO_RESPONSE | jq -rc ".buildStatus")
        echo "Build status: ${BUILD_STATUS}"

        case "${BUILD_STATUS}" in
            success)
                declare BUILD_TARGET=$(echo $BUILD_INFO_RESPONSE | jq -rc ".buildTargetName")
                declare BUILD_NUMBER=$(echo $BUILD_INFO_RESPONSE | jq -rc ".build")
                declare LOCAL_ARTIFACT_FILENAME="${OUTPUT_DIR?}/${BUILD_TARGET}-${BUILD_NUMBER}.zip"
                echo "Downloading artifacts to ${LOCAL_ARTIFACT_FILENAME}..."                
                declare ARTIFACTS_LINK=$(echo $BUILD_INFO_RESPONSE | jq -rc ".links.artifacts[0].files[0].href")                
                curl \
                    "${ARTIFACTS_LINK}" \
                    --output "${LOCAL_ARTIFACT_FILENAME}"
                ;;
            failure | canceled)
                echo "Failed! Aborting job..."
                EXIT_CODE=1
                exit ${EXIT_CODE}        
                ;;
            *)
                echo "still pending..."
                NEW_BUILD_LINKS_LEFT+=("${LINK}")
        esac
    done
    if [ ${EXIT_CODE} != 0 ]; then
        exit ${EXIT_CODE}
    fi

    echo "Builds pending: ${#NEW_BUILD_LINKS_LEFT[@]}"
    
    if [ "${#NEW_BUILD_LINKS_LEFT[@]}" -eq 0 ]; then
        echo "Finished!"
        exit 0
    fi
    
    echo "Sleeping ${SLEEP_FOR_SECS}..."
    sleep ${SLEEP_FOR_SECS}

    BUILD_LINKS_LEFT=("${NEW_BUILD_LINKS_LEFT[@]}")
done

exit ${EXIT_CODE}